# coding: utf-8

import inject

from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import is_spec_complete
from awacs.lib.order_processor.runner import StateRunner
from awacs.model import events, cache, zk
from awacs.model.domain.order.processors import DomainOrder, get_domain_state_processors
from infra.awacs.proto import model_pb2


class DomainOrderCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage

    EVENTS_QUEUE_GET_TIMEOUT = 5
    PROCESSING_INTERVAL = 3

    _state_runner = StateRunner(entity_class=DomainOrder,
                                initial_state=DomainOrder.states.STARTED,
                                final_state=DomainOrder.states.FINISHED,
                                final_cancelled_state=DomainOrder.states.CANCELLED,
                                processors=get_domain_state_processors(),
                                processing_interval=PROCESSING_INTERVAL)

    def __init__(self, namespace_id, domain_id):
        name = 'domain-order-ctl("{}:{}")'.format(namespace_id, domain_id)
        super(DomainOrderCtl, self).__init__(name)
        self._domain_id = domain_id
        self._namespace_id = namespace_id
        self._pb = None  # type: model_pb2.Domain or None

    def _accept_event(self, event):
        return (isinstance(event, events.DomainUpdate) and
                event.pb.meta.namespace_id == self._namespace_id and
                event.pb.meta.id == self._domain_id and
                not is_spec_complete(event.pb))

    def _start(self, ctx):
        try:
            self._process(ctx)
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            ctx.log.exception('failed to process domain order on start: %s', e)
        self._cache.bind(self._callback)

    def _stop(self):
        self._cache.unbind(self._callback)

    def _process(self, ctx, event=None):
        """
        :type event: events.DomainUpdate
        :type ctx: context.OpCtx
        """
        cached_pb = self._cache.must_get_domain(self._namespace_id, self._domain_id)
        if event and self._should_discard_event(ctx, event, cached_pb):
            return
        self._pb = cached_pb

        if not self._pb.HasField('order'):
            raise RuntimeError('Domain has no "order" field')

        return self._state_runner.process(ctx, self._pb)

    def _process_event(self, ctx, event):
        assert isinstance(event, events.DomainUpdate)
        self._process(ctx, event)

    def _process_empty_queue(self, ctx):
        self._process(ctx)
