import abc
import enum
import gevent
import inject
import six

from awacs.lib import certs
from awacs.lib.order_processor.model import (
    WithOrder,
    BaseProcessor,
    cancel_order,
    is_order_in_progress,
)
from awacs.model import dao, zk, cache, errors, util
from infra.awacs.proto import model_pb2


class State(enum.Enum):
    STARTED = 1
    CHECKING_CERT_INFO = 2
    CREATING_CERT_ORDER = 3
    WAITING_FOR_CERT_ORDER = 4
    SAVING_DOMAIN_SPEC = 5
    FINISHED = 20
    CANCELLING = 30
    WAITING_FOR_CERT_CANCEL = 31
    CANCELLED = 50


def get_domain_state_processors():
    return (
        Started,
        CheckingCertsInfo,
        CreatingCertOrders,
        WaitingForCertOrders,
        SavingDomainSpec,
        Cancelling,
        WaitingForCertsCancel,
    )


class DomainOrder(WithOrder):
    __slots__ = (u'order',)

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = u'Domain'
    states = State

    def __init__(self, pb):
        super(DomainOrder, self).__init__(pb)
        self.order = self.pb.order.content  # type: model_pb2.DomainOrder.Content

    def zk_update(self):
        return self.zk.update_domain(namespace_id=self.namespace_id,
                                     domain_id=self.id)

    def dao_update(self, comment):
        return self.dao.update_domain(namespace_id=self.namespace_id,
                                      domain_id=self.id,
                                      version=self.pb.meta.version,
                                      comment=comment,
                                      login=self.pb.meta.author,
                                      updated_spec_pb=self.pb.spec)


class DomainOrderProcessor(six.with_metaclass(abc.ABCMeta, BaseProcessor)):
    __slots__ = (u'order',)

    def __init__(self, entity):
        """
        :type entity: DomainOrder
        """
        super(DomainOrderProcessor, self).__init__(entity)
        self.order = entity.order  # type: model_pb2.DomainOrder.Content


class Started(DomainOrderProcessor):
    __slots__ = ()
    state = State.STARTED
    next_state = State.CHECKING_CERT_INFO
    cancelled_state = State.CANCELLING

    def process(self, ctx):
        return self.next_state


class CheckingCertsInfo(DomainOrderProcessor):
    __slots__ = ()
    state = State.CHECKING_CERT_INFO
    next_state = State.CREATING_CERT_ORDER
    next_state_saving = State.SAVING_DOMAIN_SPEC
    cancelled_state = State.CANCELLING

    def _should_skip_cert_creation(self):
        return self.order.protocol == model_pb2.DomainSpec.Config.HTTP_ONLY

    def process(self, ctx):
        if self._should_skip_cert_creation():
            return self.next_state_saving
        if self.order.HasField('cert_order'):
            return self.next_state
        return self.next_state_saving


class CreatingCertOrders(DomainOrderProcessor):
    __slots__ = ()
    state = State.CREATING_CERT_ORDER
    next_state = State.WAITING_FOR_CERT_ORDER
    cancelled_state = State.CANCELLING

    MAX_CERT_IDX = 50

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def _generate_cert_id(self, common_name, suffix):
        if all(ord(c) < 128 for c in common_name):
            # if CN is ASCII, we can use it as a certificate id
            base_generated_id = common_name
        else:
            # otherwise we use domain id, as it's guaranteed to be ASCII
            base_generated_id = self.entity.id
        if suffix is not None:
            base_generated_id += u'_{}'.format(suffix)
        normalized_cert_ids = {certs.normalize_cert_id(c_pb.meta.id)
                               for c_pb in self.cache.list_all_certs(self.entity.namespace_id)}
        generated_id = base_generated_id
        idx = 0
        while True:
            if certs.normalize_cert_id(generated_id) not in normalized_cert_ids:
                return generated_id
            idx += 1
            if idx > self.MAX_CERT_IDX:
                raise RuntimeError(
                    u'Maximum amount ({}) of auto-generated certificate IDs is reached for cert id "{}"'.format(
                        self.MAX_CERT_IDX, base_generated_id))
            generated_id = u'{}_{}'.format(base_generated_id, idx)
            gevent.idle()

    def _make_cert_meta_pb(self, cert_order_id):
        ns_pb = self.cache.must_get_namespace(self.entity.namespace_id)
        cert_meta_pb = model_pb2.CertificateMeta(id=cert_order_id, namespace_id=self.entity.namespace_id)
        cert_meta_pb.auth.type = cert_meta_pb.auth.STAFF
        cert_meta_pb.auth.staff.owners.logins.append(self.entity.pb.meta.author)
        cert_meta_pb.auth.staff.owners.logins.extend(ns_pb.meta.auth.staff.owners.logins)
        cert_meta_pb.auth.staff.owners.group_ids.extend(ns_pb.meta.auth.staff.owners.group_ids)
        util.omit_duplicate_items_from_auth(cert_meta_pb.auth)
        return cert_meta_pb

    def _save_context(self, cert_order_id, secondary_cert_order_id):
        self.entity.context[u'cert_order_id'] = cert_order_id
        if secondary_cert_order_id:
            self.entity.context[u'secondary_cert_order_id'] = secondary_cert_order_id
        self.entity.save_context()

    def _make_cert_order_ids(self):
        cert_order_id = self.entity.context.get(u'cert_order_id')
        secondary_cert_order_id = self.entity.context.get(u'secondary_cert_order_id')

        if not cert_order_id:
            if self.order.HasField('secondary_cert_order') or self.order.secondary_cert_ref.id:
                primary_suffix = self.order.cert_order.content.public_key_algorithm_id
            else:
                primary_suffix = None
            cert_order_id = self._generate_cert_id(common_name=self.order.cert_order.content.common_name,
                                                   suffix=primary_suffix)

        if not secondary_cert_order_id and self.order.HasField('secondary_cert_order'):
            secondary_cert_order_id = self._generate_cert_id(
                common_name=self.order.secondary_cert_order.content.common_name,
                suffix=self.order.secondary_cert_order.content.public_key_algorithm_id)

        return cert_order_id, secondary_cert_order_id

    def process(self, ctx):
        cert_order_id, secondary_cert_order_id = self._make_cert_order_ids()
        # save ids to context before creating certificates, so they're not orphaned if we crash
        self._save_context(cert_order_id, secondary_cert_order_id)

        self.entity.dao.create_cert_if_missing(
            meta_pb=self._make_cert_meta_pb(cert_order_id),
            order_content_pb=self.order.cert_order.content,
            login=util.NANNY_ROBOT_LOGIN)
        if secondary_cert_order_id:
            self.entity.dao.create_cert_if_missing(
                meta_pb=self._make_cert_meta_pb(secondary_cert_order_id),
                order_content_pb=self.order.secondary_cert_order.content,
                login=util.NANNY_ROBOT_LOGIN)
        return self.next_state


class WaitingForCertOrders(DomainOrderProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_CERT_ORDER
    next_state = State.SAVING_DOMAIN_SPEC
    cancelled_state = State.CANCELLING

    cache = inject.attr(cache.IAwacsCache)

    def process(self, ctx):
        cert_order_id = self.entity.context.get(u'cert_order_id')
        secondary_cert_order_id = self.entity.context.get(u'secondary_cert_order_id')
        if not cert_order_id and not secondary_cert_order_id:
            return self.next_state

        cert_pb = self.cache.get_cert(namespace_id=self.entity.namespace_id, cert_id=cert_order_id)
        if cert_pb and not is_order_in_progress(cert_pb):
            primary_ready = True
        else:
            primary_ready = False
            ctx.log.debug(u'primary cert is not ready yet')

        if not secondary_cert_order_id:
            secondary_ready = True
        else:
            secondary_cert_pb = self.cache.get_cert(namespace_id=self.entity.namespace_id,
                                                    cert_id=secondary_cert_order_id)
            if secondary_cert_pb and not is_order_in_progress(secondary_cert_pb):
                secondary_ready = True
            else:
                secondary_ready = False
                ctx.log.debug(u'secondary cert is not ready yet')

        if primary_ready and secondary_ready:
            return self.next_state
        return self.state


class SavingDomainSpec(DomainOrderProcessor):
    __slots__ = ()
    state = State.SAVING_DOMAIN_SPEC
    next_state = State.FINISHED
    cancelled_state = None

    def _set_cert_ids(self, domain_config_pb):
        if domain_config_pb.protocol == model_pb2.DomainSpec.Config.HTTP_ONLY:
            domain_config_pb.cert.Clear()
        else:
            if self.order.HasField('cert_ref'):
                domain_config_pb.cert.id = self.order.cert_ref.id
            elif u'cert_order_id' in self.entity.context:
                domain_config_pb.cert.id = self.entity.context[u'cert_order_id']
            if self.order.HasField('secondary_cert_ref'):
                domain_config_pb.secondary_cert.id = self.order.secondary_cert_ref.id
            elif u'secondary_cert_order_id' in self.entity.context:
                domain_config_pb.secondary_cert.id = self.entity.context[u'secondary_cert_order_id']

    def process(self, ctx):
        domain_config_pb = self.entity.pb.spec.yandex_balancer.config
        domain_config_pb.type = self.order.type
        del domain_config_pb.fqdns[:]
        del domain_config_pb.shadow_fqdns[:]
        domain_config_pb.fqdns.extend(self.order.fqdns)
        domain_config_pb.shadow_fqdns.extend(self.order.shadow_fqdns)
        domain_config_pb.include_upstreams.CopyFrom(self.order.include_upstreams)
        domain_config_pb.protocol = self.order.protocol
        if self.order.HasField('redirect_to_https'):
            domain_config_pb.redirect_to_https.CopyFrom(self.order.redirect_to_https)
        if self.order.HasField('verify_client_cert'):
            domain_config_pb.verify_client_cert.CopyFrom(self.order.verify_client_cert)
        self._set_cert_ids(domain_config_pb)

        self.entity.pb.spec.incomplete = False
        self.entity.dao_update(u'Finished order, spec.incomplete = False')
        return self.next_state


class Cancelling(DomainOrderProcessor):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.WAITING_FOR_CERT_CANCEL
    next_state_cancelled = State.CANCELLED
    cancelled_state = None

    def _cancel_cert_order(self, cert_order_id):
        try:
            for cert_pb in self.entity.zk.update_cert(namespace_id=self.entity.namespace_id, cert_id=cert_order_id):
                cancel_order(cert_pb,
                             author=util.NANNY_ROBOT_LOGIN,
                             comment=u'Cancelled because domain order was cancelled')
        except errors.NotFoundError:
            pass

    def process(self, ctx):
        for cert_id_field in (u'cert_order_id', u'secondary_cert_order_id'):
            cert_order_id = self.entity.context.get(cert_id_field)
            if cert_order_id:
                self._cancel_cert_order(cert_order_id)
        return self.next_state


class WaitingForCertsCancel(DomainOrderProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_CERT_CANCEL
    next_state = State.CANCELLED
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)

    def process(self, ctx):
        cert_order_id = self.entity.context.get(u'cert_order_id')
        if cert_order_id:
            cert_pb = self.cache.get_cert(namespace_id=self.entity.namespace_id, cert_id=cert_order_id)
            if cert_pb and is_order_in_progress(cert_pb):
                ctx.log.debug(u'primary cert is still in progress')
                return self.state
        secondary_cert_order_id = self.entity.context.get(u'secondary_cert_order_id')
        if secondary_cert_order_id:
            secondary_cert_pb = self.cache.get_cert(namespace_id=self.entity.namespace_id,
                                                    cert_id=secondary_cert_order_id)
            if secondary_cert_pb and is_order_in_progress(secondary_cert_pb):
                ctx.log.debug(u'secondary cert is still in progress')
                return self.state
        self.entity.pb.spec.incomplete = False
        self.entity.dao_update(u'Cancelled order, spec.incomplete = False')
        return self.next_state
