from collections import namedtuple

import six

from awacs.model.cache import AwacsCache, IAwacsCache
from awacs.model.db import find_l3_balancer_revision_spec, find_endpoint_set_revision
from awacs.model.l3_balancer.state_handler import L3BalancerStateHandler
from awacs.model.l3_balancer.vector import L3Vector, L3BalancerVersion, BackendVersion, EndpointSetVersion
from awacs.model.util import newer
from awacs.model.zk import IZkStorage
from infra.awacs.proto import model_pb2


l3_vectors = namedtuple('l3_vectors', ('current', 'valid', 'in_progress', 'active'))


def get_l3_balancer_version(cache, namespace_id, l3_balancer_id):
    """
    :type cache: AwacsCache
    :type namespace_id: six.text_type
    :type l3_balancer_id: six.text_type
    :rtype: L3BalancerVersion
    """
    return L3BalancerVersion.from_pb(cache.get_l3_balancer(namespace_id=namespace_id, l3_balancer_id=l3_balancer_id))


def get_backend_versions(cache, namespace_id):
    """
    :type cache: AwacsCache
    :param six.text_type namespace_id:
    :rtype: dict[six.text_type, BackendVersion]
    """
    return {backend_pb.meta.id: BackendVersion.from_pb(backend_pb)
            for backend_pb in cache.list_all_backends(namespace_id=namespace_id)}


def get_endpoint_set_versions(cache, namespace_id):
    """
    :type cache: AwacsCache
    :param six.text_type namespace_id:
    :rtype: dict[six.text_type, EndpointSetVersion]
    """
    return {endpoint_set_pb.meta.id: EndpointSetVersion.from_pb(endpoint_set_pb)
            for endpoint_set_pb in cache.list_all_endpoint_sets(namespace_id=namespace_id)}


def set_newer_version(collection, obj_id, obj_version, condition_pb):
    if condition_pb is None or condition_pb.status == 'True':
        collection[obj_id] = newer(collection.get(obj_id), obj_version)


def newer_version(prev_version, obj_version, condition_pb):
    if condition_pb.status == 'True':
        return newer(prev_version, obj_version)
    return prev_version


def l3_balancer_state_to_vectors(l3_balancer_state_pb):
    """
    :type l3_balancer_state_pb: infra.awacs.proto.model_pb2.L3BalancerState
    :rtype l3_vector(L3Vector, L3Vector, L3Vector, L3Vector)
    """
    l3_balancer_id = l3_balancer_state_pb.l3_balancer_id
    l3_latest_version = None
    l3_latest_valid_version = None
    l3_latest_in_progress_version = None
    l3_active_version = None

    backend_latest_versions = {}
    backend_latest_valid_versions = {}
    backend_latest_in_progress_versions = {}
    backend_active_versions = {}

    es_latest_versions = {}
    es_latest_valid_versions = {}
    es_active_versions = {}
    es_latest_in_progress_versions = {}

    for rev_pb in l3_balancer_state_pb.l3_balancer.l3_statuses:
        v = L3BalancerVersion.from_rev_status_pb(l3_balancer_id, rev_pb)
        l3_latest_version = newer(l3_latest_version, v)
        l3_latest_valid_version = newer_version(l3_latest_valid_version, v, rev_pb.validated)
        l3_latest_in_progress_version = newer_version(l3_latest_in_progress_version, v, rev_pb.in_progress)
        l3_active_version = newer_version(l3_active_version, v, rev_pb.active)

    for backend_id, backend_state_pb in six.iteritems(l3_balancer_state_pb.backends):
        for rev_pb in backend_state_pb.l3_statuses:
            v = BackendVersion.from_rev_status_pb(backend_id, rev_pb)
            set_newer_version(backend_latest_versions, backend_id, v, None)
            set_newer_version(backend_latest_valid_versions, backend_id, v, rev_pb.validated)
            set_newer_version(backend_latest_in_progress_versions, backend_id, v, rev_pb.in_progress)
            set_newer_version(backend_active_versions, backend_id, v, rev_pb.active)

    for endpoint_set_id, endpoint_set_state_pb in six.iteritems(l3_balancer_state_pb.endpoint_sets):
        for rev_pb in endpoint_set_state_pb.l3_statuses:
            v = EndpointSetVersion.from_rev_status_pb(endpoint_set_id, rev_pb)
            set_newer_version(es_latest_versions, endpoint_set_id, v, None)
            set_newer_version(es_latest_valid_versions, endpoint_set_id, v, rev_pb.validated)
            set_newer_version(es_latest_in_progress_versions, endpoint_set_id, v, rev_pb.in_progress)
            set_newer_version(es_active_versions, endpoint_set_id, v, rev_pb.active)

    current_vector = L3Vector(l3_latest_version, backend_latest_versions, es_latest_versions)
    valid_vector = L3Vector(l3_latest_valid_version, backend_latest_valid_versions, es_latest_valid_versions)
    in_progress_vector = L3Vector(l3_latest_in_progress_version, backend_latest_in_progress_versions,
                                  es_latest_in_progress_versions)
    active_vector = L3Vector(l3_active_version, backend_active_versions, es_active_versions)

    return l3_vectors(current_vector, valid_vector, in_progress_vector, active_vector)


def find_l3_balancer_revision_spec_and_use_cache(namespace_id, l3_balancer_id, version):
    """
    :type namespace_id: six.text_type
    :type l3_balancer_id: six.text_type
    :type version: six.text_type
    :rtype: model_pb2.L3BalancerSpec
    """
    return find_l3_balancer_revision_spec(
        version, current_pb=IAwacsCache.instance().get_l3_balancer(namespace_id, l3_balancer_id))


def find_endpoint_set_revision_and_use_cache(namespace_id, endpoint_set_id, version):
    """
    :type namespace_id: six.text_type
    :type endpoint_set_id: six.text_type
    :type version: six.text_type
    :rtype: model_pb2.EndpointSet
    """
    return find_endpoint_set_revision(
        version, current_pb=IAwacsCache.instance().get_endpoint_set(namespace_id, endpoint_set_id))


def find_endpoint_set_revision_spec_and_use_cache(namespace_id, endpoint_set_id, version):
    """
    :type namespace_id: six.text_type
    :type endpoint_set_id: six.text_type
    :type version: six.text_type
    :rtype: model_pb2.EndpointSetSpec
    """
    return find_endpoint_set_revision_and_use_cache(namespace_id, endpoint_set_id, version).spec


def get_included_backend_ids(l3_balancer_spec_pb):
    """
    :type l3_balancer_spec_pb: infra.awacs.proto.model_pb2.L3BalancerSpec
    :rtype: set[six.text_type]
    """
    selector_type = l3_balancer_spec_pb.real_servers.type
    if selector_type == model_pb2.L3BalancerRealServersSelector.BACKENDS:
        return [backend_pb.id for backend_pb in l3_balancer_spec_pb.real_servers.backends]
    elif selector_type == model_pb2.L3BalancerRealServersSelector.BALANCERS:
        # system backend's id always matches balancer id
        return [balancer_pb.id for balancer_pb in l3_balancer_spec_pb.real_servers.balancers]
    else:
        raise AssertionError('Unknown L3 selector type {}'.format(selector_type))


def modify_state_revisions(namespace_id, l3_balancer_id, l3_balancer_state_pb, versions_to_add, backend_ids_to_delete):
    """
    :type namespace_id: six.text_type
    :type l3_balancer_id: six.text_type
    :type l3_balancer_state_pb: infra.awacs.proto.model_pb2.L3BalancerState
    :type versions_to_add: set
    :type backend_ids_to_delete: set
    :rtype: bool
    """
    updated = False

    for l3_balancer_state_pb in IZkStorage.instance().update_l3_balancer_state(namespace_id,
                                                                               l3_balancer_id,
                                                                               l3_balancer_state_pb):
        h = L3BalancerStateHandler(l3_balancer_state_pb)
        updated = False
        for version in versions_to_add:
            updated |= h.add_new_if_missing(version)
        for backend_id in backend_ids_to_delete:
            updated |= h.delete_backend(backend_id)
            updated |= h.delete_endpoint_set(backend_id)
        if not updated:
            break

    return updated


def skip_in_progress_l3mgr_config(namespace_id, l3_balancer_id,
                                  l3mgr_service_id_to_skip, l3mgr_config_id_to_skip,
                                  l3_balancer_state_pb=None,
                                  skipped_vector_hash=None,
                                  ignore_existing_config=True):
    """
    :type namespace_id: six.text_type
    :type l3_balancer_id: six.text_type
    :type l3mgr_service_id_to_skip: six.text_type
    :type l3mgr_config_id_to_skip: six.text_type
    :type l3_balancer_state_pb: model_pb2.L3BalancerState | None
    :type skipped_vector_hash: six.text_type | None
    :type ignore_existing_config: bool
    :rtype: model_pb2.L3BalancerState
    """
    zk = IZkStorage.instance()

    def omit_in_progress_config(pb):
        if pb.in_progress.status != 'True':
            return False

        l3mgr_pb = pb.in_progress.meta.l3mgr  # type: model_pb2.L3mgrTransportMeta

        updated_config_pbs = []
        for config_pb in l3mgr_pb.configs:
            if (config_pb.service_id != l3mgr_service_id_to_skip or
                    config_pb.config_id != l3mgr_config_id_to_skip):
                updated_config_pbs.append(config_pb)

        if list(l3mgr_pb.configs) == updated_config_pbs:
            return False

        del l3mgr_pb.configs[:]
        l3mgr_pb.configs.extend(updated_config_pbs)
        if not l3mgr_pb.configs:
            pb.in_progress.status = 'False'
            pb.in_progress.ClearField('meta')
        return True

    updated = False
    for l3_balancer_state_pb in zk.update_l3_balancer_state(namespace_id, l3_balancer_id,
                                                            l3_balancer_state_pb=l3_balancer_state_pb):
        updated = False
        h = L3BalancerStateHandler(l3_balancer_state_pb)
        updated |= h.select_l3_balancer().update_revs(omit_in_progress_config)
        if ignore_existing_config:
            set_ignore_updated = h.set_ignore_existing_l3mgr_config(
                True, author=u'awacs', comment=u'Forcing creation of fresh config in L3Mgr')
        else:
            set_ignore_updated = False
        if set_ignore_updated:
            updated = True
            if skipped_vector_hash is not None:
                h.increment_skip_count(skipped_vector_hash)
        for backend_id in l3_balancer_state_pb.backends:
            updated |= h.select_backend(backend_id).update_revs(omit_in_progress_config)
            updated |= h.select_endpoint_set(backend_id).update_revs(omit_in_progress_config)
        if not updated:
            break

    return l3_balancer_state_pb, updated
