import random

import inject
import monotonic

from awacs.lib import ctlmanager
from awacs.model import events, cache
from awacs.model.l3_balancer import discoverer, transport, validator


class L3BalancerCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    POLL_INTERVAL = 20
    PROCESS_INTERVAL = 5
    FORCE_PROCESS_INTERVAL = 120
    FORCE_PROCESS_INTERVAL_JITTER = 20
    EVENTS_QUEUE_GET_TIMEOUT = 5

    def __init__(self, namespace_id, l3_balancer_id):
        name = u'l3-balancer-ctl("{}:{}")'.format(namespace_id, l3_balancer_id)
        super(L3BalancerCtl, self).__init__(name)

        self._namespace_id = namespace_id
        self._l3_balancer_id = l3_balancer_id

        self._init_processors()

        current_time = monotonic.monotonic()
        self._waiting_for_processing_since = None
        self._processed_at = None
        self._processing_deadline = None
        self._force_processing_deadline = current_time  # to trigger first processing immediately after start
        self._polled_at = None
        self._polling_deadline = current_time  # to trigger first polling immediately after start

        self._l3_balancer_event_path = u'{}/{}'.format(self._namespace_id, self._l3_balancer_id)
        self._namespace_event_prefix = self._namespace_id + u'/'

    def _init_processors(self):
        self._discoverer = discoverer.Discoverer(self._namespace_id, self._l3_balancer_id)
        self._transport = transport.Transport(self._namespace_id, self._l3_balancer_id)
        self._validator = validator.Validator(self._namespace_id, self._l3_balancer_id)

    def _accept_event(self, event):
        """
        :type event: events.*
        :rtype: bool
        """
        if isinstance(event, (events.L3BalancerUpdate, events.L3BalancerRemove,
                              events.L3BalancerStateUpdate, events.L3BalancerStateRemove)):
            return event.path == self._l3_balancer_event_path
        elif isinstance(event, (events.BackendUpdate, events.BackendRemove,
                                events.EndpointSetUpdate, events.EndpointSetRemove)):
            return event.path.startswith(self._namespace_event_prefix)
        else:
            return False

    def _start(self, ctx):
        self._log.info(u'starting...')
        self._cache.bind(self._callback)
        self._log.info(u'started')

    def _stop(self):
        self._log.info(u'stopping...')
        self._cache.unbind(self._callback)
        self._log.info(u'stopped')

    def _process_event(self, ctx, event):
        """
        Don't process every event, it is too costly. Instead, just start a timer.
        """
        if self._waiting_for_processing_since is None:
            self._waiting_for_processing_since = monotonic.monotonic()
            self._processing_deadline = self._waiting_for_processing_since + self.PROCESS_INTERVAL

    def _process_empty_queue(self, ctx):
        curr_time = monotonic.monotonic()
        should_process = self._should_process(curr_time)
        should_poll = self._should_poll(curr_time)
        if not should_process and not should_poll:
            return

        l3_balancer_state_pb = self._cache.must_get_l3_balancer_state(self._namespace_id, self._l3_balancer_id)

        is_state_updated = False
        if should_process:
            is_state_updated = self._do_process(ctx, l3_balancer_state_pb)
            self._reset_processing_timers()

        if not is_state_updated and should_poll:
            self._do_poll(ctx, l3_balancer_state_pb)
            self._reset_polling_timers()

    def _should_process(self, curr_time):
        """
        :type curr_time: float

        Process changes at most every self.PROCESS_INTERVAL seconds,
        and at least every self.FORCE_PROCESS_INTERVAL seconds
        """
        if self._processing_deadline is not None and curr_time >= self._processing_deadline:
            return True
        return curr_time >= self._force_processing_deadline

    def _do_process(self, ctx, l3_balancer_state_pb):
        is_state_updated = self._discoverer.discover(ctx, l3_balancer_state_pb)
        if not is_state_updated:
            is_state_updated = self._validator.validate(ctx, l3_balancer_state_pb)
            if not is_state_updated:
                is_state_updated = self._transport.transport(ctx, l3_balancer_state_pb)
                if not is_state_updated:
                    is_state_updated = self._transport.skip_stuck(ctx, l3_balancer_state_pb)
                    if is_state_updated:
                        ctx.log.debug(u'state is updated after skip_stuck()')
                else:
                    ctx.log.debug(u'state is updated after transport()')
            else:
                ctx.log.debug(u'state is updated after validate()')
        else:
            ctx.log.debug(u'state is updated after discover()')
        return is_state_updated

    def _reset_processing_timers(self):
        self._waiting_for_processing_since = None
        self._processing_deadline = None
        self._processed_at = monotonic.monotonic()
        self._force_processing_deadline = self._processed_at + random.randint(
            self.FORCE_PROCESS_INTERVAL - self.FORCE_PROCESS_INTERVAL_JITTER,
            self.FORCE_PROCESS_INTERVAL + self.FORCE_PROCESS_INTERVAL_JITTER)

    def _should_poll(self, curr_time):
        """
        :type curr_time: float
        """
        return curr_time >= self._polling_deadline

    def _do_poll(self, ctx, l3_balancer_state_pb):
        is_state_updated = self._transport.poll_configs(ctx, l3_balancer_state_pb)
        if not is_state_updated:
            is_state_updated = self._discoverer.clean_l3_balancer_state(ctx, l3_balancer_state_pb)
        return is_state_updated

    def _reset_polling_timers(self):
        self._polled_at = monotonic.monotonic()
        self._polling_deadline = self._polled_at + self.POLL_INTERVAL
