import functools

import attr
import datetime
import re
import six
from boltons import dictutils
from six.moves import http_client as httplib, zip
from typing import List, Dict, Optional

from awacs.lib import l3mgrclient
from awacs.lib.strutils import removeprefix
from awacs.model.l3_balancer import errors
from awacs.model.util import get_diff_between_dicts
from infra.awacs.proto import model_pb2


l3mgr_rs_normalize_line_re = re.compile(r'\s*=\s*')


@attr.s(slots=True, weakref_slot=True, frozen=True)
class Service(object):
    id = attr.ib(type=six.text_type)
    config_id = attr.ib(type=six.text_type)
    fqdn = attr.ib(type=six.text_type)
    abc_slug = attr.ib(type=six.text_type)
    virtual_servers = attr.ib(type=List[Dict])
    config = attr.ib(type=Dict)
    meta = attr.ib(type=Dict)

    @classmethod
    def from_api(cls, client, service_id):
        """
        :type client: l3mgrclient.L3MgrClient
        :type service_id: six_text_type | int
        :rtype: Service
        """
        raw_data = client.get_service(service_id)
        return cls(id=service_id,
                   fqdn=raw_data[u'fqdn'],
                   abc_slug=six.text_type(raw_data.get(u'abc')),
                   virtual_servers=raw_data.get(u'vs'),
                   config=raw_data.get(u'config'),
                   config_id=raw_data.get(u'config', {}).get(u'id'),
                   meta=raw_data.get(u'meta'))

    @property
    def has_active_config(self):
        return self.config_id is not None

    def get_new_ip(self, client, is_v4, is_external):
        ip_resp = client.get_new_ip(
            abc_code=self.abc_slug,
            v4=is_v4,
            external=is_external,
            fqdn=self.fqdn)
        return ip_resp[u'object']


@attr.s(slots=True, weakref_slot=True, frozen=True)
class ServiceConfig(object):
    _datetime_str = u'%Y-%m-%dT%H:%M:%S.%fZ'

    service_id = attr.ib(type=six.text_type)
    id = attr.ib(type=six.text_type)
    state = attr.ib(type=six.text_type)
    timestamp = attr.ib(type=six.text_type)
    raw_data = attr.ib(type=Dict)

    @classmethod
    def from_raw_data(cls, service_id, raw_data):
        """
        :type service_id: six.text_type
        :type raw_data: Dict[six.text_type, Any]
        :rtype: ServiceConfig
        """
        return cls(service_id=service_id,
                   id=raw_data[u'id'],
                   state=raw_data[u'state'],
                   timestamp=raw_data[u'timestamp'],
                   raw_data=raw_data)

    @property
    def is_active(self):
        return self.state == u'ACTIVE'

    @property
    def is_new(self):
        return self.state == u'NEW'

    @property
    def vs_ids(self):
        return self.raw_data.get(u'vs_id', [])

    @classmethod
    def from_api(cls, client, service_id, config_id):
        """
        :type client: l3mgrclient.L3MgrClient
        :type service_id: six.text_type
        :type config_id: six.text_type | int
        :rtype: ServiceConfig
        """
        config_raw_data = client.get_config(service_id, config_id)
        return cls.from_raw_data(service_id=service_id, raw_data=config_raw_data)

    @classmethod
    def latest_from_api_if_exists(cls, client, service_id):
        """
        :type client: l3mgrclient.L3MgrClient
        :type service_id: six.text_type
        :rtype: ServiceConfig | None
        """
        latest_config = client.get_latest_config(service_id)
        if latest_config is not None:
            return cls.from_raw_data(service_id=service_id, raw_data=latest_config)
        else:
            return None

    @classmethod
    def latest_from_api(cls, client, service_id):
        """
        :type client: l3mgrclient.L3MgrClient
        :type service_id: six.text_type
        :rtype: ServiceConfig
        :raises errors.L3ConfigValidationError
        """
        latest_config = cls.latest_from_api_if_exists(client, service_id)
        if latest_config is not None:
            return latest_config
        raise errors.L3ConfigValidationError(
            u'L3Manager service "{}" does not have any VS configs'.format(service_id))

    @classmethod
    def from_api_if_exists(cls, client, service_id, config_id):
        """
        :type client: l3mgrclient.L3MgrClient
        :type service_id: six.text_type
        :type config_id: six.text_type
        :rtype Optional[ServiceConfig]
        """
        try:
            config_raw_data = client.get_config(service_id, config_id)
            return cls.from_raw_data(service_id=service_id, raw_data=config_raw_data)
        except l3mgrclient.L3MgrException as e:
            if e.resp is not None and e.resp.status_code == httplib.NOT_FOUND:
                return None
            else:
                raise

    @classmethod
    def create_and_process(cls, client, service_id, vs_ids, comment=None, force_process=False):
        """
        :type client: l3mgrclient.L3MgrClient
        :type service_id: six.text_type
        :type vs_ids: list[int | six.text_type]
        :type comment: six.text_type
        :type force_process: bool
        """
        resp = client.create_config_with_vs(svc_id=service_id,
                                            vs_ids=vs_ids,
                                            comment=comment or u'Add virtual servers',
                                            use_etag=False)
        result = resp[u'result']
        if result != u'OK':
            raise errors.L3BalancerTransportError(u'create_config_with_vs: result is not "OK": {}'.format(result))
        return cls._process_and_get(client, service_id, resp[u'object'][u'id'], force=force_process)

    @classmethod
    def update_vs_and_process(cls, client, service_id, latest_config, vs_ids, comment=None, force_process=False):
        """
        :type client: l3mgrclient.L3MgrClient
        :type service_id: six.text_type
        :type latest_config: ServiceConfig
        :type vs_ids: list[int | six.text_type]
        :type comment: six.text_type
        :type force_process: bool
        """
        resp = client.create_config_with_vs(svc_id=service_id,
                                            vs_ids=vs_ids,
                                            comment=comment or u'Update virtual servers',
                                            use_etag=True,
                                            latest_cfg_id=latest_config.id)
        result = resp[u'result']
        if result != u'OK':
            raise errors.L3BalancerTransportError(u'create_config_with_vs: result is not "OK": {}'.format(result))
        return cls._process_and_get(client, service_id, resp[u'object'][u'id'], force=force_process)

    def process(self, client, force=False):
        return self._process_and_get(client, self.service_id, self.id, force=force)

    @classmethod
    def _process_and_get(cls, client, service_id, config_id, force):
        resp = client.process_config(service_id, config_id, use_etag=True, latest_cfg_id=config_id, force=force)
        result = resp[u'result']
        if result != u'OK':
            raise errors.L3BalancerTransportError(u'process_config: result is not "OK": {}'.format(result))
        return cls.from_api(client, service_id, config_id)

    def to_pb(self):
        config_pb = model_pb2.L3mgrConfig(service_id=six.text_type(self.service_id), config_id=six.text_type(self.id))
        ctime_dt = datetime.datetime.strptime(self.timestamp, self._datetime_str)
        config_pb.ctime.FromDatetime(ctime_dt)
        return config_pb


@attr.s(slots=True, weakref_slot=True, frozen=True, cmp=False)
@functools.total_ordering
class RealServer(object):
    ALLOWED_PARAMS = (u'weight',)

    fqdn = attr.ib(type=six.text_type)
    ip = attr.ib(type=six.text_type)
    weight = attr.ib(default=None, type=Optional[int], converter=lambda x: int(x) if x is not None else None)
    config = attr.ib(init=False, factory=dict, type=dict)

    def __attrs_post_init__(self):
        if self.weight is not None:
            object.__setattr__(self, 'config', {u'weight': self.weight})

    @classmethod
    def from_l3mgr_string(cls, line):
        """
        :param six.text_type line: configuration string from the L3mgr form, like `domain[=ip][ param=x]`
        :raises UnsupportedParameter: if a param other than ALLOWED_PARAMS is encountered
        """
        line = l3mgr_rs_normalize_line_re.sub(r'=', six.ensure_str(line)).strip()
        fqdn_and_ip, _, params = line.partition(u' ')
        fqdn, _, ip = fqdn_and_ip.partition(u'=')
        weight = None
        for param in params.split():
            param, eq, value = param.partition(u'=')
            if param and param not in RealServer.ALLOWED_PARAMS:
                raise errors.UnsupportedParameter(u'Unsupported RS parameter "{}"'.format(param))
            if param == u'weight':
                weight = value
        return cls(fqdn=fqdn, ip=ip, weight=weight)

    def matches(self, other, ignore_ip=False):
        """
        :type other: RealServer
        :type ignore_ip: bool
        :rtype: bool
        """
        assert isinstance(other, RealServer)
        return (
                self.fqdn == other.fqdn
                and self.config == other.config
                and (self.ip == other.ip or
                     (ignore_ip and (not self.ip or not other.ip)))
        )

    def __lt__(self, other):
        return self.fqdn < other.fqdn

    def __eq__(self, other):
        return self.fqdn == other.fqdn and self.ip == other.ip and self.config == other.config

    def __hash__(self):
        return hash(six.text_type(self))

    def __str__(self):
        config = []
        for k, v in six.iteritems(self.config):
            config.append(u'{}={}'.format(k, v))

        return u'{fqdn}{ip} {config}'.format(
            fqdn=self.fqdn,
            ip=u'={}'.format(self.ip) if self.ip else u'',
            config=u' '.join(sorted(config))).strip()

    __unicode__ = __repr__ = __str__


class RSGroup(object):
    __slots__ = (u'real_servers', u'need_to_update_ip_addresses',)

    def __init__(self, real_servers=None, need_to_update_ip_addresses=False):
        """
        :type real_servers: Iterable[RealServer]
        :type need_to_update_ip_addresses: bool
        """
        if real_servers is None:
            real_servers = set()
        self.real_servers = real_servers
        self.need_to_update_ip_addresses = need_to_update_ip_addresses

    def add(self, fqdn, ip, weight=None):
        self.real_servers.add(RealServer(fqdn, ip, weight))

    def add_rs(self, rs):
        assert isinstance(rs, RealServer)
        self.real_servers.add(rs)

    @classmethod
    def from_vs(cls, vs):
        return cls(real_servers=sorted(RealServer.from_l3mgr_string(rs) for rs in vs[u'group']))

    @classmethod
    def from_l3mgr_virtual_servers(cls, virtual_servers):
        """
        :type virtual_servers: Iterable[dict]
        """
        first_rs_group = None
        first_vs_id = None
        need_to_update_ip_addresses = False
        for vs in virtual_servers:
            rs_group = vs[u'group']
            new_rs_group = sorted(RealServer.from_l3mgr_string(rs) for rs in rs_group)
            if first_rs_group is None:
                first_rs_group = new_rs_group
                first_vs_id = vs[u'id']
                continue  # remember the first group as canonical
            groups_have_significant_differences = len(first_rs_group) != len(new_rs_group)
            if not groups_have_significant_differences:
                for rs, new_rs in zip(first_rs_group, new_rs_group):
                    if not rs.matches(new_rs, ignore_ip=True):
                        groups_have_significant_differences = True
                        break
                    elif bool(rs.ip) != bool(new_rs.ip):
                        # https://st.yandex-team.ru/AWACS-968
                        # If RS has unfilled IP address in one VS and filled IP address in another,
                        # we want to push awacs config to update it everywhere.
                        need_to_update_ip_addresses = True
                        break
            if groups_have_significant_differences:
                raise errors.RSGroupsConflict(
                    u'RS groups are not the same: {{vs[{0}]: {2}, vs[{1}]: {3}}}'.format(
                        first_vs_id, vs[u'id'],
                        [six.text_type(rs) for rs in first_rs_group],
                        [six.text_type(rs) for rs in new_rs_group]))
        return cls(real_servers=first_rs_group, need_to_update_ip_addresses=need_to_update_ip_addresses)

    def matches(self, other, ignore_ip=False):
        """
        :type other: RSGroup
        :type ignore_ip: bool
        :rtype: bool
        """
        assert isinstance(other, RSGroup)
        if len(self.real_servers) != len(other.real_servers):
            return False
        for rs1, rs2 in zip(sorted(self.real_servers), sorted(other.real_servers)):
            if not rs1.matches(rs2, ignore_ip=ignore_ip):
                return False
        return True


class VSConfigKey(object):
    # shared between VS
    scheduler = u'SCHEDULER'
    method = u'METHOD'
    announce = u'ANNOUNCE'
    dc_filter = u'DC_FILTER'
    dynamicweight = u'DYNAMICWEIGHT'
    dynamicweight_allow_zero = u'DYNAMICWEIGHT_ALLOW_ZERO'
    dynamicweight_ratio = u'DYNAMICWEIGHT_RATIO'
    status_code = u'STATUS_CODE'
    mh_fallback = u'MH_FALLBACK'

    # individual for each VS
    check_url = u'CHECK_URL'
    check_type = u'CHECK_TYPE'
    connect_port = u'CONNECT_PORT'


@attr.s(slots=True, weakref_slot=True, frozen=True, cmp=False)
class VirtualServer(object):
    default_config = {
        VSConfigKey.scheduler: u'wrr',
        VSConfigKey.method: u'TUN',
        VSConfigKey.announce: True,
        VSConfigKey.dc_filter: True,
        VSConfigKey.dynamicweight: True,
        VSConfigKey.dynamicweight_allow_zero: True,
        VSConfigKey.dynamicweight_ratio: 30,
        VSConfigKey.status_code: 200,
    }

    ignored_config_values = {
        # we don't send these config values to L3 Manager ourselves,
        # they are used only for diff checking during MODE_RS_AND_VS migration
        u'CHECK_RETRY_TIMEOUT': 1,
        u'QUORUM': 1,
        u'DELAY_LOOP': 10,
        u'CHECK_TIMEOUT': 1,
        u'CHECK_RETRY': 1,
        u'MH_FALLBACK': True,
    }

    shared_config_keys = list(default_config)
    _per_vs_config_keys = [VSConfigKey.check_url, VSConfigKey.check_type, VSConfigKey.connect_port]
    _important_config_keys = shared_config_keys + _per_vs_config_keys

    service_id = attr.ib(type=six.text_type, converter=six.text_type)
    ip = attr.ib(type=six.text_type, converter=six.text_type)
    port = attr.ib(type=int, converter=int)
    id = attr.ib(type=Optional[int])
    config = attr.ib(type=Dict)
    traffic_type = attr.ib(type=model_pb2.L3BalancerSpec.VirtualServer.TrafficType)
    rs_group = attr.ib(type=RSGroup)

    @property
    def shared_config(self):
        return dictutils.subdict(self.config, keep=self.shared_config_keys)

    @property
    def connect_port(self):
        return int(self.config[VSConfigKey.connect_port]) if VSConfigKey.connect_port in self.config else None

    @classmethod
    def get_default_config_value(cls, key):
        return cls.default_config.get(key)

    @classmethod
    def get_ignored_config_value(cls, key):
        return cls.ignored_config_values.get(key)

    @classmethod
    def from_raw_data(cls, service_id, raw_data):
        """
        :type service_id: six.text_type
        :type raw_data: Dict[six.text_type, Any]
        :rtype VirtualServer
        """
        return cls(service_id=service_id,
                   config=dictutils.subdict(raw_data[u'config'], keep=cls._important_config_keys),
                   ip=raw_data[u'ip'],
                   port=raw_data[u'port'],
                   id=raw_data[u'id'],
                   rs_group=RSGroup.from_vs(raw_data),
                   traffic_type=None,  # noqa TRAFFIC-12333
                   )

    @classmethod
    def from_api(cls, client, service_id, vs_id):
        vs = client.get_vs(service_id, vs_id)
        return cls.from_raw_data(service_id, vs)

    def to_pb(self, traffic_type=None):
        """
        :type traffic_type: model_pb2.L3BalancerSpec.VirtualServer.TrafficType
        :rtype: model_pb2.L3BalancerSpec.VirtualServer
        """
        vs_pb = model_pb2.L3BalancerSpec.VirtualServer(ip=self.ip, port=self.port)
        vs_pb.health_check_settings.url = self.config[VSConfigKey.check_url]
        vs_pb.health_check_settings.check_type = get_check_type_enum(self.config[VSConfigKey.check_type])
        tt = traffic_type or self.traffic_type
        if not tt:
            raise ValueError(u'traffic_type must be set')
        vs_pb.traffic_type = tt
        return vs_pb

    @classmethod
    def from_vs_pb(cls, service_id, vs_pb, rs_group):
        """
        :type service_id: six.text_type
        :type vs_pb: model_pb2.L3BalancerSpec.VirtualServer
        :type rs_group: RSGroup
        :rtype VirtualServer
        """
        config = dict(cls.default_config)
        config.update({
            VSConfigKey.check_url: vs_pb.health_check_settings.url,
            VSConfigKey.check_type: get_check_type_str(vs_pb.health_check_settings),
            VSConfigKey.connect_port: vs_pb.port,
        })
        return cls(service_id,
                   ip=vs_pb.ip,
                   port=vs_pb.port,
                   id=None,
                   config=config,
                   traffic_type=vs_pb.traffic_type,
                   rs_group=rs_group)

    def create_in_l3mgr(self, client, config=None):
        """
        :type client: l3mgrclient.L3MgrClient
        :type config: Dict | None
        :rtype VirtualServer
        """
        groups = sorted(six.text_type(rs) for rs in self.rs_group.real_servers)
        resp = client.create_virtual_server(svc_id=self.service_id,
                                            ip=self.ip,
                                            port=self.port,
                                            protocol=u'TCP',
                                            config=config or self.config,
                                            groups=groups)
        vs_id = resp[u'object'][u'id']
        return self.from_api(client, self.service_id, vs_id)

    def matches(self, other):
        """
        :type other: VirtualServer
        """
        assert isinstance(other, VirtualServer)
        return self.config == other.config and self.rs_group.matches(other.rs_group)


class VirtualServers(object):
    __slots__ = (u'service_id', u'shared_config', u'virtual_servers')

    def __init__(self, svc_id, shared_config, virtual_servers):
        """
        :type svc_id: six.text_type
        :type shared_config: dict[six.text_type, six.text_type]
        :type virtual_servers: dict[[six.text_type, int], VirtualServer]
        """
        self.service_id = svc_id
        self.shared_config = shared_config
        self.virtual_servers = virtual_servers

    @property
    def vs_ids(self):
        return sorted(vs.id for vs in six.itervalues(self.virtual_servers))

    @classmethod
    def from_l3mgr_raw_virtual_servers(cls, svc_id, l3mgr_vs, allow_l3mgr_configs_mismatch=False):
        """
        :type svc_id: six.text_type
        :type l3mgr_vs: list[dict]
        :type allow_l3mgr_configs_mismatch: bool
        :rtype: VirtualServers
        """
        first_vs = None
        virtual_servers = {}
        for raw_vs in l3mgr_vs:
            vs = VirtualServer.from_raw_data(svc_id, raw_vs)
            if first_vs is None:
                first_vs = vs
            elif not allow_l3mgr_configs_mismatch:
                cls._compare_vs(first_vs, vs)
            virtual_servers[(vs.ip, vs.port)] = vs
        return cls(svc_id, first_vs.shared_config, virtual_servers)

    @classmethod
    def from_api(cls, client, svc_id, vs_ids, allow_l3mgr_configs_mismatch=False):
        """
        :type client: l3mgrclient.L3MgrClient
        :type svc_id: six.text_type
        :type vs_ids: list[six.text_type]
        :type allow_l3mgr_configs_mismatch: bool
        :rtype: VirtualServers
        """
        if not vs_ids:
            return cls(svc_id, {}, {})
        first_vs = None
        virtual_servers = {}
        for vs_id in vs_ids:
            vs = VirtualServer.from_api(client, svc_id, vs_id)
            if first_vs is None:
                first_vs = vs
            elif not allow_l3mgr_configs_mismatch:
                cls._compare_vs(first_vs, vs)
            virtual_servers[(vs.ip, vs.port)] = vs
        return cls(svc_id, first_vs.shared_config, virtual_servers)

    def add_or_modify_vs(self, client, vs_pb, rs_group):
        """
        :type client: l3mgrclient.L3MgrClient
        :type vs_pb: model_pb2.L3BalancerSpec.VirtualServer
        :type rs_group: RSGroup
        :return: True if any changes were made in L3mgr
        :rtype: bool
        """
        vs_key = (vs_pb.ip, vs_pb.port)
        new_vs = VirtualServer.from_vs_pb(self.service_id, vs_pb, rs_group)
        if vs_key not in self.virtual_servers or not self.virtual_servers[vs_key].matches(new_vs):
            self.virtual_servers[vs_key] = new_vs.create_in_l3mgr(client)
            return True
        return False

    def update_shared_config(self, client, updated_items):
        assert set(updated_items).issubset(VirtualServer.shared_config_keys)

        updated = False
        for vs_key, vs in six.iteritems(self.virtual_servers):
            old_config = vs.config
            new_config = dict(old_config)
            new_config.update(updated_items)
            if new_config != old_config:
                updated = True
                self.virtual_servers[vs_key] = vs.create_in_l3mgr(client, config=new_config)
        return updated

    @staticmethod
    def _compare_vs(vs1, vs2):
        if vs1.shared_config != vs2.shared_config:
            config_diffs = get_diff_between_dicts(vs1.shared_config, vs2.shared_config)
            raise errors.VSConfigsConflict(
                u'VS configs are not the same, diff: {{vs[{0}]: {2}, vs[{1}]: {3}}}'.format(
                    vs1.id, vs2.id, *config_diffs))
        if not vs1.rs_group.matches(vs2.rs_group, ignore_ip=True):
            raise errors.RSGroupsConflict(
                u'RS groups are not the same: {{vs[{0}]: {2}, vs[{1}]: {3}}}'.format(
                    vs1.id, vs2.id,
                    [six.text_type(rs) for rs in vs1.rs_group.real_servers],
                    [six.text_type(rs) for rs in vs2.rs_group.real_servers]))


def get_check_type_str(health_check_pb):
    return removeprefix(health_check_pb.CheckType.Name(health_check_pb.check_type), u'CT_')


def get_check_type_enum(check_type_str):
    if check_type_str == u'HTTP_GET':
        return model_pb2.L3BalancerSpec.VirtualServer.HealthCheckSettings.CT_HTTP_GET
    elif check_type_str == u'SSL_GET':
        return model_pb2.L3BalancerSpec.VirtualServer.HealthCheckSettings.CT_SSL_GET
    raise errors.UnsupportedParameter(u'Unknown check type "{}"'.format(check_type_str))


def is_fully_managed(l3_balancer_spec_or_order_pb):
    return l3_balancer_spec_or_order_pb.config_management_mode == model_pb2.L3BalancerSpec.MODE_REAL_AND_VIRTUAL_SERVERS
