from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import has_actionable_spec, is_order_in_progress, needs_removal
from awacs.model import events
from .ctl import L3BalancerCtl
from .ctl_v2 import L3BalancerCtlV2
from .l3mgr import is_fully_managed
from .order.ctl import L3BalancerOrderCtl


class L3BalancerCtlManager(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(L3BalancerCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='l3-balancer',
            subscribed_events=(events.L3BalancerUpdate, events.L3BalancerRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.L3BalancerRemove):
            return False
        return has_actionable_spec(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for l3_balancer_pb in self._cache.list_all_l3_balancers(namespace_id):
                yield events.L3BalancerUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, l3_balancer_pb.meta.id),
                    pb=l3_balancer_pb)

    def _should_restart_ctl(self, ctl_id):
        """
        :type ctl_id: (six.text_type, six.text_type)
        :rtype bool
        """
        l3_balancer_pb = self._cache.get_l3_balancer(*ctl_id)
        if l3_balancer_pb is None:
            return False
        ctl = self._ctls.get(ctl_id)
        if ctl is not None:
            if l3_balancer_pb.spec.ctl_version >= 2 or is_fully_managed(l3_balancer_pb.spec):
                return isinstance(ctl.controller, L3BalancerCtl)
            else:
                return isinstance(ctl.controller, L3BalancerCtlV2)

    def _create_ctl(self, ctl_id):
        l3_balancer_pb = self._cache.must_get_l3_balancer(*ctl_id)
        if l3_balancer_pb.spec.ctl_version >= 2 or is_fully_managed(l3_balancer_pb.spec):
            return L3BalancerCtlV2(*ctl_id)
        else:
            return L3BalancerCtl(*ctl_id)


class L3BalancerOrderCtlManager(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(L3BalancerOrderCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            entity_name='l3-balancer-order',
            subscribed_events=(events.L3BalancerUpdate, events.L3BalancerRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.L3BalancerRemove):
            return False
        return is_order_in_progress(event.pb) and not needs_removal(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            for l3_balancer_pb in self._cache.list_all_l3_balancers(namespace_id):
                yield events.L3BalancerUpdate(
                    path=self._make_event_path_from_ctl_id(namespace_id, l3_balancer_pb.meta.id),
                    pb=l3_balancer_pb)

    def _create_ctl(self, ctl_id):
        return L3BalancerOrderCtl(*ctl_id)
