import inject

from awacs.lib import ctlmanager
from awacs.lib.order_processor.model import is_spec_complete
from awacs.lib.order_processor.runner import StateRunner
from awacs.model import events, cache
from awacs.model.l3_balancer.order.processors import L3BalancerOrder
from infra.awacs.proto import model_pb2


class L3BalancerOrderCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    EVENTS_QUEUE_GET_TIMEOUT = 5
    PROCESSING_INTERVAL = 3

    _runner = StateRunner(entity_class=L3BalancerOrder,
                          initial_state=L3BalancerOrder.states.STARTED,
                          final_state=L3BalancerOrder.states.FINISHED,
                          final_cancelled_state=L3BalancerOrder.states.CANCELLED,
                          processors=L3BalancerOrder.get_processors(),
                          processing_interval=PROCESSING_INTERVAL)

    def __init__(self, namespace_id, l3_balancer_id):
        name = u'l3-balancer-order-ctl("{}:{}")'.format(namespace_id, l3_balancer_id)
        super(L3BalancerOrderCtl, self).__init__(name)
        self._l3_balancer_id = l3_balancer_id
        self._namespace_id = namespace_id
        self._pb = None  # type: model_pb2.L3Balancer or None

    def _accept_event(self, event):
        return (isinstance(event, events.L3BalancerUpdate) and
                event.pb.meta.namespace_id == self._namespace_id and
                event.pb.meta.id == self._l3_balancer_id and
                not is_spec_complete(event.pb))

    def _start(self, ctx):
        try:
            self._process(ctx)
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            ctx.log.exception(u'failed to process l3 balancer order on start: %s', e)
        self._cache.bind_on_specific_events(self._callback, (events.L3BalancerUpdate,))

    def _stop(self):
        self._cache.unbind_from_specific_events(self._callback, (events.L3BalancerUpdate,))

    def _process(self, ctx, event=None):
        """
        :type event: events.L3BalancerUpdate
        :type ctx: context.OpCtx
        """
        cached_pb = self._cache.must_get_l3_balancer(self._namespace_id, self._l3_balancer_id)
        if event and self._should_discard_event(ctx, event, cached_pb):
            return
        self._pb = cached_pb

        if not self._pb.HasField('order'):
            raise RuntimeError(u'L3 balancer has no "order" field')

        return self._runner.process(ctx, self._pb)

    def _process_event(self, ctx, event):
        self._process(ctx, event)

    def _process_empty_queue(self, ctx):
        self._process(ctx)
