import enum
import inject
import six
from boltons import strutils
from sepelib.core import config as appconfig

from awacs import yamlparser
from awacs.lib import l3mgrclient
from awacs.lib.order_processor.model import BaseStandaloneProcessor, WithOrder, Result
from awacs.lib.strutils import quote_join_sorted
from awacs.model import dao, zk, cache, util
from awacs.model.balancer.container_spec import configure_tunnels_for_ip_addresses
from awacs.model.balancer.stateholder import BalancerStateHolder
from awacs.model.balancer.vector import BalancerVersion
from awacs.model.balancer.generator import get_rev_index_pb
from awacs.model.l3_balancer import l3mgr, l3_balancer
from awacs.model.util import get_balancer_location, NANNY_ROBOT_LOGIN
from awacs.wrappers.base import Holder
from infra.awacs.proto import model_pb2, modules_pb2
from infra.swatlib.auth import abc


@enum.unique
class State(enum.Enum):
    STARTED = 1
    CREATING_SLB_PING_UPSTREAM = 2
    FINISHED = 4
    WAITING_FOR_ACTIVATION = 5
    ASSIGNING_FIREWALL_GRANTS = 6
    GETTING_ABC_SLUGS = 7
    CREATING_L3_MGR_SERVICE = 8
    UPDATING_L3_MGR_SERVICE_PERMISSIONS = 9
    ACQUIRING_IPV6_ADDRESS = 10
    ACQUIRING_IPV4_ADDRESS = 11
    CREATING_VIRTUAL_SERVERS = 12
    SAVING_VS_CONFIG = 13
    SAVING_SPEC = 14
    CANCELLING = 15
    CANCELLED = 16
    UPDATING_L7_CONTAINER_SPEC = 17


class L3BalancerOrder(WithOrder):
    __slots__ = ()

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = u'L3Balancer'
    states = State

    state_descriptions = {
        State.STARTED: u'Starting',
        State.CREATING_SLB_PING_UPSTREAM: u'Creating upstream for L3 health checks',
        State.GETTING_ABC_SLUGS: u'Getting ABC service info',
        State.CREATING_L3_MGR_SERVICE: u'Creating service in L3 Manager',
        State.UPDATING_L3_MGR_SERVICE_PERMISSIONS: u'Updating permissions in L3 Manager',
        State.ACQUIRING_IPV6_ADDRESS: u'Acquiring IPv6 address',
        State.ACQUIRING_IPV4_ADDRESS: u'Acquiring IPv4 address',
        State.CREATING_VIRTUAL_SERVERS: u'Creating virtual servers in L3 Manager',
        State.SAVING_VS_CONFIG: u'Configuring virtual servers',
        State.UPDATING_L7_CONTAINER_SPEC: u'Configuring tunnels in L7 balancers',
        State.WAITING_FOR_ACTIVATION: u'Waiting for L3 balancer activation',
        State.ASSIGNING_FIREWALL_GRANTS: u'Setting up firewall grants',
        State.SAVING_SPEC: u'Updating L3 balancer spec',
        State.FINISHED: u'Finishing',
        State.CANCELLING: u'Cancelling',
        State.CANCELLED: u'Cancelling',
    }

    def is_external(self):
        return self.pb.order.content.traffic_type == model_pb2.L3BalancerOrder.Content.EXTERNAL

    @property
    def config_management_mode(self):
        if self.pb.spec.config_management_mode:
            return self.pb.spec.config_management_mode
        if self.pb.order.content.config_management_mode:
            return self.pb.order.content.config_management_mode
        if self.pb.order.content.ctl_version >= 2:
            return model_pb2.L3BalancerSpec.MODE_REAL_AND_VIRTUAL_SERVERS
        return model_pb2.L3BalancerSpec.MODE_REAL_SERVERS_ONLY

    def is_fully_managed(self):
        return self.config_management_mode == model_pb2.L3BalancerSpec.MODE_REAL_AND_VIRTUAL_SERVERS

    @staticmethod
    def get_processors():
        return (
            Started,
            CreatingSlbPingUpstream,
            GettingAbcSlugs,
            CreatingL3mgrService,
            UpdatingL3mgrServicePermissions,
            AcquiringIPv6Address,
            AcquiringIPv4Address,
            CreatingVirtualServers,
            SavingVsConfig,
            UpdatingL7ContainerSpec,
            SavingSpec,
            WaitingForActivation,
            AssigningFirewallGrants,
            Cancelling,
        )

    def zk_update(self):
        return self.zk.update_l3_balancer(namespace_id=self.namespace_id, l3_balancer_id=self.id)

    def dao_update(self, comment):
        return self.dao.update_l3_balancer(
            namespace_id=self.namespace_id,
            l3_balancer_id=self.id,
            version=self.pb.meta.version,
            login=self.pb.meta.author,
            comment=comment,
            updated_spec_pb=self.pb.spec,
        )


class L3BalancerOrderProcessor(BaseStandaloneProcessor):
    state = None
    next_state = None
    cancelled_state = None

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _l3mgr_client = inject.attr(l3mgrclient.IL3MgrClient)  # type: l3mgrclient.L3MgrClient

    @classmethod
    def process(cls, ctx, order):
        """
        :type ctx: context.OpCtx
        :type order: L3BalancerOrder
        :rtype: Result
        """
        raise NotImplementedError


class Started(L3BalancerOrderProcessor):
    state = State.STARTED
    next_state = State.GETTING_ABC_SLUGS
    next_state_create_slbping = State.CREATING_SLB_PING_UPSTREAM
    cancelled_state = State.CANCELLING

    @classmethod
    def process(cls, ctx, order):
        announce_checks_configured = set()
        announce_checks_not_configured = set()
        ping_urls = set()
        balancer_pbs = cls._cache.list_all_balancers(order.namespace_id)
        for b_pb in balancer_pbs:
            config_pb = b_pb.spec.yandex_balancer.config
            if config_pb.HasField('l7_macro'):
                if config_pb.l7_macro.HasField('announce_check_reply'):
                    announce_checks_configured.add(b_pb.meta.id)
                    ping_urls.add(config_pb.l7_macro.announce_check_reply.url_re)
                else:
                    announce_checks_not_configured.add(b_pb.meta.id)

        if not announce_checks_configured:
            return order.make_result(cls.next_state_create_slbping)
        if announce_checks_not_configured:
            return order.make_result(
                description=u'"l7_macro.announce_check_reply" must be configured in L7 balancers: "{}"'.format(
                    quote_join_sorted(announce_checks_not_configured)),
                content_pb=model_pb2.L3BalancerOrder.OrderFeedback.AnnounceCheckMisconfigured(
                    balancer_ids=sorted(announce_checks_not_configured)
                ),
                next_state=cls.state
            )
        if len(ping_urls) == 1:
            order.context[u'ping_url'] = ping_urls.pop()
            return order.make_result(cls.next_state)

        return order.make_result(
            description=u'"l7_macro.announce_check_reply.url_re" is not identical in all L7 balancers, '
                        u'please configure them to be the same.',
            content_pb=model_pb2.L3BalancerOrder.OrderFeedback.AnnounceCheckMisconfigured(),
            next_state=cls.state
        )


class CreatingSlbPingUpstream(L3BalancerOrderProcessor):
    state = State.CREATING_SLB_PING_UPSTREAM
    next_state = State.GETTING_ABC_SLUGS
    cancelled_state = State.CANCELLING

    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    SLBPING_UPSTREAM_ID = u'slbping'
    SLBPING_UPSTREAM_YAML = u'''---
    regexp_section:
      matcher:
        match_fsm:
          url: '/ping'
      modules:
        - slb_ping_macro:
            errordoc: true'''
    SLBPING_UPSTREAM_HOLDER = yamlparser.parse(modules_pb2.Holder, SLBPING_UPSTREAM_YAML)

    @classmethod
    def _make_upstream_meta_pb(cls, order):
        meta_pb = model_pb2.UpstreamMeta(
            id=cls.SLBPING_UPSTREAM_ID,
            namespace_id=order.namespace_id
        )
        ns_pb = cls._cache.must_get_namespace(order.namespace_id)
        meta_pb.auth.type = meta_pb.auth.STAFF
        meta_pb.auth.staff.owners.logins.append(order.pb.meta.author)
        meta_pb.auth.staff.owners.logins.extend(ns_pb.meta.auth.staff.owners.logins)
        meta_pb.auth.staff.owners.group_ids.extend(ns_pb.meta.auth.staff.owners.group_ids)
        return meta_pb

    @classmethod
    def _make_upstream_spec_pb(cls):
        spec_pb = model_pb2.UpstreamSpec()
        spec_pb.yandex_balancer.yaml = cls.SLBPING_UPSTREAM_YAML
        spec_pb.yandex_balancer.config.CopyFrom(cls.SLBPING_UPSTREAM_HOLDER)
        spec_pb.labels[u'order'] = u'00001000'
        return spec_pb

    @classmethod
    def process(cls, ctx, order):
        meta_pb = cls._make_upstream_meta_pb(order)
        spec_pb = cls._make_upstream_spec_pb()
        rev_index_pb = get_rev_index_pb(meta_pb.namespace_id, spec_pb)

        cls._dao.create_upstream_if_missing(meta_pb=meta_pb, spec_pb=spec_pb, login=util.NANNY_ROBOT_LOGIN,
                                            rev_index_pb=rev_index_pb,
                                            compare_with_existing=False,  # trust existing "slbping"
                                            )
        return order.make_result(cls.next_state)


class GettingAbcSlugs(L3BalancerOrderProcessor):
    state = State.GETTING_ABC_SLUGS
    next_state = State.CREATING_L3_MGR_SERVICE
    cancelled_state = State.CANCELLING

    _abc_client = inject.attr(abc.IAbcClient)  # type: abc.AbcClient

    @classmethod
    def process(cls, ctx, order):
        if order.pb.order.content.abc_service_id:
            abc_slug = cls._abc_client.get_service_slug(order.pb.order.content.abc_service_id)
            ctx.log.info(u'L3 balancer has custom ABC service id "%s" with slug "%s"',
                         order.pb.order.content.abc_service_id, abc_slug)
            order.context[u'permissions_abc_slug'] = abc_slug
            order.context[u'owner_abc_slug'] = abc_slug
        else:
            ns_pb = cls._cache.must_get_namespace(order.namespace_id)
            namespace_abc_slug = cls._abc_client.get_service_slug(ns_pb.meta.abc_service_id)
            ctx.log.info(u'L3 balancer has default ABC service id "rclb". '
                         u'L3 permissions will be given to namespace abc_service_id "%s" with slug "%s"',
                         ns_pb.meta.abc_service_id, namespace_abc_slug)
            order.context[u'permissions_abc_slug'] = namespace_abc_slug
            order.context[u'owner_abc_slug'] = u'rclb'
        return order.make_result(cls.next_state)


class CreatingL3mgrService(L3BalancerOrderProcessor):
    state = State.CREATING_L3_MGR_SERVICE
    next_state = State.UPDATING_L3_MGR_SERVICE_PERMISSIONS
    next_state_skip_permissions = State.ACQUIRING_IPV6_ADDRESS
    cancelled_state = None

    @classmethod
    def _get_existing_svc_id(cls, order):
        """
        API already checks that service with the same FQDN doesn't exist when order is created.
        This checks the case when we crashed right after creating the service, without saving it to our zk.
        """
        for service in cls._l3mgr_client.list_services_by_fqdn(fqdn=order.pb.order.content.fqdn,
                                                               full=True,
                                                               request_timeout=3):
            if service[u'fqdn'] == order.pb.order.content.fqdn:
                return service[u'id']
        else:
            return None

    @classmethod
    def process(cls, ctx, order):
        existing_svc_id = cls._get_existing_svc_id(order)
        if existing_svc_id is not None:
            order.context[u'svc_id'] = existing_svc_id
            ctx.log.info(u'Found existing service in L3mgr: id="%s"', order.context[u'svc_id'])
            return order.make_result(cls.next_state)

        ctx.log.info(u'Creating service: fqdn="%s", abc_slug="%s"',
                     order.pb.order.content.fqdn, order.context[u'owner_abc_slug'])
        data = l3_balancer.make_l3mgr_balancer_meta(order.namespace_id, order.id,
                                                    include_meta_prefix=True,
                                                    locked=order.is_fully_managed())
        resp = cls._l3mgr_client.create_service(fqdn=order.pb.order.content.fqdn,
                                                abc_code=order.context[u'owner_abc_slug'],
                                                data=data)
        order.context[u'svc_id'] = resp[u'object'][u'id']
        ctx.log.info(u'Created service in L3mgr: id="%s" with owner %s',
                     order.context[u'svc_id'], order.context[u'owner_abc_slug'])
        if order.is_fully_managed():
            # don't give users any permissions
            return order.make_result(cls.next_state_skip_permissions)
        return order.make_result(cls.next_state)


class UpdatingL3mgrServicePermissions(L3BalancerOrderProcessor):
    state = State.UPDATING_L3_MGR_SERVICE_PERMISSIONS
    next_state = State.ACQUIRING_IPV6_ADDRESS
    cancelled_state = None

    USER_L3MGR_PERMISSIONS = [
        u'l3mgr.editrs_service',
        u'l3mgr.editvs_service',
        u'l3mgr.deploy_service',
    ]

    @classmethod
    def process(cls, ctx, order):
        svc_id = order.context[u'svc_id']
        permissions_to_add = set(cls.USER_L3MGR_PERMISSIONS)

        # check existing permissions
        for role in cls._l3mgr_client.list_roles(svc_id=svc_id)[u'objects']:
            if role[u'abc'] == order.context[u'permissions_abc_slug']:
                permissions_to_add.discard(role[u'permission'])

        for permission in permissions_to_add:
            ctx.log.info(u'Adding role: svc_id="%s", subject="%s", permission="%s"',
                         svc_id, order.context[u'permissions_abc_slug'], permission)
            cls._l3mgr_client.add_role(svc_id=svc_id,
                                       subject=order.context[u'permissions_abc_slug'],
                                       permission=permission)
        order.context[u'permissions_updated'] = True
        return order.make_result(cls.next_state)


class AcquiringIPv6Address(L3BalancerOrderProcessor):
    state = State.ACQUIRING_IPV6_ADDRESS
    next_state = State.CREATING_VIRTUAL_SERVERS
    next_state_ipv4 = State.ACQUIRING_IPV4_ADDRESS
    cancelled_state = None

    @classmethod
    def process(cls, ctx, order):
        ctx.log.info(u'Acquiring IPv6: abc_slug="%s", is_external="%s", fqdn="%s"',
                     order.context[u'owner_abc_slug'], order.is_external, order.pb.order.content.fqdn)
        resp = cls._l3mgr_client.get_new_ip(
            abc_code=order.context[u'owner_abc_slug'],
            v4=False,
            external=order.is_external(),
            fqdn=order.pb.order.content.fqdn)
        order.context[u'ipv6_addr'] = resp[u'object']

        if order.is_external():
            return order.make_result(cls.next_state_ipv4)
        else:
            return order.make_result(cls.next_state)


class AcquiringIPv4Address(L3BalancerOrderProcessor):
    state = State.ACQUIRING_IPV4_ADDRESS
    next_state = State.CREATING_VIRTUAL_SERVERS
    cancelled_state = None

    @classmethod
    def process(cls, ctx, order):
        ctx.log.info(u'Acquiring IPv4: abc_slug="%s", is_external="%s", fqdn="%s"',
                     order.context[u'owner_abc_slug'], order.is_external, order.pb.order.content.fqdn)
        resp = cls._l3mgr_client.get_new_ip(
            abc_code=order.context[u'owner_abc_slug'],
            v4=True,
            external=order.is_external(),
            fqdn=order.pb.order.content.fqdn)
        order.context[u'ipv4_addr'] = resp[u'object']
        return order.make_result(cls.next_state)


class CreatingVirtualServers(L3BalancerOrderProcessor):
    state = State.CREATING_VIRTUAL_SERVERS
    next_state = State.SAVING_VS_CONFIG
    cancelled_state = None

    @classmethod
    def _get_ports(cls, order):
        if order.pb.order.content.protocol == order.pb.order.content.HTTP:
            return [80]
        elif order.pb.order.content.protocol == order.pb.order.content.HTTPS:
            return [443]
        elif order.pb.order.content.protocol == order.pb.order.content.HTTP_AND_HTTPS:
            return [80, 443]
        else:
            raise AssertionError()

    @classmethod
    def process(cls, _, order):
        if order.is_external():
            traffic_type = model_pb2.L3BalancerSpec.VirtualServer.TT_EXTERNAL
        else:
            traffic_type = model_pb2.L3BalancerSpec.VirtualServer.TT_INTERNAL
        svc_id = order.context[u'svc_id']
        health_check_url = order.context.get(u'ping_url') or u'/ping'
        if u'vs_ids' not in order.context:
            order.context[u'vs_ids'] = []
        vs_ids = set(order.context[u'vs_ids'])
        created_vs = set((vs.ip, vs.port) for vs in order.pb.spec.virtual_servers)
        ports = cls._get_ports(order)
        for ip_field_name in (u'ipv6_addr', u'ipv4_addr'):
            ip = order.context.get(ip_field_name)
            if ip is None:
                continue
            for port in ports:
                if (ip, port) in created_vs:
                    continue
                vs_pb = model_pb2.L3BalancerSpec.VirtualServer(ip=ip, port=port, traffic_type=traffic_type)
                vs_pb.health_check_settings.url = health_check_url
                hc_pb = model_pb2.L3BalancerSpec.VirtualServer.HealthCheckSettings
                # keepalived uses SSL_GET without SNI and without checking cert contents, so this should be safe
                vs_pb.health_check_settings.check_type = hc_pb.CT_SSL_GET if port == 443 else hc_pb.CT_HTTP_GET
                vs = l3mgr.VirtualServer.from_vs_pb(svc_id, vs_pb, l3mgr.RSGroup())
                vs = vs.create_in_l3mgr(cls._l3mgr_client)
                vs_ids.add(vs.id)
                order.context[u'vs_ids'] = list(vs_ids)
                pb = order.pb.spec.virtual_servers.add()
                pb.CopyFrom(vs_pb)
                for pb in order.zk_update():
                    pb.CopyFrom(order.pb)
        return order.make_result(cls.next_state)


class SavingVsConfig(L3BalancerOrderProcessor):
    state = State.SAVING_VS_CONFIG
    next_state = State.UPDATING_L7_CONTAINER_SPEC
    next_state_skip_tunnels = State.SAVING_SPEC
    cancelled_state = None

    @classmethod
    def _save_config(cls, ctx, order):
        vs_ids = sorted(set(order.context[u'vs_ids']))
        ctx.log.info(u'Saving config: svc_id="%s", vs_ids="%s"', order.context[u'svc_id'], vs_ids)
        resp = cls._l3mgr_client.create_config_with_vs(svc_id=order.context[u'svc_id'],
                                                       vs_ids=vs_ids,
                                                       comment=u'Add virtual servers',
                                                       use_etag=False)
        return resp[u'object'][u'id']

    @classmethod
    def process(cls, ctx, order):
        if u'l3mgr_cfg_id' not in order.context:
            cfg_id = cls._save_config(ctx, order)
            order.context[u'l3mgr_cfg_id'] = cfg_id
        else:
            cfg_id = order.context[u'l3mgr_cfg_id']

        ctx.log.info(u'Processing config: svc_id="%s", cfg_id="%s"', order.context[u'svc_id'], cfg_id)
        cls._l3mgr_client.process_config(svc_id=order.context[u'svc_id'],
                                         cfg_id=cfg_id,
                                         use_etag=True,
                                         latest_cfg_id=cfg_id,
                                         force=True)
        order.context[u'l3mgr_cfg_processed'] = True

        if appconfig.get_value(u'run.disable_l7_tunnels_autoconfig_on_l3_order', default=False):
            return order.make_result(cls.next_state_skip_tunnels)
        ns_pb = cls._cache.must_get_namespace(order.namespace_id)
        if ns_pb.spec.easy_mode_settings.disable_l7_tunnels_autoconfig.value:
            return order.make_result(cls.next_state_skip_tunnels)

        l3_amount = len(cls._cache.list_all_l3_balancers(order.namespace_id))
        if l3_amount > 1:
            # when we need to support multiple L3s, we need to modify "configure_tunnels_for_ip_addresses",
            # since it removes all existing tunnels from L7 spec
            ctx.log.debug(u'There are more than 1 L3 balancer in this namespace (%s), '
                          u'for now skip configuring L7 tunnels', l3_amount)
            return order.make_result(cls.next_state_skip_tunnels)

        return order.make_result(cls.next_state)


class UpdatingL7ContainerSpec(L3BalancerOrderProcessor):
    state = State.UPDATING_L7_CONTAINER_SPEC
    next_state = State.SAVING_SPEC
    cancelled_state = None

    _dao = inject.attr(dao.IDao)  # type: dao.Dao
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage

    @classmethod
    def l7_is_in_progress(cls, l7_balancer_pb):
        balancer_state_pb = cls._cache.get_balancer_state(l7_balancer_pb.meta.namespace_id, l7_balancer_pb.meta.id)
        h = BalancerStateHolder(namespace_id=l7_balancer_pb.meta.namespace_id,
                                balancer_id=l7_balancer_pb.meta.id,
                                balancer_state_pb=balancer_state_pb)
        current_ver = BalancerVersion.from_pb(l7_balancer_pb)
        return h.balancer_latest_in_progress_version is not None or (h.balancer_active_version is None or h.balancer_active_version < current_ver)

    @classmethod
    def update_container_spec(cls, ctx, l3_balancer_pb, l7_balancer_pb, ip_addresses):
        updated = configure_tunnels_for_ip_addresses(cls._cache,
                                                     l7_balancer_pb=l7_balancer_pb,
                                                     description=l3_balancer_pb.meta.id,
                                                     ip_addresses=ip_addresses)
        if not updated:
            return False
        msg = u'Configured tunnels for L3 balancer "%s"'
        ctx.log.debug(msg + u' in L7 balancer "%s"', l3_balancer_pb.meta.id, l7_balancer_pb.meta.id)
        cls._dao.update_balancer(
            namespace_id=l7_balancer_pb.meta.namespace_id,
            balancer_id=l7_balancer_pb.meta.id,
            comment=msg % l3_balancer_pb.meta.id,
            login=NANNY_ROBOT_LOGIN,
            updated_spec_pb=l7_balancer_pb.spec,
            version=l7_balancer_pb.meta.version
        )
        return True

    @classmethod
    def get_l7_balancers_status(cls, ctx, namespace_id, l7_balancer_ids, processed_locations):
        ctx.log.debug(u'Already processed locations: "%s"', quote_join_sorted(processed_locations) or u'[]')
        balancer_to_process = None
        in_progress_locations = set()
        for l7_balancer_id in l7_balancer_ids:
            ctx.log.debug(u'Checking state and tunnels in L7 balancer "%s"', l7_balancer_id)
            balancer_pb = cls._zk.must_get_balancer(namespace_id, l7_balancer_id)
            location = get_balancer_location(balancer_pb, logger=ctx.log)
            if cls.l7_is_in_progress(balancer_pb):
                ctx.log.debug(u'L7 balancer "%s" is deploying, wait until it finishes', l7_balancer_id)
                in_progress_locations.add(location)
            elif not in_progress_locations and location not in processed_locations and balancer_to_process is None:
                ctx.log.debug(u'L7 balancer "%s" is idle, can configure its tunnels', l7_balancer_id)
                balancer_to_process = (location, balancer_pb)
        return in_progress_locations, balancer_to_process

    @classmethod
    def process(cls, ctx, order):
        processed_locations = set(order.context.get(u'processed_locations', []))
        in_progress_locations, balancer_to_process = cls.get_l7_balancers_status(
            ctx, order.namespace_id,
            (l7.id for l7 in order.pb.order.content.real_servers.balancers),
            processed_locations)

        if in_progress_locations:
            # even if we updated all specs, we need to wait until all L7 balancers have finished deploying
            desc = u'Configuring tunnels in L7 {} in {} "{}"'.format(
                strutils.cardinalize(u'balancer', len(in_progress_locations)),
                strutils.cardinalize(u'location', len(in_progress_locations)),
                quote_join_sorted(in_progress_locations))
            return Result(
                description=desc,
                content_pb=model_pb2.L3BalancerOrder.OrderFeedback.L7LocationsInProgress(
                    locations=sorted(in_progress_locations)
                ),
                next_state=cls.state
            )
        if not balancer_to_process:
            # all done
            return order.make_result(next_state=cls.next_state)

        location, l7_balancer_pb = balancer_to_process
        l3_ips = set(vs_pb.ip for vs_pb in order.pb.spec.virtual_servers)
        cls.update_container_spec(ctx, l3_balancer_pb=order.pb, l7_balancer_pb=l7_balancer_pb, ip_addresses=l3_ips)
        processed_locations.add(location)
        order.context[u'processed_locations'] = sorted(processed_locations)
        # ready to process next balancer (or wait until it's idle)
        return order.make_result(cls.state)


class SavingSpec(L3BalancerOrderProcessor):
    state = State.SAVING_SPEC
    next_state = State.WAITING_FOR_ACTIVATION
    cancelled_state = None

    @classmethod
    def process(cls, ctx, order):
        order.pb.spec.incomplete = False
        order.pb.spec.l3mgr_service_id = six.text_type(order.context[u'svc_id'])
        order.pb.spec.use_endpoint_weights = order.pb.order.content.use_endpoint_weights
        order.pb.spec.ctl_version = order.pb.order.content.ctl_version
        order.pb.spec.preserve_foreign_real_servers = order.pb.order.content.preserve_foreign_real_servers
        order.pb.spec.real_servers.CopyFrom(order.pb.order.content.real_servers)
        order.pb.spec.config_management_mode = order.config_management_mode
        if order.config_management_mode == model_pb2.L3BalancerSpec.MODE_REAL_SERVERS_ONLY:
            order.pb.spec.ClearField(b'virtual_servers')
        elif order.is_fully_managed():
            order.pb.spec.enforce_configs = True  # ignore future user edits in L3mgr
        order.dao_update(u'Finished order, spec.incomplete = False')
        return order.make_result(cls.next_state)


class WaitingForActivation(L3BalancerOrderProcessor):
    state = State.WAITING_FOR_ACTIVATION
    next_state = State.ASSIGNING_FIREWALL_GRANTS
    cancelled_state = None

    @classmethod
    def process(cls, ctx, order):
        ctx.log.info(u'Waiting for L3 balancer activation')
        l3_balancer_state_pb = cls._cache.get_l3_balancer_state(order.namespace_id, order.id)
        if l3_balancer_state_pb is not None:
            h = l3_balancer.L3BalancerStateHandler(l3_balancer_state_pb)
            if not h.generate_vectors().active.is_empty():
                return order.make_result(cls.next_state)
        ctx.log.info(u'Active vector is empty, continue waiting')
        return order.make_result(cls.state)


class AssigningFirewallGrants(L3BalancerOrderProcessor):
    state = State.ASSIGNING_FIREWALL_GRANTS
    next_state = State.FINISHED
    cancelled_state = None

    @classmethod
    def process(cls, ctx, order):
        ctx.log.info(u'Assigning firewall grants')
        subject = u'svc_{}'.format(order.context[u'permissions_abc_slug'])
        svc_id = order.context[u'svc_id']

        resp = cls._l3mgr_client.list_grants(svc_id=svc_id)
        existing_granted_subjects = resp[u'objects']
        if subject not in existing_granted_subjects:
            ctx.log.info(u'Setting grants: svc_id="%s", subject="%s"', svc_id, subject)
            cls._l3mgr_client.set_grants(svc_id=svc_id, subjects=[subject])
        return order.make_result(cls.next_state)


class Cancelling(L3BalancerOrderProcessor):
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    @classmethod
    def process(cls, ctx, order):
        order.pb.spec.incomplete = False
        order.dao_update(u'Cancelled order, spec.incomplete = False')
        return order.make_result(cls.next_state)
