import collections
import logging

import datetime
import inject
import itertools
import six
import time
from sepelib.core import config as appconfig
from six.moves import http_client as httplib

from awacs.lib import l3mgrclient
from awacs.model import zk, cache
from awacs.model.l3_balancer import cacheutil, events, errors, l3_balancer, state_handler, vector, l3mgr
from infra.awacs.proto import model_pb2
from infra.swatlib import orly_client


class Transport(object):
    DEFAULT_MAX_SKIP_COUNT = 3

    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _l3mgr_client = inject.attr(l3mgrclient.IL3MgrClient)  # type: l3mgrclient.L3MgrClient
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    _orly_brake = orly_client.OrlyBrake(
        rule=u'awacs-push-l3-config',
        metrics_registry=events.L3_CTL_REGISTRY)

    def __init__(self, namespace_id, l3_balancer_id):
        self._namespace_id = namespace_id
        self._l3_balancer_id = l3_balancer_id
        self._log = logging.getLogger(u'l3-transport("{}:{}")'.format(namespace_id, l3_balancer_id))

    def _get_config_or_none(self, ctx, service_id, config_id):
        try:
            return self._l3mgr_client.get_config(service_id, config_id)
        except l3mgrclient.L3MgrException as e:
            if e.resp is not None and e.resp.status_code == httplib.NOT_FOUND:
                return None
            else:
                ctx.log.exception(u'failed to call get_config')
                events.reporter.report(events.L3MgrEvent.ERROR, ctx=ctx)
                raise

    def _poll_configs_for_service(self, ctx, service_id, configs_from_old_to_new, config_ctimes, l3_balancer_state_pb):
        rv = False
        processed_versions = set()
        activated_versions = set()
        activated_config_full_ids = []

        configs_from_new_to_old = list(reversed(configs_from_old_to_new))
        for config_id, versions in configs_from_new_to_old:
            full_id_str = u'{}:{}'.format(service_id, config_id)
            ctx.log.debug(u'checking config "%s"...', full_id_str)

            resp = self._get_config_or_none(ctx, service_id, config_id)
            if resp is None:
                ctx.log.warn(u'config is missing from l3mgr: "%s"', full_id_str)
                continue

            if resp[u'state'] == u'ACTIVE':
                activated_config_full_ids.append((service_id, config_id))
                processed_versions.update(versions)
                activated_versions.update(versions)
                for older_config_id, older_config_versions in configs_from_new_to_old:
                    ctx.log.debug(u'considering config as active: "%s"', full_id_str)
                    processed_versions.update(older_config_versions)
                    activated_config_full_ids.append((service_id, older_config_id))
                break
            else:
                ctx.log.debug(u'continue checking older configs, config "%s" is not active...', full_id_str)

        if not activated_config_full_ids:
            try:
                resp = self._l3mgr_client.get_service(service_id)
            except l3mgrclient.L3MgrException:
                ctx.log.exception(u'failed to call get_service')
                events.reporter.report(events.L3MgrEvent.ERROR, ctx=ctx)
            else:
                active_config = resp.get(u'config')

                if active_config:
                    active_config_ctime = time.mktime(time.strptime(
                        active_config[u'timestamp'], '%Y-%m-%dT%H:%M:%S.%fZ'))

                    latest_config_id, latest_config_versions = configs_from_new_to_old[0]
                    latest_config_full_id = (service_id, latest_config_id)

                    if int(active_config_ctime * 1000000) > config_ctimes[latest_config_full_id]:
                        ctx.log.debug(u'current config is newer than latest known')
                        activated_config_full_ids.append(latest_config_full_id)
                        processed_versions.update(latest_config_versions)
                        activated_versions.update(latest_config_versions)
                        ctx.log.debug(u'considering config as active: "%s"', latest_config_id[:10])

                        for config_id, config_versions in configs_from_new_to_old:
                            processed_versions.update(config_versions)
                            activated_config_full_ids.append((service_id, config_id))
                        else:
                            ctx.log.debug(u'current config %s:%s is not active', service_id, active_config[u'id'])

        # TODO: too consecutive requests to zk can be replaced with just one
        updated = self._update_in_progress_statuses(
            config_full_ids_to_remove=activated_config_full_ids,
            versions_to_remove_configs_from=processed_versions,
            l3_balancer_state_pb=l3_balancer_state_pb)
        ctx.log.debug(u'updated in progress statuses: %s', updated)
        rv |= updated

        updated = self._update_active_statuses(
            activated_versions=activated_versions,
            l3_balancer_state_pb=l3_balancer_state_pb)
        ctx.log.debug(u'updated active statuses: %s', updated)
        rv |= updated
        return rv

    def poll_configs(self, ctx, l3_balancer_state_pb):
        """
        :type l3_balancer_state_pb: infra.awacs.proto.model_pb2.L3BalancerState
        :type ctx: context.OpCtx
        :rtype: bool
        """
        ctx = ctx.with_op(op_id=u'transport_poll_configs')

        versions_by_config_ids = collections.defaultdict(set)
        config_ctimes = self._get_in_progress_full_config_ids(l3_balancer_state_pb)

        for rev_status_pb in l3_balancer_state_pb.l3_balancer.l3_statuses:
            l3_balancer_version = vector.L3BalancerVersion.from_rev_status_pb(self._l3_balancer_id, rev_status_pb)
            if rev_status_pb.in_progress.status != u'True':
                continue
            for config_pb in rev_status_pb.in_progress.meta.l3mgr.configs:
                config_full_id = (config_pb.service_id, config_pb.config_id)
                versions_by_config_ids[config_full_id].add(l3_balancer_version)

        for backend_id, backend_state_pb in six.iteritems(l3_balancer_state_pb.backends):
            for rev_status_pb in backend_state_pb.l3_statuses:
                backend_version = vector.BackendVersion.from_rev_status_pb(backend_id, rev_status_pb)
                if rev_status_pb.in_progress.status != u'True':
                    continue
                for config_pb in rev_status_pb.in_progress.meta.l3mgr.configs:
                    config_full_id = (config_pb.service_id, config_pb.config_id)
                    versions_by_config_ids[config_full_id].add(backend_version)

        for endpoint_set_id, endpoint_set_state_pb in six.iteritems(l3_balancer_state_pb.endpoint_sets):
            for rev_status_pb in endpoint_set_state_pb.l3_statuses:
                endpoint_set_version = vector.EndpointSetVersion.from_rev_status_pb(endpoint_set_id, rev_status_pb)
                if rev_status_pb.in_progress.status != u'True':
                    continue
                for config_pb in rev_status_pb.in_progress.meta.l3mgr.configs:
                    config_full_id = (config_pb.service_id, config_pb.config_id)
                    versions_by_config_ids[config_full_id].add(endpoint_set_version)

        config_ids_from_old_to_new = list(versions_by_config_ids)
        config_ids_from_old_to_new.sort(key=config_ctimes.get)

        by_service_ids = collections.defaultdict(list)
        for config_full_id in config_ids_from_old_to_new:
            service_id, config_id = config_full_id
            versions = versions_by_config_ids[config_full_id]
            by_service_ids[service_id].append((config_id, versions))

        updated = False
        for service_id, service_configs_from_old_to_new in six.iteritems(by_service_ids):
            updated |= self._poll_configs_for_service(ctx=ctx,
                                                      service_id=service_id,
                                                      configs_from_old_to_new=service_configs_from_old_to_new,
                                                      config_ctimes=config_ctimes,
                                                      l3_balancer_state_pb=l3_balancer_state_pb)
        return updated

    @staticmethod
    def _iter_l3_rev_statuses(l3_balancer_state_pb):
        """
        :type l3_balancer_state_pb: infra.awacs.proto.model_pb2.L3BalancerState
        """
        revs = [l3_balancer_state_pb.l3_balancer.l3_statuses]
        revs.extend(pb.l3_statuses for pb in six.itervalues(l3_balancer_state_pb.backends))
        revs.extend(pb.l3_statuses for pb in six.itervalues(l3_balancer_state_pb.endpoint_sets))
        return itertools.chain.from_iterable(revs)

    @classmethod
    def _get_in_progress_full_config_ids(cls, l3_balancer_state_pb):
        """
        :type l3_balancer_state_pb: infra.awacs.proto.model_pb2.L3BalancerState
        :rtype: dict[(six.text_type, six.text_type), int]
        """
        rv = {}
        for rev_pb in cls._iter_l3_rev_statuses(l3_balancer_state_pb):
            if rev_pb.in_progress.status != u'True':
                continue
            meta_pb = rev_pb.in_progress.meta  # type: model_pb2.L3ConfigTransportMeta
            assert meta_pb.type == model_pb2.L3MGR
            for config_pb in meta_pb.l3mgr.configs:
                config_full_id = (config_pb.service_id, config_pb.config_id)
                if config_full_id in rv:
                    assert rv[config_full_id] == config_pb.ctime.ToMicroseconds()
                else:
                    rv[config_full_id] = config_pb.ctime.ToMicroseconds()
        return rv

    def _update_active_statuses(self, activated_versions, l3_balancer_state_pb):
        """
        :type activated_versions: set[L3BalancerVersion | BackendVersion | EndpointSetVersion]
        :type l3_balancer_state_pb: infra.awacs.proto.model_pb2.L3BalancerState
        :rtype: bool
        """
        updated = False
        for l3_balancer_state_pb in self._zk.update_l3_balancer_state(self._namespace_id, self._l3_balancer_id,
                                                                      l3_balancer_state_pb=l3_balancer_state_pb):
            h = state_handler.L3BalancerStateHandler(l3_balancer_state_pb)
            updated = False
            for version in activated_versions:
                updated |= h.set_active_rev(version)
            if not updated:
                break
            l3_balancer_state_pb.ClearField('skip_counts')
        return updated

    def _update_in_progress_statuses(self, config_full_ids_to_remove, versions_to_remove_configs_from,
                                     l3_balancer_state_pb):
        """
        :type config_full_ids_to_remove: list[tuple[(six.text_type, six.text_type)]]
        :type versions_to_remove_configs_from: set[L3BalancerVersion | BackendVersion | EndpointSetVersion]
        :type l3_balancer_state_pb: model_pb2.L3BalancerState
        :rtype: bool
        """

        def remove_activated_configs(cond_pb):
            """
            :type cond_pb: model_pb2.L3InProgressCondition
            """
            rv = False
            config_pbs = cond_pb.meta.l3mgr.configs
            updated_config_pbs = []
            for config_pb in config_pbs:
                full_config_id = (config_pb.service_id, config_pb.config_id)
                if full_config_id not in config_full_ids_to_remove:
                    updated_config_pbs.append(config_pb)
            del config_pbs[:]
            if len(config_pbs) != len(updated_config_pbs):
                rv |= True
            config_pbs.extend(updated_config_pbs)
            prev_status_pb = cond_pb.status
            cond_pb.status = u'True' if config_pbs else u'False'
            rv |= (prev_status_pb != cond_pb.status)
            return rv

        updated = False
        for l3_balancer_state_pb in self._zk.update_l3_balancer_state(self._namespace_id, self._l3_balancer_id,
                                                                      l3_balancer_state_pb=l3_balancer_state_pb):
            h = state_handler.L3BalancerStateHandler(l3_balancer_state_pb)
            updated = False
            for version in versions_to_remove_configs_from:
                updated |= h.select_rev(version).modify_in_progress(remove_activated_configs)
            if not updated:
                break
        return updated

    def _mark_vector_as_in_progress(self, vec, l3mgr_config_pb, l3_balancer_state_pb):
        """
        :type vec: L3Vector
        :type l3mgr_config_pb: model_pb2.L3mgrConfig
        :type l3_balancer_state_pb: infra.awacs.proto.model_pb2.L3BalancerState
        :rtype: bool
        """

        def add_config_and_set_in_progress(cond_pb):
            """
            :type cond_pb: model_pb2.L3InProgressCondition
            :rtype: bool
            """
            l3_config_pbs = cond_pb.meta.l3mgr.configs
            rv = False
            if l3mgr_config_pb not in l3_config_pbs:
                l3_config_pbs.add().CopyFrom(l3mgr_config_pb)
                rv = True
            if cond_pb.status != u'True':
                cond_pb.status = u'True'
                rv = True
            return rv

        updated = False
        for l3_balancer_state_pb in self._zk.update_l3_balancer_state(self._namespace_id, self._l3_balancer_id,
                                                                      l3_balancer_state_pb=l3_balancer_state_pb):
            h = state_handler.L3BalancerStateHandler(l3_balancer_state_pb)
            updated = False
            for version in vec:
                updated |= h.select_rev(version).modify_in_progress(add_config_and_set_in_progress)
            updated |= h.set_ignore_existing_l3mgr_config(value=False, author=u'awacs', comment=u'')
            if not updated:
                break
        return updated

    def skip_stuck(self, ctx, l3_balancer_state_pb):
        """
        :type l3_balancer_state_pb: infra.awacs.proto.model_pb2.L3BalancerState
        :type ctx: context.OpCtx
        :rtype: bool
        """
        ctx = ctx.with_op(op_id=u'transport_skip_stuck')

        _, _, in_progress_vector, _ = cacheutil.l3_balancer_state_to_vectors(l3_balancer_state_pb)

        if in_progress_vector.is_empty():
            return False
        in_progress_vector_hash = in_progress_vector.get_weak_hash_str()

        ctx.log.debug(u'in_progress_vector: %s', in_progress_vector)

        max_skip_count = appconfig.get_value(u'run.max_l3_config_skip_count', default=self.DEFAULT_MAX_SKIP_COUNT)
        skip_count = l3_balancer_state_pb.skip_counts.get(in_progress_vector_hash, 0)
        if skip_count > max_skip_count:
            ctx.log.debug(u'skip count for vector %s is %s and over allowed limit of %s, NOT skipping',
                          in_progress_vector_hash, skip_count, max_skip_count)
            events.reporter.report(events.L3MgrEvent.CONFIG_SKIP_IS_NOT_ALLOWED, ctx=ctx)
            return False

        in_progress_config_ids = self._get_in_progress_full_config_ids(l3_balancer_state_pb)
        if len(in_progress_config_ids) != 1:
            ctx.log.warn(u'len(in_progress_config_ids) != 1 in skip_stuck(): %s', in_progress_config_ids)
            return False

        (l3mgr_service_id, l3mgr_config_id), _ = in_progress_config_ids.popitem()

        resp = self._get_config_or_none(ctx, l3mgr_service_id, l3mgr_config_id)

        if resp is None:
            ctx.log.debug(u'%s:%s is missing from l3mgr, skipping', l3mgr_service_id, l3mgr_service_id)
            skip_reason = u'NOT_FOUND'
        elif resp[u'state'] in (u'TEST_FAIL', u'VCS_FAIL'):
            ctx.log.debug(u'%s:%s state is %s, skipping', l3mgr_service_id, l3mgr_config_id, resp[u'state'])
            skip_reason = resp[u'state']
        else:
            ctx.log.debug(u'%s:%s state is %s, continue waiting', l3mgr_service_id, l3mgr_config_id, resp[u'state'])
            return False

        _, updated = cacheutil.skip_in_progress_l3mgr_config(
            namespace_id=self._namespace_id,
            l3_balancer_id=self._l3_balancer_id,
            l3mgr_service_id_to_skip=l3mgr_service_id,
            l3mgr_config_id_to_skip=l3mgr_config_id,
            l3_balancer_state_pb=l3_balancer_state_pb,
            skipped_vector_hash=in_progress_vector_hash
        )
        ctx.log.debug(u'skipped %s:%s (skip_count: %s)', l3mgr_service_id, l3mgr_config_id, skip_count + 1)

        events.reporter.report(events.L3MgrEvent.CONFIG_SKIPPED, ctx=ctx)
        if skip_reason == u'TEST_FAIL':
            events.reporter.report(events.L3MgrEvent.CONFIG_SKIPPED_DUE_TO_TEST_FAIL, ctx=ctx)
        elif skip_reason == u'VCS_FAIL':
            events.reporter.report(events.L3MgrEvent.CONFIG_SKIPPED_DUE_TO_VCS_FAIL, ctx=ctx)
        elif skip_reason == u'NOT_FOUND':
            events.reporter.report(events.L3MgrEvent.CONFIG_SKIPPED_DUE_TO_NOT_FOUND, ctx=ctx)

        return updated

    def _make_rs_group(self, vec, l3_balancer_spec_pb):
        rs_group = l3mgr.RSGroup()
        included_backend_ids = cacheutil.get_included_backend_ids(l3_balancer_spec_pb)
        for es_id, endpoint_set_version in six.iteritems(vec.endpoint_set_versions):
            if endpoint_set_version.deleted:
                continue
            if es_id not in vec.backend_versions:
                continue
            if vec.backend_versions[es_id].deleted:
                continue
            if es_id not in included_backend_ids:
                continue
            es_spec_pb = cacheutil.find_endpoint_set_revision_spec_and_use_cache(
                namespace_id=self._namespace_id,
                endpoint_set_id=es_id,
                version=endpoint_set_version.version)
            for instance_pb in es_spec_pb.instances:  # type: model_pb2.EndpointSetSpec.Instance
                fqdn = instance_pb.host
                ip = instance_pb.ipv6_addr or instance_pb.ipv4_addr
                weight = l3_balancer.get_instance_weight(l3_balancer_spec_pb, instance_pb)
                rs_group.add(fqdn, ip, weight)
        return rs_group

    @staticmethod
    def _get_l3mgr_rs_group(ctx, virtual_servers):
        try:
            return l3mgr.RSGroup.from_l3mgr_virtual_servers(virtual_servers)
        except errors.L3BalancerTransportError:
            events.reporter.report(events.L3MgrEvent.ERROR, ctx=ctx)
            events.reporter.report(events.L3MgrEvent.ERROR_VS_NOT_SAME, ctx=ctx)
            raise

    def _augment_rs_group_with_foreign_rs(self, ctx, vectors, rs_group, l3mgr_rs_group):
        """
        To get foreign real servers:
         1) look at the active vector to get real servers that awacs pushed to L3mgr last time
         2) remove those RS from the real servers that are currently present in L3mgr
        This should leave only foreign RS, which we then add to the valid RS group

        :type ctx: context.Ctx
        :param vectors
        :type rs_group: l3mgr.RSGroup
        :type l3mgr_rs_group: l3mgr.RSGroup
        """
        # if awacs successfully pushed something to L3mgr even once, take it as ground truth
        if vectors.active.balancer_version is not None:
            vec = vectors.active
        else:  # otherwise, consider currently valid vector as the ground truth
            vec = vectors.valid

        l3_balancer_spec_pb = cacheutil.find_l3_balancer_revision_spec_and_use_cache(
            self._namespace_id, self._l3_balancer_id, vec.balancer_version.version)

        foreign_rs = set()
        known_rs_group = self._make_rs_group(vec, l3_balancer_spec_pb)
        known_real_servers = {rs.fqdn: rs for rs in known_rs_group.real_servers}
        for l3mgr_rs in l3mgr_rs_group.real_servers:
            if l3mgr_rs.fqdn not in known_real_servers:  # foreign RS, add it to the group
                rs_group.add_rs(l3mgr_rs)
                foreign_rs.add(l3mgr_rs)
        ctx.log.debug(u'Considering real servers from L3mgr as foreign and preserving them: %s',
                      sorted(six.text_type(rs) for rs in foreign_rs))
        return rs_group

    def transport(self, ctx, l3_balancer_state_pb):
        """
        :type l3_balancer_state_pb: infra.awacs.proto.model_pb2.L3BalancerState
        :type ctx: context.OpCtx
        :rtype: bool
        """
        ctx = ctx.with_op(op_id=u'transport')

        l3_vectors = cacheutil.l3_balancer_state_to_vectors(l3_balancer_state_pb)
        if l3_vectors.valid == l3_vectors.active:
            return
        if l3_vectors.valid == l3_vectors.in_progress:
            return

        ctx.log.debug(u'transport(), l3_balancer_state_pb.generation is %s', l3_balancer_state_pb.generation)
        ctx.log.debug(u'processing changes in l3 balancer, backend and endpoint set statuses')
        ctx.log.debug(u'valid vector: %s', l3_vectors.valid)
        ctx.log.debug(u'in-progress vector: %s', l3_vectors.in_progress)
        ctx.log.debug(u'active vector: %s', l3_vectors.active)

        if appconfig.get_value(u'run.disable_l3_transport', default=False):
            ctx.log.warn(u'l3 transport is disabled')
            return

        l3_balancer_pb = self._cache.get_l3_balancer(self._namespace_id, self._l3_balancer_id)
        if l3_balancer_pb.meta.transport_paused.value:
            ctx.log.debug(u'l3 transport is paused')
            return

        in_progress_config_ids = self._get_in_progress_full_config_ids(l3_balancer_state_pb)

        if len(in_progress_config_ids) > 1:
            ctx.log.warn(u'len(in_progress_config_ids) > 1 in transport(): %s', in_progress_config_ids)

        if in_progress_config_ids:
            # We don't want to rely on l3mgr to "linearize" configurations history.
            # Thus we create a new configuration only when the previous one is active.
            # See SWAT-5029 for the description of the case when this is crucial.
            ctx.log.debug(u'not proceeding: waiting for the activation of in-progress configurations: %s',
                          in_progress_config_ids)
            return

        valid_l3_balancer_spec_pb = cacheutil.find_l3_balancer_revision_spec_and_use_cache(
            self._namespace_id, self._l3_balancer_id, l3_vectors.valid.balancer_version.version)
        l3mgr_service_id = valid_l3_balancer_spec_pb.l3mgr_service_id
        with events.reporter.report_error(ctx, u'failed to get service from L3mgr'):
            l3mgr_service = l3mgr.Service.from_api(self._l3mgr_client, l3mgr_service_id)

        # take VS from active config. this can potentially lead to duplicate new configs if transport crashes after
        # creating config, but before activating it. on the other hand, it avoids a situation
        # where someone else edits L3mgr config without activating it (some untested draft changes, perhaps).
        # ctl_v1 doesn't claim full ownership of the balancer in L3mgr, so we don't want to activate or
        # edit than inactive config.
        if l3mgr_service.config_id is not None:
            with events.reporter.report_error(ctx, u'failed to get active service config from L3mgr'):
                active_config = l3mgr.ServiceConfig.from_api(self._l3mgr_client, l3mgr_service_id,
                                                             l3mgr_service.config_id)
        else:
            active_config = None

        virtual_servers = l3mgr_service.virtual_servers
        if not virtual_servers:
            ctx.log.error(u'l3mgr service %s has no virtual servers yet', l3mgr_service_id)
            return

        rs_group = self._make_rs_group(l3_vectors.valid, valid_l3_balancer_spec_pb)
        if active_config is not None:
            force_process = valid_l3_balancer_spec_pb.skip_tests.value
            l3mgr_config_id = active_config.id
            l3mgr_config_ctime = six.text_type(active_config.timestamp)
            if valid_l3_balancer_spec_pb.preserve_foreign_real_servers:
                l3mgr_rs_group = self._get_l3mgr_rs_group(ctx, virtual_servers)
                rs_group = self._augment_rs_group_with_foreign_rs(ctx, l3_vectors, rs_group, l3mgr_rs_group)
                need_to_update_rs = l3mgr_rs_group.need_to_update_ip_addresses or not l3mgr_rs_group.matches(rs_group)
            else:
                try:
                    l3mgr_rs_group = l3mgr.RSGroup.from_l3mgr_virtual_servers(virtual_servers)
                except errors.L3BalancerTransportError:
                    # couldn't parse current real servers, but it's fine, we'll replace them anyway,
                    # because "preserve_foreign_real_servers=False" in this branch.
                    # but we should still check that RS groups are the same in all virtual servers
                    need_to_update_rs = True
                    self._validate_raw_rs_groups(virtual_servers)
                else:
                    # successfully parsed current real servers, so we can reliably check if we need to update them
                    need_to_update_rs = (l3mgr_rs_group.need_to_update_ip_addresses
                                         or not l3mgr_rs_group.matches(rs_group))
        else:
            force_process = True
            need_to_update_rs = True
            l3mgr_config_id = None
            l3mgr_config_ctime = None

        need_to_update_rs = need_to_update_rs or l3_balancer_state_pb.ignore_existing_l3mgr_config.value
        ctx.log.debug(u'need to update: %s', need_to_update_rs)

        if need_to_update_rs:
            orly_op_id = l3_vectors.valid.get_weak_hash_str()
            ctx.log.debug(u'orly operation id: %s', orly_op_id)
            self._orly_brake.maybe_apply(op_id=orly_op_id, op_log=ctx.log, op_labels=[
                (u'namespace-id', self._namespace_id),
                (u'l3-balancer-id', self._l3_balancer_id),
            ])

            try:
                resp = self._l3mgr_client.create_config_with_rs(
                    svc_id=l3mgr_service_id,
                    groups=sorted(six.text_type(rs) for rs in rs_group.real_servers),
                    use_etag=False)
            except l3mgrclient.L3MgrException as e:
                ctx.log.exception(u'failed to call create_config_with_rs: %s', e)
                events.reporter.report(events.L3MgrEvent.ERROR, ctx=ctx)
                raise

            assert resp[u'result'] == u'OK'
            l3mgr_config_id = int(resp[u'object'][u'id'])
            ctx.log.debug(u'created l3mgr config: %s:%s', l3mgr_service_id, l3mgr_config_id)
            events.reporter.report(events.L3MgrEvent.CONFIG_CREATED, ctx=ctx)
            try:
                resp = self._l3mgr_client.process_config(l3mgr_service_id, l3mgr_config_id,
                                                         use_etag=True,
                                                         latest_cfg_id=l3mgr_config_id,
                                                         force=force_process)
                assert resp[u'result'] == u'OK'
            except l3mgrclient.L3MgrException as e:
                ctx.log.exception(u'failed to call process_config(%s, %s): %s',
                                  l3mgr_service_id, l3mgr_config_id, e)
                events.reporter.report(events.L3MgrEvent.ERROR, ctx=ctx)
                raise

            try:
                resp = self._l3mgr_client.get_config(l3mgr_service_id, l3mgr_config_id)
            except l3mgrclient.L3MgrException as e:
                ctx.log.exception(u'failed to call get_config: %s', e)
                events.reporter.report(events.L3MgrEvent.ERROR, ctx=ctx)
                raise
            l3mgr_config_ctime = resp[u'timestamp']

            ctx.log.debug(u'l3mgr config scheduled for processing: %s:%s', l3mgr_service_id, l3mgr_config_id)

        l3mgr_config_pb = model_pb2.L3mgrConfig(service_id=six.text_type(l3mgr_service_id),
                                                config_id=six.text_type(l3mgr_config_id))
        l3mgr_config_ctime_dt = datetime.datetime.strptime(l3mgr_config_ctime, u'%Y-%m-%dT%H:%M:%S.%fZ')
        l3mgr_config_pb.ctime.FromDatetime(l3mgr_config_ctime_dt)

        updated = self._mark_vector_as_in_progress(l3_vectors.valid, l3mgr_config_pb, l3_balancer_state_pb)
        return updated

    @staticmethod
    def _validate_raw_rs_groups(virtual_servers):
        first_raw_rs_group = None
        first_vs_id = None
        for vs in virtual_servers:
            raw_rs_group = vs[u'group']
            if first_raw_rs_group is None:
                first_raw_rs_group = raw_rs_group
                first_vs_id = vs[u'id']
            elif first_raw_rs_group != raw_rs_group:
                raise errors.RSGroupsConflict(
                    u'RS groups are not the same: {{vs[{}]: {}, vs[{}]: {}}}'.format(
                        first_vs_id, first_raw_rs_group, vs[u'id'], raw_rs_group))
