# coding: utf-8
import inject
from six.moves import http_client as httplib

from awacs.lib import l7heavy_client, itsclient
from awacs.lib.models.controllers import ModelCtl, Subscription
from awacs.lib.order_processor.model import has_actionable_spec, needs_removal
from awacs.model import objects, l7heavy_config, util
from awacs.model.cache import IAwacsCache


class L7HeavyConfigCtl(ModelCtl):
    model = objects.L7HeavyConfig
    subscriptions = {
        objects.L7HeavyConfig: Subscription(),
        objects.WeightSection: Subscription(watch_removals=True, check_uid=False)
    }
    _l7heavy_client = inject.attr(l7heavy_client.IL7HeavyClient)  # type: l7heavy_client.L7HeavyClient

    def should_process(self, full_uid, model, pb):
        if model == objects.L7HeavyConfig:
            return pb and has_actionable_spec(pb)
        return True

    def process(self, ctx):
        """
        :rtype: bool
        """
        state_pb = objects.L7HeavyConfig.state.zk.must_get(*self.full_uid)
        state_handler = l7heavy_config.L7HeavyConfigStateHandler(state_pb)

        l7heavy_config_pb = objects.L7HeavyConfig.cache.must_get(*self.full_uid)
        if needs_removal(l7heavy_config_pb):
            return self._self_delete(ctx, l7heavy_config_pb)

        vectors = l7heavy_config.L7HeavyConfigDiscoverer.discover(ctx, state_handler)
        if state_handler.was_updated:
            ctx.log.debug(u'state was updated after discovery')
            return
        l7heavy_config.L7HeavyConfigValidator.validate(ctx, state_handler)
        if state_handler.was_updated:
            ctx.log.debug(u'state was updated after validation')
            return
        l7heavy_config.L7HeavyConfigTransport.transport(ctx, state_handler, vectors)
        if state_handler.was_updated:
            ctx.log.debug(u'state was updated after transport')
            return
        l7heavy_config.L7HeavyConfigDiscoverer.cleanup_state(ctx, state_handler)
        if state_handler.was_updated:
            ctx.log.debug(u'state was updated after cleanup')

    @classmethod
    def _self_delete(cls, ctx, l7heavy_config_pb):
        """
        :type ctx: context.OpCtx
        :type l7heavy_config_pb: model_pb2.L7HeavyConfig
        """
        ctx.log.info(u'started self deletion')
        if l7heavy_config_pb.spec.state == l7heavy_config_pb.spec.REMOVED_FROM_AWACS:
            version, config = cls._l7heavy_client.get_config(l7heavy_config_pb.spec.l7_heavy_config_id)
            if config.get('metadata', {}).get('owner') == util.AWACS_L7HEAVY_OWNER_NAME:
                ctx.log.info(u'unmarking config as awacs-managed')
                config.pop('metadata')
                cls._l7heavy_client.update_config(l7heavy_config_pb.spec.l7_heavy_config_id, version, config)

            ctx.log.info(u'removing config from awacs')
            objects.L7HeavyConfig.remove(l7heavy_config_pb.meta.namespace_id, l7heavy_config_pb.meta.id)
        elif l7heavy_config_pb.spec.state == l7heavy_config_pb.spec.REMOVED_FROM_AWACS_AND_L7HEAVY:
            try:
                version, config = cls._l7heavy_client.get_config(l7heavy_config_pb.spec.l7_heavy_config_id)
            except l7heavy_client.L7HeavyError as e:
                if e.resp is not None and e.resp.status_code == httplib.NOT_FOUND:
                    config = None
                else:
                    raise
            if config is not None:
                if config['group_id'] == util.COMMON_L7HEAVY_GROUP_ID:
                    config['group_id'] = 'removing'
                    version, config = cls._l7heavy_client.update_config(l7heavy_config_pb.spec.l7_heavy_config_id, version, config)
                version, _ = cls._l7heavy_client.save_sections(l7heavy_config_pb.spec.l7_heavy_config_id, version, [])
                cls._l7heavy_client.remove_config(l7heavy_config_pb.spec.l7_heavy_config_id)

            c = IAwacsCache.instance()
            namespace_pb = c.must_get_namespace(l7heavy_config_pb.meta.namespace_id)
            normalised_prj = c.normalise_prj_tag(namespace_pb.spec.balancer_constraints.instance_tags.prj)
            its_value_path = util.ITS_VALUE_PATH_TEMPLATE.format(normalised_prj)

            its_client = itsclient.IItsClient.instance()
            try:
                version, _ = its_client.get_ruchka_value(its_value_path)
            except itsclient.ItsError as e:
                if e.resp is None or e.resp.status_code != httplib.NOT_FOUND:
                    raise
            else:
                its_client.remove_ruchka_value(its_value_path, version)
            objects.L7HeavyConfig.remove(l7heavy_config_pb.meta.namespace_id, l7heavy_config_pb.meta.id)


