import enum
import inject
import six
from abc import ABCMeta

import six

from awacs.lib.order_processor.model import BaseProcessor, WithOrder
from awacs.lib import l7heavy_client
from awacs.model import dao, zk, cache, util, objects
from awacs.model.l7heavy_config.consts import MARTY_STAFF_GROUP_ID
from awacs.wrappers.l7upstreammacro import L7UpstreamMacroDc
from infra.awacs.proto import model_pb2


class State(enum.Enum):
    STARTED = 1
    CREATING_ITS_KNOB = 5
    CREATING_L7HEAVY_DASHBOARD = 10
    PUSH_EMPTY_CONFIG_TO_ITS = 15
    SAVING_SPEC = 20
    CREATING_WEIGHT_SECTIONS = 30

    CANCELLING = 40
    CANCELLED = 41
    FINISHED = 50

    MARKING_CONFIG_AS_MANAGED_BY_AWACS = 60
    EXTRACTING_WEIGHT_SECTIONS_FROM_L7HEAVY = 70



class L7HeavyConfigOrder(WithOrder):
    __slots__ = ()

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = u'L7HeavyConfig'
    states = State

    def zk_update(self):
        return objects.L7HeavyConfig.zk.update(self.namespace_id, self.id)

    def dao_update(self, comment):
        return objects.L7HeavyConfig.update(self.namespace_id, self.id, self.pb.meta.version, comment,
                                            self.pb.meta.author, updated_spec_pb=self.pb.spec)

    @staticmethod
    def get_processors():
        return (
            Started,
            CreatingItsKnob,
            CreatingL7HeavyDashboard,
            PushEmptyConfigToIts,
            SavingSpec,
            CreatingWeightSections,
            Cancelling,
            MarkingConfigAsManagedByAwacs,
            ExtractingWeightSectionsFromL7Heavy,
        )


class L7HeavyConfigOrderProcessor(six.with_metaclass(ABCMeta, BaseProcessor)):
    __slots__ = (u'order', u'order_content')

    _l7heavy_client = inject.attr(l7heavy_client.IL7HeavyClient)  # type: l7heavy_client.L7HeavyClient

    force_cancelled_state = State.CANCELLING

    def __init__(self, entity):
        super(L7HeavyConfigOrderProcessor, self).__init__(entity)
        self.entity = self.entity  # type: L7HeavyConfigOrder  # noqa
        self.order_content = self.entity.pb.order.content  # type: model_pb2.L7HeavyConfigOrder.Content


class Started(L7HeavyConfigOrderProcessor):
    __slots__ = ()
    state = State.STARTED
    next_state = State.CREATING_ITS_KNOB
    next_state_import = State.MARKING_CONFIG_AS_MANAGED_BY_AWACS
    cancelled_state = State.CANCELLING

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    def process(self, ctx):
        if self.order_content.mode == self.order_content.CREATE:
            return self.next_state
        elif self.order_content.mode == self.order_content.IMPORT:
            objects.L7HeavyConfig.update(self.entity.namespace_id, self.entity.id, self.entity.pb.meta.version,
                                         'Put on pause by order controller', self.entity.pb.meta.author,
                                         updated_transport_paused_pb=model_pb2.PausedCondition(value=True))
            return self.next_state_import
        else:
            raise RuntimeError('Unknown order mode')


class CreatingItsKnob(L7HeavyConfigOrderProcessor):
    __slots__ = ()
    state = State.CREATING_ITS_KNOB
    next_state = State.CREATING_L7HEAVY_DASHBOARD
    cancelled_state = State.CANCELLING

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    def process(self, ctx):
        namespace_pb = self._zk.must_get_namespace(self.entity.namespace_id)
        if any(util.WEIGHTS_KNOB_ID == knob_pb.its_ruchka_id for knob_pb in namespace_pb.spec.its.knobs.common_knobs):
            return self.next_state
        namespace_pb.spec.its.knobs.common_knobs.add(its_ruchka_id=util.WEIGHTS_KNOB_ID)
        self._dao.update_namespace(
            namespace_id=self.entity.namespace_id,
            version=namespace_pb.meta.version,
            comment='Add ITS handler by L7Heavy config order',
            login=self.entity.pb.meta.author,
            updated_spec_pb=namespace_pb.spec
        )
        return self.next_state


class CreatingL7HeavyDashboard(L7HeavyConfigOrderProcessor):
    __slots__ = ()
    state = State.CREATING_L7HEAVY_DASHBOARD
    next_state = State.PUSH_EMPTY_CONFIG_TO_ITS
    cancelled_state = State.CANCELLING

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        namespace_pb = self._cache.must_get_namespace(self.entity.namespace_id)
        normalised_prj = self._cache.normalise_prj_tag(namespace_pb.spec.balancer_constraints.instance_tags.prj)
        config = self._l7heavy_client.create_config(
            config_id=namespace_pb.meta.id,
            group_id=self.order_content.group_id if self.order_content.group_id != util.COMMON_L7HEAVY_GROUP_ID else 'common-order',
            its_value_path=util.ITS_VALUE_PATH_TEMPLATE.format(normalised_prj),
            logins=list(namespace_pb.spec.its.acl.logins),
            groups=list(namespace_pb.spec.its.acl.staff_group_ids) + [MARTY_STAFF_GROUP_ID],
            link=util.get_namespace_link(self.entity.namespace_id)
        )
        self.entity.context['version'] = config['version']
        return self.next_state


class PushEmptyConfigToIts(L7HeavyConfigOrderProcessor):
    __slots__ = ()
    state = State.PUSH_EMPTY_CONFIG_TO_ITS
    next_state = State.CREATING_WEIGHT_SECTIONS
    cancelled_state = State.CANCELLING

    def process(self, ctx):
        self._l7heavy_client.push_weights_to_its(self.entity.namespace_id, self.entity.context['version'])
        return self.next_state


class MarkingConfigAsManagedByAwacs(L7HeavyConfigOrderProcessor):
    __slots__ = ()
    state = State.MARKING_CONFIG_AS_MANAGED_BY_AWACS
    next_state = State.EXTRACTING_WEIGHT_SECTIONS_FROM_L7HEAVY
    cancelled_state = State.CANCELLING

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    def process(self, ctx):
        version, config = self._l7heavy_client.get_config(self.order_content.l7_heavy_config_id)
        if config.get('metadata', {}).get('owner') == util.AWACS_L7HEAVY_OWNER_NAME:
            return self.next_state
        config['metadata'] = {
            'owner': util.AWACS_L7HEAVY_OWNER_NAME,
            'link': util.get_namespace_link(self.entity.namespace_id)
        }
        self._l7heavy_client.update_config(self.order_content.l7_heavy_config_id, version, config)
        self.entity.context['group_id'] = config['group_id']
        return self.next_state


class ExtractingWeightSectionsFromL7Heavy(L7HeavyConfigOrderProcessor):
    __slots__ = ()
    state = State.EXTRACTING_WEIGHT_SECTIONS_FROM_L7HEAVY
    next_state = State.SAVING_SPEC
    cancelled_state = State.CANCELLING

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _dao = inject.attr(dao.IDao)  # type: dao.Dao

    def process(self, ctx):
        version, sections = self._l7heavy_client.get_config_sections(self.order_content.l7_heavy_config_id)
        for section in sections:
            if section.get('skip_normalization'):
                raise RuntimeError('"skip_normalization" flag does not supported')
            ws_pb = model_pb2.WeightSection()
            ws_pb.meta.namespace_id = self.entity.namespace_id
            ws_pb.meta.id = section['id']
            if all(loc['id'].lower() in L7UpstreamMacroDc.ALLOWED_DC_NAMES for loc in section['locations']):
                ws_pb.meta.type = ws_pb.meta.ST_DC_WEIGHTS
            else:
                ws_pb.meta.type = ws_pb.meta.ST_TRAFFIC_SPLIT
            ws_pb.meta.comment = 'Imported by L7Heavy order'
            if section.get('exclude_from_bulk_actions'):
                ws_pb.spec.exclude_from_bulk_actions.value = True
            for loc in sorted(section['locations'], key=lambda x: x['id']):
                ws_pb.spec.locations.add(name=loc['id'], default_weight=loc['default_weight'])
            for loc in sorted(section.get('fallback_locations', ()), key=lambda x: x['id']):
                ws_pb.spec.locations.add(name=loc['id'], default_weight=loc['default_weight'], is_fallback=True)
            existing_ws_pb = objects.WeightSection.zk.get(self.entity.namespace_id, ws_pb.meta.id)
            if existing_ws_pb is not None:
                sorted_locations = sorted(existing_ws_pb.spec.locations, key=lambda x: (x.is_fallback, x.name))
                existing_ws_pb.spec.ClearField('locations')
                existing_ws_pb.spec.locations.extend(sorted_locations)
                if existing_ws_pb.spec != ws_pb.spec:
                    raise RuntimeError('Section "{}" already exists and has incompatible settings. '
                                       'Please remove section from awacs.'.format(ws_pb.meta.id))
            else:
                objects.WeightSection.create(ws_pb.meta, ws_pb.spec, self.entity.pb.meta.author)
        return self.next_state


class CreatingWeightSections(L7HeavyConfigOrderProcessor):
    __slots__ = ()
    state = State.CREATING_WEIGHT_SECTIONS
    next_state = State.SAVING_SPEC
    cancelled_state = None

    def process(self, ctx):
        for ws_pb in self.order_content.weight_sections_to_create:
            ws_id = ws_pb.meta.id
            if objects.WeightSection.zk.get(self.entity.namespace_id, ws_id) is not None:
                continue
            meta_pb = model_pb2.WeightSectionMeta(id=ws_id, namespace_id=self.entity.namespace_id,
                                                  comment='Created by L7Heavy config order')
            meta_pb.type = ws_pb.meta.type
            objects.WeightSection.create(meta_pb, ws_pb.spec, self.entity.pb.meta.author)
        return self.next_state


class SavingSpec(L7HeavyConfigOrderProcessor):
    __slots__ = ()
    state = State.SAVING_SPEC
    next_state = State.FINISHED
    cancelled_state = None

    def process(self, ctx):
        self.entity.pb.spec.incomplete = False
        self.entity.pb.spec.l7_heavy_config_id = self.entity.namespace_id
        self.entity.pb.spec.group_id = self.order_content.group_id or self.entity.context['group_id']
        self.entity.dao_update(u'Finished order, spec.incomplete = False')
        return self.next_state


class Cancelling(L7HeavyConfigOrderProcessor):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    def process(self, ctx):
        self.entity.pb.spec.incomplete = False
        self.entity.dao_update(u'Cancelled order, spec.incomplete = False')
        return self.next_state
