# coding: utf-8
import random
import time

import gevent
import inject
import monotonic
import six
from sepelib.core import config as appconfig

from awacs.lib import yasm_client, juggler_client, staffclient, context
from awacs.model import events, cache, alerting, external_clusters
from awacs.model.dao import IDao, Dao
from awacs.model.zk import IZkStorage, ZkStorage
from awacs.model.namespace.base import BaseProcessor, NAMESPACE_CTL_REGISTRY
from infra.awacs.proto import model_pb2
from infra.swatlib import orly_client
from infra.swatlib.auth import abc
from infra.swatlib.orly_client import OrlyBrakeApplied


ALERTING_CTL_REGISTRY = NAMESPACE_CTL_REGISTRY.path('alerting')

sync_success_counter = ALERTING_CTL_REGISTRY.get_counter('sync-succeeded')
sync_failed_counter = ALERTING_CTL_REGISTRY.get_counter('sync-failed')
sync_postponed_counter = ALERTING_CTL_REGISTRY.get_counter('sync-postponed')

update_version_counter = ALERTING_CTL_REGISTRY.get_counter('triggered-by-version')
update_juggler_downtimers_counter = ALERTING_CTL_REGISTRY.get_counter('triggered-by-juggler-downtimers')
update_notify_rules_counter = ALERTING_CTL_REGISTRY.get_counter('triggered-by-notify-rules')

skip_balancer_counter = ALERTING_CTL_REGISTRY.get_counter('skip-balancer')
incomplete_balancer_counter = ALERTING_CTL_REGISTRY.get_counter('incomplete-balancer')

juggler_ns_create_counter = ALERTING_CTL_REGISTRY.get_counter('juggler-ns-created')
juggler_ns_update_counter = ALERTING_CTL_REGISTRY.get_counter('juggler-ns-updated')
juggler_ns_delete_counter = ALERTING_CTL_REGISTRY.get_counter('juggler-ns-deleted')

juggler_nr_create_counter = ALERTING_CTL_REGISTRY.get_counter('juggler-nr-created')
juggler_nr_delete_counter = ALERTING_CTL_REGISTRY.get_counter('juggler-nr-deleted')

juggler_ach_changed_counter = ALERTING_CTL_REGISTRY.get_counter('juggler-ach-changed')
juggler_ach_removed_counter = ALERTING_CTL_REGISTRY.get_counter('juggler-ach-removed')

yasm_alerts_create_counter = ALERTING_CTL_REGISTRY.get_counter('yasm-alerts-created')
yasm_alerts_update_counter = ALERTING_CTL_REGISTRY.get_counter('yasm-alerts-updated')
yasm_alerts_delete_counter = ALERTING_CTL_REGISTRY.get_counter('yasm-alerts-deleted')

DEFAULT_NAMESPACE_DOWNTIMERS = ['robot-walle', '@svc_maintcoord']
DEFAULT_NAMESPACE_OWNERS = ['robot-yasm-golovan', '@svc_rclb']


class AlertingProcessor(BaseProcessor):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(IZkStorage)  # type: ZkStorage
    _dao = inject.attr(IDao)  # type: Dao
    _yasm_client = inject.attr(yasm_client.IYasmClient)  # type: yasm_client.YasmClient
    _juggler_client = inject.attr(juggler_client.IJugglerClient)  # type: juggler_client.JugglerClient
    _abc_client = inject.attr(abc.IAbcClient)  # type: abc.AbcClient
    _staff_client = inject.attr(staffclient.IStaffClient)  # type: staffclient.StaffClient
    _sync_brake = orly_client.OrlyBrake(rule='awacs-namespace-alerting-sync', metrics_registry=ALERTING_CTL_REGISTRY)

    SLEEP_AFTER_CREATE_JUGGLER_NAMESPACE = 30
    DOWNTIME_AFTER_CREATE_JUGGLER_NAMESPACE_SECONDS = 15 * 60
    DOWNTIME_AFTER_CHANGE_YASM_ALERTS_SECONDS = 5 * 60

    DEFAULT_BALANCER_UI_URL_TEMPLATE = ('https://nanny.yandex-team.ru/ui/'
                                        '#/awacs/namespaces/list/{}/balancers/list/{}/show/')

    DEFAULT_ALERTING_UI_URL_TEMPLATE = ('https://nanny.yandex-team.ru/ui/'
                                        '#/awacs/namespaces/list/{}/alerting/')
    CHECK_INTERVAL_LOWER_BOUND = 20
    CHECK_INTERVAL_UPPER_BOUND = 40

    def __init__(self, namespace_id, log, alerting_cfg):
        """
        :type namespace_id: six.text_type
        :type log: logging.Logger
        :type alerting_cfg: dict
        """
        super(AlertingProcessor, self).__init__(namespace_id, log)

        self._alerting_cfg = alerting_cfg
        self._alerting_spec_pb = None
        self._alerting_prefix = alerting_cfg['name_prefix']

        self._downtime_after_create_juggler_ns_seconds = alerting_cfg.get(
            'downtime_after_create_juggler_ns_seconds',
            self.DOWNTIME_AFTER_CREATE_JUGGLER_NAMESPACE_SECONDS)

        self._downtime_after_change_yasm_alert_seconds = alerting_cfg.get(
            'downtime_after_change_yasm_alerts_seconds',
            self.DOWNTIME_AFTER_CHANGE_YASM_ALERTS_SECONDS)

        self._balancer_ui_url_template = alerting_cfg.get(
            'balancer_ui_url_tpl',
            self.DEFAULT_BALANCER_UI_URL_TEMPLATE)

        self._alerting_ui_url_template = alerting_cfg.get(
            'alerting_ui_url_tpl',
            self.DEFAULT_ALERTING_UI_URL_TEMPLATE)

        if 'sync_delay_interval_from' in alerting_cfg:
            self._sync_delay_interval_lower_bound = alerting_cfg['sync_delay_interval_from']
        if 'sync_delay_interval_to' in alerting_cfg:
            self._sync_delay_interval_upper_bound = alerting_cfg['sync_delay_interval_to']
        self._reset_sync_check_timers()
        self._diff_check_deadline = monotonic.monotonic()

    def start(self):
        super(AlertingProcessor, self).start()
        self._alerting_spec_pb = None

    def _set_sync_status_pb(self, ctx, sync_status_pb):
        updated_namespace_pb = self._dao.update_namespace(
            namespace_id=self._namespace_id,
            updated_alerting_sync_status_pb=sync_status_pb)
        ctx.log.debug('updated alerting sync status, new meta.generation: %s',
                      updated_namespace_pb.meta.generation)

    def _get_alerting_config(self):
        if not self._pb.HasField('spec') or not self._pb.spec.HasField('alerting'):
            return None
        return alerting.NamespaceAlertingConfig(version=self._pb.spec.alerting.version,
                                                namespace_id=self._namespace_id,
                                                alerting_prefix=self._alerting_prefix)

    def _needs_alerting_sync(self, ctx, alerting_config, current_alerting_setting, alerting_settings,
                             sync_check_deadline, current_monotonic_time, is_first_start, balancers_changed):
        """

        :type ctx: context.OpCtx
        :type alerting_config: alerting.AlertingConfig
        :type current_alerting_setting: model_pb2.NamespaceSpec.AlertingSettings
        :type alerting_settings: model_pb2.NamespaceSpec.AlertingSettings
        :type sync_check_deadline: float
        :type current_monotonic_time: float
        :type is_first_start: bool
        :type balancers_changed: bool
        :return:
        """
        if current_alerting_setting != alerting_settings:
            if current_alerting_setting.version != alerting_settings.version:
                ctx.log.info('alerting version has been changed: %s -> %s',
                             current_alerting_setting.version, alerting_settings.version)
                update_version_counter.inc()

            if (alerting_config.get_juggler_raw_downtimers(current_alerting_setting)
                    != alerting_config.get_juggler_raw_downtimers(alerting_settings)):
                ctx.log.info('alerting juggler downtimers has been changed')
                update_juggler_downtimers_counter.inc()

            if (alerting_config.get_juggler_raw_notify_rules(current_alerting_setting)
                    != alerting_config.get_juggler_raw_notify_rules(alerting_settings)):
                ctx.log.info('alerting juggler notify rules has been changed')
                update_notify_rules_counter.inc()

            return True

        if balancers_changed:
            ctx.log.info("balancers set has changed, alerting needs resync")
            return True

        if is_first_start and ((time.time() - self._pb.meta.mtime.ToSeconds()) < self._sync_delay_interval):
            ctx.log.info('maybe alerting just enabled')
            return True

        if current_monotonic_time < sync_check_deadline:
            return False

        return True

    def _sync_juggler_namespace(self, alerting_config, juggler_namespace, abc_service_slug):
        """

        :type alerting_config: alerting.AlertingConfig
        :type juggler_namespace: six.text_type
        :type abc_service_slug: six.text_type
        :rtype: juggler_client.CreateOrUpdateNamespaceResult
        """
        juggler_raw_downtimers = alerting_config.get_juggler_raw_downtimers(self._pb.spec.alerting)

        # https://st.yandex-team.ru/MONITORINGOPS-793
        downtimers = set(juggler_raw_downtimers.staff_logins)
        if juggler_raw_downtimers.staff_group_ids:
            groups_info = self._staff_client.get_groups_by_ids(juggler_raw_downtimers.staff_group_ids,
                                                               fields=('id', 'url'))
            downtimers.update("@{}".format(v['url']) for v in groups_info.values())
        downtimers.update(DEFAULT_NAMESPACE_DOWNTIMERS)
        downtimers = sorted(downtimers)

        owners = list(DEFAULT_NAMESPACE_OWNERS)

        result = self._juggler_client.create_or_update_namespace(
            name=juggler_namespace,
            abc_service_slug='svc_' + abc_service_slug.lower(),
            inherit_downtimers=False,
            downtimers=downtimers,
            owners=owners
        )
        if result.created:
            # TODO: remove after fix https://st.yandex-team.ru/JUGGLER-3759
            gevent.sleep(self.SLEEP_AFTER_CREATE_JUGGLER_NAMESPACE)

        return result

    def _balancers_have_multiple_schedulers(self, balancer_pbs):
        return (
            any(
                balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.YP_CLUSTER
                for balancer_pb in balancer_pbs
            )
            and any(
                balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.GENCFG_DC
                for balancer_pb in balancer_pbs
            )
        )

    def _sync_yasm_alerts(self, ctx, alerting_prefix, alerting_config, yasm_alerts_prefix, juggler_namespace,
                          abc_service_slug, check_groups):
        """
        :type ctx: context.OpCtx
        :type alerting_prefix: six.text_type
        :type alerting_config: alerting.AlertingConfig
        :type yasm_alerts_prefix: six.text_type
        :type juggler_namespace: six.text_type
        :type abc_service_slug: six.text_type
        :type check_groups: Set[model_pb2.NamespaceSpec.AlertingSettings.NotifyRuleGroup]
        :rtype: yasm_client.YasmReplaceAlertsResult
        """
        yasm_alerts = []
        balancer_pbs = self._cache.list_all_balancers(self._namespace_id)

        have_multiple_schedulers = self._balancers_have_multiple_schedulers(balancer_pbs)

        for balancer_pb in balancer_pbs:
            if balancer_pb.spec.incomplete:
                ctx.log.info('Balancer with id: {} has incomplete order, skip...'.format(balancer_pb.meta.id))
                incomplete_balancer_counter.inc()
                continue

            if not balancer_pb.meta.HasField('location'):
                ctx.log.warn('Balancer with id: {} does not have meta.location, skip...'.format(
                    balancer_pb.meta.id))
                skip_balancer_counter.inc()
                continue

            if not balancer_pb.spec.config_transport.nanny_static_file.HasField('instance_tags'):
                ctx.log.warn('Balancer with id: {} does not have '
                             'spec.config_transport.nanny_static_file.instance_tags, '
                             'skip...'.format(balancer_pb.meta.id))
                skip_balancer_counter.inc()
                continue

            if balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.YP_CLUSTER:
                location = balancer_pb.meta.location.yp_cluster
            elif balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.GENCFG_DC:
                location = balancer_pb.meta.location.gencfg_dc
            elif balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.AZURE_CLUSTER:
                location = external_clusters.AZURE_CLUSTERS_BY_NAME[balancer_pb.meta.location.azure_cluster].yp_cluster
            else:
                ctx.log.warn('Balancer with id: {} has '
                             'unsupported meta.location.type, skip...'.format(balancer_pb.meta.id))
                skip_balancer_counter.inc()
                continue

            instance_tags = balancer_pb.spec.config_transport.nanny_static_file.instance_tags
            juggler_check_tags = [
                "cplb",  # backward capability with old alerts
                self._namespace_id,  # backward capability with old alerts
                alerting_config.build_juggler_check_namespace_tag(alerting_prefix, self._namespace_id),
                alerting_config.build_juggler_check_balancer_tag(alerting_prefix, balancer_pb.meta.id),
            ]
            if location in ('IVA', 'MYT'):
                juggler_check_tags.append('a_geo_{}'.format(location.lower()))

            yasm_alert_suffix = (
                '_yp'
                if have_multiple_schedulers
                and balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.YP_CLUSTER
                else ''
            )
            yasm_alerts += alerting_config.gen_balancer_yasm_alerts(
                alerting_prefix=self._alerting_prefix,
                yasm_alert_prefix=yasm_alerts_prefix,
                juggler_namespace=juggler_namespace,
                juggler_check_tags=juggler_check_tags,
                itype=instance_tags.itype,
                ctype=instance_tags.ctype,
                prj=instance_tags.prj,
                location=location,
                balancer_ui_url=self._balancer_ui_url_template.format(self._pb.meta.id, balancer_pb.meta.id),
                balancer_pb=balancer_pb,
                namespace_id=self._namespace_id,
                abc_service_slug=abc_service_slug,
                check_groups=check_groups,
                yasm_alert_suffix=yasm_alert_suffix,
            )
        return self._yasm_client.replace_alerts(yasm_alerts_prefix, yasm_alerts)

    def _sync_juggler_checks(self, ctx, alerting_prefix, alerting_config, juggler_namespace, check_groups):
        """
        :type ctx: context.OpCtx
        :type alerting_prefix: six.text_type
        :type alerting_config: alerting.AlertingConfig
        :type juggler_namespace: six.text_type
        :type check_groups: Set[model_pb2.NamespaceSpec.AlertingSettings.NotifyRuleGroup]
        :rtype: juggler_client.SyncChecksResult
        """
        juggler_sdk_checks = []
        balancer_pbs = self._cache.list_all_balancers(self._namespace_id)

        have_multiple_schedulers = self._balancers_have_multiple_schedulers(balancer_pbs)

        for balancer_pb in balancer_pbs:
            if balancer_pb.spec.incomplete:
                ctx.log.info('Balancer with id: {} has incomplete order, skip...'.format(balancer_pb.meta.id))
                incomplete_balancer_counter.inc()
                continue

            if not balancer_pb.meta.HasField('location'):
                ctx.log.warn('Balancer with id: {} does not have meta.location, skip...'.format(
                    balancer_pb.meta.id))
                skip_balancer_counter.inc()
                continue

            if not balancer_pb.spec.config_transport.nanny_static_file.HasField('instance_tags'):
                ctx.log.warn('Balancer with id: {} does not have '
                             'spec.config_transport.nanny_static_file.instance_tags, '
                             'skip...'.format(balancer_pb.meta.id))
                skip_balancer_counter.inc()
                continue

            if balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.YP_CLUSTER:
                location = balancer_pb.meta.location.yp_cluster
            elif balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.GENCFG_DC:
                location = balancer_pb.meta.location.gencfg_dc
            elif balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.AZURE_CLUSTER:
                location = external_clusters.AZURE_CLUSTERS_BY_NAME[balancer_pb.meta.location.azure_cluster].yp_cluster
            else:
                ctx.log.warn('Balancer with id: {} has '
                             'unsupported meta.location.type, skip...'.format(balancer_pb.meta.id))
                skip_balancer_counter.inc()
                continue

            juggler_check_tags = [
                "cplb",
                self._namespace_id,
                alerting_config.build_juggler_check_namespace_tag(alerting_prefix, self._namespace_id),
                alerting_config.build_juggler_check_balancer_tag(alerting_prefix, balancer_pb.meta.id),
                'a_geo_{}'.format(location.lower())
            ]

            alerting_suffix = (
                '_yp'
                if have_multiple_schedulers
                and balancer_pb.meta.location.type == model_pb2.BalancerMeta.Location.YP_CLUSTER
                else ''
            )

            juggler_sdk_checks += alerting_config.gen_balancer_juggler_checks(
                alerting_prefix=self._alerting_prefix,
                juggler_namespace=juggler_namespace,
                juggler_check_tags=sorted(juggler_check_tags),
                location=location,
                balancer_ui_url=self._balancer_ui_url_template.format(self._pb.meta.id, balancer_pb.meta.id),
                namespace_id=self._namespace_id,
                balancer_pb=balancer_pb,
                alerting_suffix=alerting_suffix,
                check_groups=check_groups,
            )

        juggler_checks_mark = alerting_config.build_juggler_checks_mark(alerting_prefix, self._namespace_id)
        ctx.log.debug("Syncing checks, mark %r: %s",
                      juggler_checks_mark,
                      [(check.host, check.service) for check in juggler_sdk_checks])
        return self._juggler_client.sync_checks(
            namespace=juggler_namespace,
            checks=juggler_sdk_checks,
            mark=juggler_checks_mark,
        )

    def _sync_juggler_notify_rules(self, alerting_prefix, alerting_config, juggler_namespace):
        """

        :type alerting_prefix: six.text_type
        :type alerting_config: alerting.AlertingConfig
        :type juggler_namespace: six.text_type
        :rtype: juggler_client.SyncNotifyRulesResult
        """
        alerting_ui_url = self._alerting_ui_url_template.format(self._namespace_id)
        description = 'Generated by awacs. Details: {}'.format(alerting_ui_url)

        juggler_notify_rules = alerting_config.gen_juggler_notify_rules(
            namespace_id=self._namespace_id,
            alerting_prefix=alerting_prefix,
            alerting_settings=self._pb.spec.alerting,
            description=description
        )

        return self._juggler_client.sync_notify_rules(juggler_namespace, [r[0] for r in juggler_notify_rules])

    def maybe_self_delete(self, ctx):
        alerting_config = self._get_alerting_config()
        if alerting_config is None:
            return True

        juggler_checks_mark = alerting_config.juggler_checks_mark
        yasm_alert_prefix = alerting_config.yasm_alert_prefix
        juggler_namespace = alerting_config.juggler_namespace

        replace_alerts_res = self._yasm_client.replace_alerts(yasm_alert_prefix, [])
        ctx.log.info('removed {} yasm alerts'.format(replace_alerts_res.deleted))
        yasm_alerts_delete_counter.inc(replace_alerts_res.deleted)

        sync_notify_rules_res = self._juggler_client.sync_notify_rules(juggler_namespace, [])
        ctx.log.info('removed {} juggler notify rules'.format(sync_notify_rules_res.remove))
        juggler_nr_delete_counter.inc(sync_notify_rules_res.remove)

        sync_juggler_checks_res = self._juggler_client.cleanup_checks(juggler_namespace, juggler_checks_mark)
        juggler_ach_removed_counter.inc(len(sync_juggler_checks_res.removed))

        try:
            is_juggler_ns_removed = self._juggler_client.remove_namespace_if_exists(juggler_namespace)
            if is_juggler_ns_removed:
                juggler_ns_delete_counter.inc()
            return True
        except self._juggler_client.BadRequestError as e:
            ctx.log.info('juggler namespace "{}" can not be removed at this moment, error: {}'.format(
                juggler_namespace, e))
            return False

    def process(self, ctx, event):
        current_monotonic_time = monotonic.monotonic()
        if current_monotonic_time < self._diff_check_deadline:
            return
        self._diff_check_deadline = current_monotonic_time + (
                random.randint(self.CHECK_INTERVAL_LOWER_BOUND, self.CHECK_INTERVAL_UPPER_BOUND) *
                appconfig.get_value('run.ns_alerting_processor_slowdown_factor', default=1))

        alerting_config = self._get_alerting_config()
        if alerting_config is None:
            return

        balancers_changed = isinstance(event, (events.BalancerUpdate, events.BalancerRemove))
        # prevent start ctl after deploy awacs
        self._alerting_spec_pb = self._alerting_spec_pb or self._pb.spec.alerting

        if not self._needs_alerting_sync(
                ctx, alerting_config.config, self._alerting_spec_pb,
                self._pb.spec.alerting, self._sync_check_deadline, monotonic.monotonic(), self._is_first_start,
                balancers_changed,
        ):
            return

        try:
            self._sync_brake.maybe_apply(op_id=ctx.id(), op_log=self._log, op_labels=[
                ('namespace-id', self._namespace_id),
                ('alerting-version', self._pb.spec.alerting.version),
            ])
        except OrlyBrakeApplied:
            sync_postponed_counter.inc()
            self._postpone_sync_check_deadline()
            return
        self._alerting_spec_pb = self._pb.spec.alerting
        self._reset_sync_check_timers()
        self._is_first_start = False

        ctx.log.info('Namespace sync alerting config started')

        with self._sync_attempt(ctx, sync_failed_counter, sync_success_counter, self._pb.alerting_sync_status) as attempt_step:
            # Get ABC Service Slug
            attempt_step.title = 'get abc service slug'
            abc_service_slug = self._abc_client.get_service_slug(self._pb.meta.abc_service_id)

            # Sync Juggler Namespace
            attempt_step.title = 'sync juggler namespace'
            sync_juggler_namespace_res = self._sync_juggler_namespace(alerting_config.config,
                                                                      alerting_config.juggler_namespace,
                                                                      abc_service_slug)
            ctx.log.info('Success sync juggler namespace: {}'.format(sync_juggler_namespace_res))
            juggler_ns_create_counter.inc(int(sync_juggler_namespace_res.created))
            juggler_ns_update_counter.inc(int(sync_juggler_namespace_res.updated))

            # Set Downtime if needs
            if sync_juggler_namespace_res.created:
                attempt_step.title = 'set downtime after create juggler namespace'
                self._juggler_client.set_downtime_to_namespace(
                    namespace=alerting_config.juggler_namespace,
                    seconds=self._downtime_after_create_juggler_ns_seconds,
                    description='Downtime for prevent CRIT juggler checks with NO_DATA',
                    source=self._alerting_prefix
                )

            check_groups = {alerting.PLATFORM_GROUP}
            if (
                not self._alerting_spec_pb.balancer_checks_disabled
                and self._pb.spec.env_type != model_pb2.NamespaceSpec.NS_ENV_TESTING
            ):
                check_groups.add(alerting.BALANCER_GROUP)

            # Sync Yasm Alerts
            attempt_step.title = 'sync yasm alerts'
            sync_yasm_alerts_res = self._sync_yasm_alerts(ctx, self._alerting_prefix, alerting_config.config,
                                                          alerting_config.yasm_alert_prefix,
                                                          alerting_config.juggler_namespace, abc_service_slug,
                                                          check_groups)
            ctx.log.info('Success sync yasm alerts: {}'.format(sync_yasm_alerts_res))
            yasm_alerts_create_counter.inc(sync_yasm_alerts_res.created)
            yasm_alerts_delete_counter.inc(sync_yasm_alerts_res.deleted)
            yasm_alerts_update_counter.inc(sync_yasm_alerts_res.updated)

            # Set Downtime if needs
            if sync_yasm_alerts_res.created or sync_yasm_alerts_res.deleted:
                attempt_step.title = 'set downtime after create or delete yasm alert'
                self._juggler_client.set_downtime_to_namespace(
                    namespace=alerting_config.juggler_namespace,
                    seconds=self._downtime_after_change_yasm_alert_seconds,
                    description='Downtime for prevent CRIT juggler checks with NO_DATA',
                    source=self._alerting_prefix
                )

            # Sync Juggler Checks
            attempt_step.title = 'sync juggler checks'
            sync_juggler_checks_res = self._sync_juggler_checks(ctx, self._alerting_prefix, alerting_config.config,
                                                                alerting_config.juggler_namespace, check_groups)
            ctx.log.info('Success sync juggler checks: {}'.format(sync_juggler_checks_res))
            juggler_ach_changed_counter.inc(len(sync_juggler_checks_res.changed))
            juggler_ach_removed_counter.inc(len(sync_juggler_checks_res.removed))

            # Sync Juggler Notify Rules
            attempt_step.title = 'sync juggler notify rules'
            sync_juggler_notify_rules_res = self._sync_juggler_notify_rules(self._alerting_prefix,
                                                                            alerting_config.config,
                                                                            alerting_config.juggler_namespace)
            juggler_nr_create_counter.inc(sync_juggler_notify_rules_res.add)
            juggler_nr_delete_counter.inc(sync_juggler_notify_rules_res.remove)

            ctx.log.info('Success sync juggler notify rules: {}'.format(sync_juggler_notify_rules_res))
