# coding: utf-8
import random
from contextlib import contextmanager

import gevent
import inject
import monotonic
import six
import time

from awacs.lib import ctlmanager, context
from awacs.model import util
from infra.awacs.proto import model_pb2
from infra.swatlib import metrics


NAMESPACE_CTL_REGISTRY = metrics.ROOT_REGISTRY.path('ns-ctl')


class AttemptStep(object):
    title = None


class BaseProcessor(object):
    DEFAULT_SYNC_DELAY_INTERVAL_LOWER_BOUND = 60 * 60
    DEFAULT_SYNC_DELAY_INTERVAL_UPPER_BOUND = 150 * 60

    def __init__(self, namespace_id, log):
        """
        :type namespace_id: six.text_type
        :type log: logging.Logger
        """
        self._namespace_id = namespace_id
        self._log = log

        self._sync_delay_interval_lower_bound = self.DEFAULT_SYNC_DELAY_INTERVAL_LOWER_BOUND
        self._sync_delay_interval_upper_bound = self.DEFAULT_SYNC_DELAY_INTERVAL_UPPER_BOUND

        self._reset_sync_check_timers()

        self._is_first_start = None
        self._pb = None

    def set_pb(self, pb):
        self._pb = pb

    def start(self):
        self._reset_sync_check_timers()
        self._is_first_start = True

    def stop(self):
        pass

    def _reset_sync_check_timers(self):
        self._sync_delay_interval = random.randint(self._sync_delay_interval_lower_bound,
                                                   self._sync_delay_interval_upper_bound)
        self._sync_check_deadline = monotonic.monotonic() + self._sync_delay_interval

    def _postpone_sync_check_deadline(self):
        self._sync_check_deadline += random.randint(60, 5 * 60)

    def _set_sync_status_pb(self, ctx, sync_status_pb):
        raise NotImplementedError

    def _set_sync_attempt(self, ctx, current_status_pb, updated_last_attempt_pb):
        """
        :type ctx: context.OpCtx
        :type current_status_pb: model_pb2.SyncStatus
        :type updated_last_attempt_pb: model_pb2.SyncAttempt
        """
        last_attempt_pb = current_status_pb.last_attempt
        if (last_attempt_pb.succeeded == updated_last_attempt_pb.succeeded and
                last_attempt_pb.revision_id == updated_last_attempt_pb.revision_id and
                last_attempt_pb.sync_id == updated_last_attempt_pb.sync_id):
            return

        if current_status_pb.HasField('last_successful_attempt'):
            last_successful_attempt_pb = current_status_pb.last_successful_attempt
        else:
            last_successful_attempt_pb = None
        if updated_last_attempt_pb.succeeded.status == 'True':
            last_successful_attempt_pb = updated_last_attempt_pb

        updated_sync_status_pb = util.clone_pb(current_status_pb)
        updated_sync_status_pb.last_attempt.CopyFrom(updated_last_attempt_pb)
        if last_successful_attempt_pb is not None:
            updated_sync_status_pb.last_successful_attempt.CopyFrom(last_successful_attempt_pb)
        self._set_sync_status_pb(ctx, updated_sync_status_pb)

    @contextmanager
    def _sync_attempt(self, ctx, sync_failed_counter, sync_success_counter, sync_status_pb):
        """
        :type ctx: context.OpCtx
        """
        attempt_pb = model_pb2.SyncAttempt()
        attempt_pb.revision_id = self._pb.meta.version
        attempt_pb.sync_id = ctx.id()
        attempt_pb.started_at.GetCurrentTime()

        attempt_step = AttemptStep()
        try:
            yield attempt_step
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            msg = 'Failed to {}: {}'.format(attempt_step.title, e)
            ctx.log.exception('Sync failed: %s', e)
            attempt_pb.finished_at.GetCurrentTime()
            attempt_pb.succeeded.status = 'False'
            attempt_pb.succeeded.message = msg
            self._set_sync_attempt(ctx, sync_status_pb, attempt_pb)
            sync_failed_counter.inc()
            raise e
        else:
            attempt_pb.finished_at.GetCurrentTime()
            attempt_pb.succeeded.status = 'True'
            attempt_pb.succeeded.message = ''
            self._set_sync_attempt(ctx, sync_status_pb, attempt_pb)
            sync_success_counter.inc()

    def maybe_self_delete(self, ctx):
        raise NotImplementedError

    def process(self, ctx, event):
        raise NotImplementedError