# coding: utf-8

import inject
import monotonic
import six
from sepelib.core import config as appconfig

from awacs.lib import ctlmanager, context
from awacs.lib.order_processor.model import needs_removal
from awacs.model import events, cache
from awacs.model.dao import IDao, Dao

from awacs.model.namespace.its_processor import ItsProcessor
from awacs.model.namespace.alerting_processor import AlertingProcessor
from awacs.model.namespace.gc_processor import GCProcessor


class NamespaceCtl(ctlmanager.ContextedCtl):
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _dao = inject.attr(IDao)  # type: Dao

    EVENTS_QUEUE_GET_TIMEOUT = 10
    PROCESSING_INTERVAL = 6
    SELF_ACTIONS_DELAY_INTERVAL = 60 * 5
    SELF_DELETION_DELAY_INTERVAL = 60

    def __init__(self, namespace_id, alerting_cfg):
        """
        :type namespace_id: six.text_type
        :type alerting_cfg: dict
        """
        name = 'namespace-ctl("{}")'.format(namespace_id)
        super(NamespaceCtl, self).__init__(name)
        self._namespace_id = namespace_id

        self._alerting_processor = AlertingProcessor(namespace_id, self._log, alerting_cfg)
        self._its_processor = ItsProcessor(namespace_id, self._log)
        self._gc_processor = GCProcessor(namespace_id, self._log)

        self._self_deletion_check_deadline = monotonic.monotonic()
        self._pb = None

    def _start(self, ctx):
        self._self_deletion_check_deadline = monotonic.monotonic()
        self._alerting_processor.start()
        self._its_processor.start()
        self._gc_processor.start()
        try:
            self._process(ctx)
        except ctlmanager.UNEXPECTED_EXCEPTIONS as e:
            ctx.log.exception('failed to process namespace on start: %s', e)
        self._cache.bind(self._callback)

    def _stop(self):
        self._cache.unbind(self._callback)
        self._alerting_processor.stop()
        self._its_processor.stop()
        self._gc_processor.stop()

    def _maybe_self_delete(self, ctx):
        current_monotonic_time = monotonic.monotonic()
        ctx.log.debug('namespace marked for removal')
        if current_monotonic_time < self._self_deletion_check_deadline:
            ctx.log.info('too little time passed since last deletion check: current_monotonic_time is %s, '
                         'next check at %s', current_monotonic_time, self._self_deletion_check_deadline)
            return False
        self._self_deletion_check_deadline = current_monotonic_time + self.SELF_DELETION_DELAY_INTERVAL

        if not self._alerting_processor.maybe_self_delete(ctx):
            return False
        if not self._its_processor.maybe_self_delete(ctx):
            return False

        can_delete_namespace, reason = self._dao.can_delete_namespace(self._pb)
        if not can_delete_namespace:
            ctx.log.warn(reason)
            return False
        self._dao.delete_namespace(self._namespace_id)
        ctx.log.info('namespace removed')
        return True

    def _process(self, ctx, event=None):
        """
        :type ctx: context.OpCtx
        """
        self._pb = self._cache.must_get_namespace(self._namespace_id)
        self._alerting_processor.set_pb(self._pb)
        self._its_processor.set_pb(self._pb)
        self._gc_processor.set_pb(self._pb)

        errors = []
        if needs_removal(self._pb):
            deleted = self._maybe_self_delete(ctx)
            if deleted:
                return
        else:
            try:
                if not appconfig.get_value('run.disable_ns_alerting_processor', default=False):
                    self._alerting_processor.process(ctx, event)
            except Exception as e:
                self._log.exception(e)
                errors.append(e)
        try:
            if not appconfig.get_value('run.disable_ns_its_processor', default=False):
                self._its_processor.process(ctx, event)
        except Exception as e:
            self._log.exception(e)
            errors.append(e)
        try:
            if not appconfig.get_value('run.disable_ns_gc_processor', default=False):
                self._gc_processor.process(ctx, event)
        except Exception as e:
            self._log.exception(e)
            errors.append(e)
        if errors:
            msg = '\n'.join(map(six.text_type, errors))
            raise RuntimeError(msg)

    def _accept_event(self, event):
        if isinstance(event, events.NamespaceUpdate) and event.pb.meta.id == self._namespace_id:
            return True
        if isinstance(event, events.BalancerUpdate) and event.pb.meta.namespace_id == self._namespace_id:
            return True
        if isinstance(event, events.BalancerRemove) and event.namespace_id == self._namespace_id:
            return True
        if isinstance(event, events.BalancerStateUpdate) and event.pb.namespace_id == self._namespace_id:
            return True
        if isinstance(event, events.BalancerStateRemove) and event.namespace_id == self._namespace_id:
            return True
        return False

    def _process_event(self, ctx, event):
        assert isinstance(event, (events.NamespaceUpdate, events.BalancerUpdate, events.BalancerRemove,
                                  events.BalancerStateUpdate, events.BalancerStateRemove))
        self._process(ctx, event)

    def _process_empty_queue(self, ctx):
        self._process(ctx)
