# coding: utf-8
import gevent
import inject
import monotonic
import time

from sepelib.core import config

from awacs.lib import context
from awacs.lib.gutils import gevent_idle_iter
from awacs.model import events, cache, objects
from awacs.model.dao import IDao, Dao
from awacs.lib.strutils import to_full_ids
from awacs.model.zk import IZkStorage, ZkStorage
from awacs.model.namespace.base import BaseProcessor


class GCProcessor(BaseProcessor):
    DEFAULT_SYNC_DELAY_INTERVAL_LOWER_BOUND = 2 * 60
    DEFAULT_SYNC_DELAY_INTERVAL_UPPER_BOUND = 5 * 60

    _dao = inject.attr(IDao)  # type: Dao
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _zk = inject.attr(IZkStorage)  # type: ZkStorage

    def _needs_sync(self, ctx, event):
        """
        :type ctx: context.OpCtx
        :rtype: bool
        """
        if isinstance(event, (events.BalancerStateUpdate, events.BalancerStateRemove)):
            ctx.log.info("balancers states has changed, needs resync")
            return True

        if self._is_first_start and ((time.time() - self._pb.meta.mtime.ToSeconds()) < self._sync_delay_interval):
            return True

        if monotonic.monotonic() < self._sync_check_deadline:
            return False

        return True

    def process(self, ctx, event):
        enabled_namespace_ids = config.get_value('run.enable_gc_processor_namespace_ids', ())
        if enabled_namespace_ids and self._namespace_id not in enabled_namespace_ids:
            ctx.log.info("GC processor disabled for {}".format(self._namespace_id))
            return

        if not self._needs_sync(ctx, event):
            self._is_first_start = False
            return

        self._is_first_start = False

        try:
            used_in_balancers_upstream_ids = set()
            used_in_balancers_weight_section_ids = set()
            used_in_l7heavy_weight_section_ids = set()
            balancer_ids = set()
            for balancer_state_pb in self._cache.list_all_balancer_states(namespace_id=self._namespace_id):
                used_in_balancers_upstream_ids.update(to_full_ids(self._namespace_id, balancer_state_pb.upstreams.keys()))
                used_in_balancers_weight_section_ids.update(to_full_ids(self._namespace_id, balancer_state_pb.weight_sections.keys()))
                balancer_ids.add(balancer_state_pb.balancer_id)

            l7hc_state_pb = objects.L7HeavyConfig.state.cache.get(self._namespace_id, self._namespace_id)
            if l7hc_state_pb is not None:
                used_in_l7heavy_weight_section_ids.update(to_full_ids(self._namespace_id, l7hc_state_pb.weight_sections.keys()))

            to_remove_upstream_ids = set()
            to_remove_weight_section_ids = set()
            for upstream_pb in gevent_idle_iter(self._cache.list_all_upstreams(self._namespace_id,
                                                                               query={self._cache.UpstreamsQueryTarget.DELETED: True})):
                assert upstream_pb.spec.deleted
                full_upstream_id = (self._namespace_id, upstream_pb.meta.id)
                if full_upstream_id in used_in_balancers_upstream_ids:
                    continue
                to_remove_upstream_ids.add(full_upstream_id)

            for weight_section_pb in gevent_idle_iter(
                    objects.WeightSection.cache.list(self._namespace_id,
                                                      query={objects.WeightSection.cache.QueryTarget.DELETED: True})):
                assert weight_section_pb.spec.deleted
                full_weight_section_id = (self._namespace_id, weight_section_pb.meta.id)
                if (full_weight_section_id in used_in_balancers_weight_section_ids or
                        full_weight_section_id in used_in_l7heavy_weight_section_ids):
                    continue
                to_remove_weight_section_ids.add(full_weight_section_id)

            if not to_remove_upstream_ids and not to_remove_weight_section_ids:
                return

            # Now check balancer states directly from zk to avoid races
            self._zk.sync_balancer_states(self._namespace_id)
            for balancer_id in balancer_ids:
                balancer_state_pb = self._zk.must_get_balancer_state(namespace_id=self._namespace_id,
                                                                     balancer_id=balancer_id)
                to_remove_upstream_ids -= set(to_full_ids(self._namespace_id, balancer_state_pb.upstreams.keys()))
                to_remove_weight_section_ids -= set(to_full_ids(self._namespace_id, balancer_state_pb.weight_sections.keys()))

            l7hc_state_pb = objects.L7HeavyConfig.state.zk.get(self._namespace_id, self._namespace_id, sync=True)
            if l7hc_state_pb is not None:
                to_remove_weight_section_ids -= set(to_full_ids(self._namespace_id, l7hc_state_pb.weight_sections.keys()))

            for full_upstream_id in to_remove_upstream_ids:
                self._dao.delete_upstream(*full_upstream_id)
                self._log.info('Removed upstream %s:%s', *full_upstream_id)

            for full_weight_section_id in to_remove_weight_section_ids:
                objects.WeightSection.remove(*full_weight_section_id)
                self._log.info('Removed weight section %s:%s', *full_weight_section_id)
        finally:
            self._reset_sync_check_timers()
