# coding: utf-8
import inject
import monotonic
import six
import time
import collections

from awacs.lib import context, itsclient, staffclient
from awacs.model import events, cache, util
from awacs.model.dao import IDao, Dao
from awacs.model.namespace.base import BaseProcessor, NAMESPACE_CTL_REGISTRY
from awacs.model.namespace import its_helpers
from infra.awacs.proto import model_pb2
from infra.swatlib import orly_client
from infra.swatlib.orly_client import OrlyBrakeApplied
from sepelib.core import config as appconfig


ITS_CTL_REGISTRY = NAMESPACE_CTL_REGISTRY.path('its')

sync_success_counter = ITS_CTL_REGISTRY.get_counter('sync-succeeded')
sync_failed_counter = ITS_CTL_REGISTRY.get_counter('sync-failed')
sync_postponed_counter = ITS_CTL_REGISTRY.get_counter('sync-postponed')

update_section_counter = ITS_CTL_REGISTRY.get_counter('update-section')
delete_section_counter = ITS_CTL_REGISTRY.get_counter('delete-section')

BalancerSummary = collections.namedtuple('BalancerSummary', ['prj', 'ctype', 'geo', 'dc', 'env_type'])
BalancerSummaryV2 = collections.namedtuple('BalancerSummaryV2', ['service_id', 'env_type', 'geo', 'dc'])


class ItsProcessor(BaseProcessor):
    _dao = inject.attr(IDao)  # type: Dao
    _staff_client = inject.attr(staffclient.IStaffClient)  # type: staffclient.StaffClient
    _its_client = inject.attr(itsclient.IItsClient)  # type: itsclient.ItsClient
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    _sync_brake = orly_client.OrlyBrake(rule='awacs-namespace-its-sync', metrics_registry=ITS_CTL_REGISTRY)

    def __init__(self, namespace_id, log):
        """
        :type namespace_id: six.text_type
        :type log: logging.Logger
        """
        super(ItsProcessor, self).__init__(namespace_id, log)
        self._its_spec_pb = self._cache.get_namespace(self._namespace_id).spec.its

    def start(self):
        super(ItsProcessor, self).start()
        self._its_spec_pb = self._cache.get_namespace(self._namespace_id).spec.its

    def _set_sync_status_pb(self, ctx, sync_status_pb):
        updated_namespace_pb = self._dao.update_namespace(
            namespace_id=self._namespace_id,
            updated_its_sync_status_pb=sync_status_pb)
        ctx.log.debug('updated ITS sync status, new meta.generation: %s',
                      updated_namespace_pb.meta.generation)

    def _needs_its_sync(self, ctx, current_its_settings, its_settings,
                        sync_check_deadline, current_monotonic_time, is_first_start, balancers_changed):
        """
        :type ctx: context.OpCtx
        :type current_its_settings: model_pb2.NamespaceSpec.Its
        :type its_settings: model_pb2.NamespaceSpec.Its
        :type sync_check_deadline: float
        :type current_monotonic_time: float
        :type is_first_start: bool
        :type balancers_changed: bool
        :return:
        """
        if current_its_settings != its_settings:
            return True

        if balancers_changed:
            ctx.log.info("balancers set has changed, ITS needs resync")
            return True

        if is_first_start and ((time.time() - self._pb.meta.mtime.ToSeconds()) < self._sync_delay_interval):
            return True

        if current_monotonic_time < sync_check_deadline:
            return False

        return True

    def maybe_self_delete(self, ctx):
        if not self._pb.spec.HasField('its'):
            return True
        # If namespace is being deleted, there are no balancers in it. All ITS knobs should be removed, lets check it

        content = self._its_client.get_location('balancer')['content']
        normalised_prj = self._cache.normalise_prj_tag(self._pb.spec.balancer_constraints.instance_tags.prj)
        if normalised_prj in content['groups']:
            return False
        for node in six.itervalues(content['groups']['all-service']['groups']):
            if normalised_prj in node['groups']:
                return False
        return True

    def process(self, ctx, event):
        if not self._pb.spec.HasField('its'):
            return
        if appconfig.get_value('run.disable_its_sync', False):
            ctx.log.info('ITS updating is disabled')
            return

        balancers_changed = isinstance(event, (events.BalancerUpdate, events.BalancerRemove))

        if not self._needs_its_sync(
                ctx, self._its_spec_pb, self._pb.spec.its, self._sync_check_deadline,
                monotonic.monotonic(), self._is_first_start, balancers_changed,
        ):
            return

        try:
            self._sync_brake.maybe_apply(op_id=ctx.id(), op_log=self._log, op_labels=[
                ('namespace-id', self._namespace_id),
            ])
        except OrlyBrakeApplied:
            sync_postponed_counter.inc()
            self._postpone_sync_check_deadline()
            return
        self._its_spec_pb = self._pb.spec.its
        self._reset_sync_check_timers()
        self._is_first_start = False

        ctx.log.info('Namespace sync ITS config started')

        with self._sync_attempt(ctx, sync_failed_counter, sync_success_counter, self._pb.its_sync_status) as attempt_step:
            normalised_prj = self._cache.normalise_prj_tag(self._pb.spec.balancer_constraints.instance_tags.prj)
            if not normalised_prj:
                raise AssertionError('Namespace without "spec.balancer_constraints.instance_tags.prj" found')
            if normalised_prj == 'all-service':
                raise AssertionError('"all-service" is blacklisted prj, ITS can not be enabled')

            # List balancer summaries
            attempt_step.title = 'list balancer summaries'
            balancer_pbs = self._cache.list_balancers(namespace_id=self._namespace_id)
            summaries = set()
            seen_ctypes_by_geo_and_env_type = {}
            for balancer_pb in balancer_pbs.items:
                if balancer_pb.spec.incomplete:
                    continue
                instance_tags = balancer_pb.spec.config_transport.nanny_static_file.instance_tags
                prj, ctype = instance_tags.prj, instance_tags.ctype
                assert prj == self._pb.spec.balancer_constraints.instance_tags.prj
                if balancer_pb.meta.location.type == balancer_pb.meta.location.YP_CLUSTER:
                    dc = util.yp_cluster_to_balancer_location(balancer_pb.meta.location.yp_cluster.upper()).lower()
                elif balancer_pb.meta.location.type == balancer_pb.meta.location.GENCFG_DC:
                    dc = util.gencfg_dc_to_balancer_location(balancer_pb.meta.location.gencfg_dc.upper()).lower()
                elif balancer_pb.meta.location.type == balancer_pb.meta.location.AZURE_CLUSTER:
                    raise AssertionError('ITS is not supported for external clouds')
                else:
                    raise AssertionError('Unknown balancer location type')
                if dc == 'unknown':
                    raise AssertionError('Found balancer with unknown location')

                geo = 'msk' if dc in ('myt', 'iva') else dc
                env_type = balancer_pb.spec.env_type
                if self._its_spec_pb.ctl_version < 2:
                    summaries.add(BalancerSummary(prj=prj, ctype=ctype, geo=geo, dc=dc, env_type=env_type))
                else:
                    service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
                    summaries.add(BalancerSummaryV2(service_id=service_id, env_type=env_type, geo=geo, dc=dc))

                key = (geo, env_type == model_pb2.BalancerSpec.L7_ENV_PRESTABLE)
                if seen_ctypes_by_geo_and_env_type.get(key, ctype) != ctype:
                    raise AssertionError('Found balancers with same geo and env type, but different ctypes')
                seen_ctypes_by_geo_and_env_type[key] = ctype

            # Construct ITS sections
            attempt_step.title = 'construct ITS sections'

            if not summaries or (not self._its_spec_pb.knobs.common_knobs and
                                 not self._its_spec_pb.knobs.by_balancer_knobs):
                main_section = None
            elif self._its_spec_pb.ctl_version < 2:
                main_section = self._construct_main_section(summaries)
            else:
                main_section = self._construct_main_section_v2(summaries)

            all_service_sections = {}
            if self._its_spec_pb.knobs.create_announce_knob_for_marty.value:
                if self._its_spec_pb.ctl_version == 0:
                    ctypes_by_geo = collections.defaultdict(set)
                    for summary in summaries:
                        ctypes_by_geo[summary.geo].add(summary.ctype)
                    for geo, ctypes in six.iteritems(ctypes_by_geo):
                        all_service_sections[geo] = its_helpers.announce_template(
                            self._pb.spec.balancer_constraints.instance_tags.prj, ctypes, geo)
                elif self._its_spec_pb.ctl_version == 1:
                    ctypes_by_geo_and_dc = collections.defaultdict(set)
                    for summary in summaries:
                        ctypes_by_geo_and_dc[(summary.geo, summary.dc)].add(summary.ctype)
                    for (geo, dc), ctypes in six.iteritems(ctypes_by_geo_and_dc):
                        all_service_sections[dc] = its_helpers.announce_template(
                            self._pb.spec.balancer_constraints.instance_tags.prj, ctypes, geo, dc=dc)
                else:
                    by_dc = collections.defaultdict(list)
                    for summary in summaries:
                        by_dc[summary.dc].append(summary.service_id)
                    for dc, service_ids in six.iteritems(by_dc):
                        all_service_sections[dc] = its_helpers.announce_template_v2(service_ids)

            # Get current ITS config
            attempt_step.title = 'get current config'
            location = self._its_client.get_location('balancer')
            config, version = location['content'], location['version']

            # Parse existing ITS sections
            attempt_step.title = 'parse existing ITS sections'
            current_main_section = config['groups'].get(normalised_prj)
            current_all_service_sections = {}
            all_service_node = config['groups']['all-service']['groups']
            for geo in all_service_node:
                current_all_service_sections[geo] = all_service_node[geo]['groups'].get(normalised_prj)

            # Update sections
            attempt_step.title = 'updating main section'
            main_section_location_path = 'balancer/{}'.format(normalised_prj)
            if current_main_section is not None and main_section is None:
                self._remove_its_location(ctx, main_section_location_path, version)
            elif current_main_section != main_section:
                self._update_its_location(ctx, main_section_location_path, version, main_section)

            for geo, current in six.iteritems(current_all_service_sections):
                location_path = 'balancer/all-service/{}/{}'.format(geo, normalised_prj)
                section = all_service_sections.get(geo)
                if current is not None and section is None:
                    self._remove_its_location(ctx, location_path, version)
                elif current != section:
                    self._update_its_location(ctx, location_path, version, section)

    def _remove_its_location(self, ctx, location_path, version):
        self._its_client.remove_location(location_path, version)
        ctx.log.info('Deleted section {}'.format(location_path))
        delete_section_counter.inc()
        return

    def _update_its_location(self, ctx, location_path, version, new_config):
        self._its_client.create_or_update_location(location_path, {'version': version, 'content': new_config})
        ctx.log.info('Updated section {}'.format(location_path))
        update_section_counter.inc()

    def _construct_main_section(self, summaries):
        """
        :type summaries: list[BalancerSummary]
        :rtype: dict
        """
        groups = []
        if self._its_spec_pb.acl.staff_group_ids:
            groups = [g['url']
                      for g in six.itervalues(self._staff_client.get_groups_by_ids(self._its_spec_pb.acl.staff_group_ids))]

        content = {'groups': {}}
        if self._its_spec_pb.knobs.common_knobs:
            ctypes = {summary.ctype for summary in summaries}
            geos = {summary.geo for summary in summaries}
            prj = self._pb.spec.balancer_constraints.instance_tags.prj
            its_helpers.fill_common_knobs(content, prj, ctypes, geos, self._its_spec_pb.knobs.common_knobs,
                                          groups, list(self._its_spec_pb.acl.logins))

        if self._its_spec_pb.knobs.by_balancer_knobs:
            for summary in summaries:
                its_helpers.fill_balancer_knobs(content, summary, self._its_spec_pb.knobs.by_balancer_knobs,
                                                self._its_spec_pb.knobs.split_msk.value, groups, list(self._its_spec_pb.acl.logins))
        return content

    def _construct_main_section_v2(self, summaries):
        """
        :type summaries: list[BalancerSummaryV2]
        :rtype: dict
        """
        groups = []
        if self._its_spec_pb.acl.staff_group_ids:
            groups = [g['url']
                      for g in six.itervalues(self._staff_client.get_groups_by_ids(self._its_spec_pb.acl.staff_group_ids))]

        content = {'groups': {}}
        if self._its_spec_pb.knobs.common_knobs:
            its_helpers.fill_common_knobs_v2(content, self._its_spec_pb.knobs.common_knobs, groups,
                                             list(self._its_spec_pb.acl.logins), [summary.service_id for summary in summaries])

        if self._its_spec_pb.knobs.by_balancer_knobs:
            by_env_type_and_geo = collections.defaultdict(list)
            for summary in summaries:
                geo_or_dc = summary.dc if self._its_spec_pb.knobs.split_msk.value else summary.geo
                by_env_type_and_geo[(summary.env_type, geo_or_dc)].append(summary.service_id)
            for (env_type, geo_or_dc), service_ids in six.iteritems(by_env_type_and_geo):
                its_helpers.fill_balancer_knobs_v2(content, self._its_spec_pb.knobs.by_balancer_knobs, groups,
                                                   list(self._its_spec_pb.acl.logins), env_type, geo_or_dc, service_ids)
        return content


