from sepelib.core import config as appconfig

from awacs.lib import ctlmanager
from awacs.lib.models.controllers import ModelCtlManager
from awacs.lib.order_processor.model import is_order_in_progress, has_actionable_spec, needs_removal
from awacs.model import events, objects
from awacs.model.namespace.ctl import NamespaceCtl, AlertingProcessor
from awacs.model.namespace.operations.ctl import NamespaceOperationCtl
from awacs.model.namespace.order.ctl import NamespaceOrderCtl


class NamespaceOrderCtlManager(ctlmanager.CtlManager):
    def __init__(self, coord, member_id, party_suffix, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type member_id: six.text_type
        :type party_suffix: six.text_type
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(NamespaceOrderCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            name='namespace-order-ctl-manager',
            starting_events=(events.NamespaceUpdate,),
            stopping_events=(events.NamespaceRemove,),
            # member_id=member_id,
            # party_suffix=party_suffix,
        )
        self.allowed_namespace_id_matcher = allowed_namespace_id_matcher

    def _get_ctl_human_readable_name(self, ctl_id):
        """
        :type ctl_id: six.text_type
        """
        return 'namespace-order("{}") ctl'.format(ctl_id)

    def _is_allowed_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: bool
        """
        return self.allowed_namespace_id_matcher is None or self.allowed_namespace_id_matcher(namespace_id)

    def _is_starting_event(self, event):
        return (super(NamespaceOrderCtlManager, self)._is_starting_event(event) and
                self._is_allowed_namespace(event.pb.meta.id) and
                is_order_in_progress(event.pb))

    def _is_stopping_event(self, event):
        if super(NamespaceOrderCtlManager, self)._is_stopping_event(event):
            return True
        return (super(NamespaceOrderCtlManager, self)._is_starting_event(event) and
                not is_order_in_progress(event.pb))

    def _list_all_ctl_ids(self):
        """
        :rtype: [six.text_type]
        """
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            if self._is_allowed_namespace(namespace_id) and is_order_in_progress(namespace_pb):
                yield namespace_id

    def _create_ctl(self, ctl_id):
        """
        :type ctl_id: six.text_type
        """
        return NamespaceOrderCtl(ctl_id)

    def _get_ctl_id_from_event(self, event):
        """
        :rtype: six.text_type
        """
        ctl_id = event.path.strip('/')
        if not ctl_id:
            raise AssertionError('"{}" can not be parsed into a correct namespace ctl id'.format(event.path))
        return ctl_id


class NamespaceCtlManager(ctlmanager.CtlManager):
    def __init__(self, coord, member_id, party_suffix, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type member_id: six.text_type
        :type party_suffix: six.text_type
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(NamespaceCtlManager, self).__init__(
            coord=coord,
            cache=cache,
            name='namespace-ctl-manager',
            starting_events=(events.NamespaceUpdate,),
            stopping_events=(events.NamespaceRemove,),
            # member_id=member_id,
            # party_suffix=party_suffix,
        )
        self.allowed_namespace_id_matcher = allowed_namespace_id_matcher

    @staticmethod
    def get_alerting_cfg():
        alerting_cfg = appconfig.get_value('alerting')
        if 'name_prefix' not in alerting_cfg:
            raise RuntimeError('config: "alerting.name_prefix" must be set')

        sync_delay_interval_lower_bound = alerting_cfg.get(
            'sync_delay_interval_from',
            AlertingProcessor.DEFAULT_SYNC_DELAY_INTERVAL_LOWER_BOUND)
        sync_delay_interval_upper_bound = alerting_cfg.get(
            'sync_delay_interval_to',
            AlertingProcessor.DEFAULT_SYNC_DELAY_INTERVAL_UPPER_BOUND)

        if sync_delay_interval_upper_bound - sync_delay_interval_lower_bound < 30 * 60:
            raise RuntimeError('config: "alerting.sync_delay_interval_to" must be at least 30 minutes (1800 seconds) '
                               'greater than the "alerting.sync_delay_interval_from". '
                               'Currently: sync_delay_interval_from={}, sync_delay_interval_to={}'
                               .format(sync_delay_interval_lower_bound, sync_delay_interval_upper_bound))
        return alerting_cfg

    def _get_ctl_human_readable_name(self, ctl_id):
        """
        :type ctl_id: six.text_type
        """
        return 'namespace("{}") ctl'.format(ctl_id)

    def _is_allowed_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: bool
        """
        return self.allowed_namespace_id_matcher is None or self.allowed_namespace_id_matcher(namespace_id)

    def _is_starting_event(self, event):
        return (super(NamespaceCtlManager, self)._is_starting_event(event) and
                self._is_allowed_namespace(event.pb.meta.id) and
                has_actionable_spec(event.pb)
                )

    def _list_all_ctl_ids(self):
        """
        :rtype: list[six.text_type]
        """
        for namespace_pb in self._cache.list_all_namespaces():
            namespace_id = namespace_pb.meta.id
            if self._is_allowed_namespace(namespace_id) and has_actionable_spec(namespace_pb):
                yield namespace_id

    def _create_ctl(self, ctl_id):
        """
        :type ctl_id: six.text_type
        """
        alerting_cfg = self.get_alerting_cfg()  # sanity check for dynamic update from ITS
        return NamespaceCtl(ctl_id, alerting_cfg)

    def _get_ctl_id_from_event(self, event):
        """
        :rtype: six.text_type
        """
        ctl_id = event.path.strip('/')
        if not ctl_id:
            raise AssertionError('"{}" can not be parsed into a correct namespace ctl id'.format(event.path))
        return ctl_id


class NamespaceOrderCtlManagerV2(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(NamespaceOrderCtlManagerV2, self).__init__(
            coord=coord,
            cache=cache,
            entity_name=u'namespace-order',
            subscribed_events=(events.NamespaceUpdate, events.NamespaceRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.NamespaceRemove):
            return False
        return is_order_in_progress(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            yield events.NamespaceUpdate(
                path=self._make_event_path_from_ctl_id(namespace_pb.meta.id),
                pb=namespace_pb)

    def _get_ctl_human_readable_name(self, ctl_id):
        return '{}("{}") ctl'.format(self._entity_name, ctl_id)

    def _get_ctl_id_from_event(self, event):
        ctl_id = event.path.strip('/')
        if not ctl_id:
            raise AssertionError('"{}" can not be parsed into a correct {} id'.format(event.path, self._entity_name))
        return ctl_id

    def _create_ctl(self, ctl_id):
        return NamespaceOrderCtl(ctl_id)


class NamespaceCtlManagerV2(ctlmanager.CtlManagerV2):
    def __init__(self, coord, cache, allowed_namespace_id_matcher=None):
        """
        :type coord: awacs.lib.zookeeper_client.ZookeeperClient
        :type cache: awacs.model.cache.AwacsCache
        :type allowed_namespace_id_matcher: callable | None
        """
        super(NamespaceCtlManagerV2, self).__init__(
            coord=coord,
            cache=cache,
            entity_name=u'namespace',
            subscribed_events=(events.NamespaceUpdate, events.NamespaceRemove),
            allowed_namespace_id_matcher=allowed_namespace_id_matcher,
        )

    def _get_ctl_human_readable_name(self, ctl_id):
        return u'{}("{}") ctl'.format(self._entity_name, ctl_id)

    def _get_ctl_id_from_event(self, event):
        ctl_id = event.path.strip('/')
        if not ctl_id:
            raise AssertionError(u'"{}" can not be parsed into a correct {} id'.format(event.path, self._entity_name))
        return ctl_id

    def _should_ctl_be_running(self, event):
        if isinstance(event, events.NamespaceRemove):
            return False
        return has_actionable_spec(event.pb)

    def _yield_starting_events(self):
        for namespace_pb in self._cache.list_all_namespaces():
            yield events.NamespaceUpdate(
                path=self._make_event_path_from_ctl_id(namespace_pb.meta.id),
                pb=namespace_pb)

    def _create_ctl(self, ctl_id):
        """
        :type ctl_id: six.text_type
        """
        alerting_cfg = NamespaceCtlManager.get_alerting_cfg()  # sanity check for dynamic update from ITS
        return NamespaceCtl(ctl_id, alerting_cfg)


class NamespaceOperationCtlManager(ModelCtlManager):
    model = objects.NamespaceOperation

    def should_ctl_be_running(self, full_uid, model, pb=None):
        if not pb:
            return False
        return is_order_in_progress(pb) or needs_removal(pb)

    def create_ctl(self, full_uid):
        return NamespaceOperationCtl(full_uid=full_uid, cache=self.cache)

    def should_restart_ctl(self, full_uid, model, pb=None):
        return False
