from awacs.lib.models.controllers import ModelCtl, Subscription
from awacs.lib.order_processor.model import is_spec_complete, needs_removal
from awacs.lib.order_processor.runner import StateRunner
from awacs.model import objects
from awacs.model.namespace.operations.op_add_ip_address_to_l3_balancer import AddIpToL3BalancerOp
from awacs.model.namespace.operations.op_import_vs_from_l3mgr import ImportVsFromL3MgrOp


class NamespaceOperationCtl(ModelCtl):
    model = objects.NamespaceOperation
    subscriptions = {
        objects.NamespaceOperation: Subscription(immediate=True)
    }

    runners = {
        objects.NamespaceOperation.ADD_IP_TO_L3: StateRunner(
            entity_class=AddIpToL3BalancerOp,
            initial_state=AddIpToL3BalancerOp.states.REQUESTING_IP_ADDRESS,
            final_state=AddIpToL3BalancerOp.states.FINISHED,
            final_cancelled_state=AddIpToL3BalancerOp.states.CANCELLED,
            processors=AddIpToL3BalancerOp.get_processors(),
            processing_interval=3
        ),
        objects.NamespaceOperation.IMPORT_VS_FROM_L3MGR: StateRunner(
            entity_class=ImportVsFromL3MgrOp,
            initial_state=ImportVsFromL3MgrOp.states.LOCKING_L3MGR_SERVICE,
            final_state=ImportVsFromL3MgrOp.states.FINISHED,
            final_cancelled_state=ImportVsFromL3MgrOp.states.CANCELLED,
            processors=ImportVsFromL3MgrOp.get_processors(),
            processing_interval=3
        )
    }

    def should_process(self, full_uid, model, pb):
        return not is_spec_complete(pb)

    def process(self, ctx):
        """
        :type ctx: context.OpCtx
        """
        op_pb = objects.NamespaceOperation.cache.must_get(*self.full_uid)
        if needs_removal(op_pb):
            return self._self_delete(ctx, op_pb)
        op_name = objects.NamespaceOperation.get_operation_name(op_pb.order.content)
        return self.runners[op_name].process(ctx, op_pb)

    @staticmethod
    def _self_delete(ctx, op_pb):
        """
        :type ctx: context.OpCtx
        :type op_pb: model_pb2.NamespaceOperation
        """
        ctx.log.info(u'started self deletion')
        objects.NamespaceOperation.remove(op_pb)
