import enum
import inject
from boltons import strutils
from sepelib.core import config as appconfig
from six.moves import range  # noqa

from awacs.lib import l3mgrclient
from awacs.lib.order_processor.model import WithOrder, FeedbackMessage, BaseStandaloneProcessor, Result
from awacs.lib.strutils import quote_join_sorted
from awacs.lib.vectors import version
from awacs.model import zk, objects, cache, dao
from awacs.model.l3_balancer import l3mgr, l3_balancer
from awacs.model.l3_balancer.order import processors as order_processors
from awacs.model.util import NANNY_ROBOT_LOGIN
from infra.awacs.proto import model_pb2


class State(enum.Enum):
    REQUESTING_IP_ADDRESS = 1
    UPDATING_L7_CONTAINER_SPEC = 2
    UPDATING_L3_SPEC = 3
    WAITING_FOR_L3_ACTIVATION = 4
    FINISHED = 5
    CANCELLING = 6
    CANCELLED = 7


class AddIpToL3BalancerOp(WithOrder):
    __slots__ = ()

    name = strutils.under2camel(objects.NamespaceOperation.ADD_IP_TO_L3)
    states = State

    state_descriptions = {
        State.REQUESTING_IP_ADDRESS: u'Requesting IP address',
        State.UPDATING_L7_CONTAINER_SPEC: u'Updating tunnels in L7 balancers',
        State.UPDATING_L3_SPEC: u'Adding IP address to L3 balancer',
        State.WAITING_FOR_L3_ACTIVATION: u'Waiting for L3 balancer to finish deploying',
    }

    def zk_update(self):
        return objects.NamespaceOperation.zk.update(namespace_id=self.namespace_id,
                                                    op_id=self.id,
                                                    pb=self.pb)

    @property
    def l3_balancer_id(self):
        return self.pb.order.content.add_ip_address_to_l3_balancer.l3_balancer_id

    @staticmethod
    def get_processors():
        return (
            RequestingIpAddress,
            UpdatingL7ContainerSpec,
            UpdatingL3Spec,
            WaitingForL3Activation,
            Cancelling,
        )


class AddIpToL3BalancerProcessor(BaseStandaloneProcessor):
    state = None
    next_state = None
    cancelled_state = None

    @classmethod
    def process(cls, ctx, operation):
        """
        :type ctx: context.OpCtx
        :type operation: AddIpToL3BalancerOp
        :rtype: State | FeedbackMessage
        """
        raise NotImplementedError


class RequestingIpAddress(AddIpToL3BalancerProcessor):
    state = State.REQUESTING_IP_ADDRESS
    next_state = State.UPDATING_L7_CONTAINER_SPEC
    next_state_skip_tunnels = State.UPDATING_L3_SPEC
    cancelled_state = State.CANCELLING

    _l3mgr_client = inject.attr(l3mgrclient.IL3MgrClient)  # type: l3mgrclient.L3MgrClient
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    @classmethod
    def process(cls, ctx, operation):
        l3_pb = cls._cache.must_get_l3_balancer(operation.namespace_id, operation.l3_balancer_id)
        svc_id = l3_pb.spec.l3mgr_service_id
        vs_order_pb = operation.pb.order.content.add_ip_address_to_l3_balancer
        ip = l3mgr.Service.from_api(cls._l3mgr_client, svc_id).get_new_ip(
            cls._l3mgr_client,
            is_v4=vs_order_pb.ip_version == vs_order_pb.VER_IPV4,
            is_external=vs_order_pb.traffic_type == model_pb2.L3BalancerSpec.VirtualServer.TT_EXTERNAL)
        # IP addresses that are not used in any VS can be reclaimed by L3mgr, so we create a temporary VS
        vs_pbs = l3_balancer.make_vs_spec_pbs_from_vs_order_pb(vs_order_pb, ip)
        if not vs_pbs:
            raise RuntimeError
        vs = l3mgr.VirtualServer.from_vs_pb(svc_id, vs_pbs[0], l3mgr.RSGroup())
        operation.context[u'temporary_vs_id'] = vs.create_in_l3mgr(cls._l3mgr_client).id
        operation.context[u'ip_address'] = ip

        if appconfig.get_value(u'run.disable_l7_tunnels_autoconfig_on_l3_update', default=False):
            return operation.make_result(cls.next_state_skip_tunnels)
        ns_pb = cls._cache.must_get_namespace(operation.namespace_id)
        if ns_pb.spec.easy_mode_settings.disable_l7_tunnels_autoconfig.value:
            return operation.make_result(cls.next_state_skip_tunnels)

        return operation.make_result(cls.next_state)


class UpdatingL7ContainerSpec(AddIpToL3BalancerProcessor):
    state = State.UPDATING_L7_CONTAINER_SPEC
    next_state = State.UPDATING_L3_SPEC
    cancelled_state = State.CANCELLING

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _dao = inject.attr(dao.IDao)  # type: dao.Dao
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage

    @classmethod
    def get_ip_addresses(cls, operation, l3_balancer_pb):
        l3_ips = set(vs_pb.ip for vs_pb in l3_balancer_pb.spec.virtual_servers)
        l3_ips.add(operation.context[u'ip_address'])
        return l3_ips

    @classmethod
    def update_next_l7_balancer(cls, ctx, operation, l3_balancer_pb, ip_addresses):
        processed_locations = set(operation.context.get(u'processed_locations', []))
        in_progress_locations, balancer_to_process = order_processors.UpdatingL7ContainerSpec.get_l7_balancers_status(
            ctx, l3_balancer_pb.meta.namespace_id,
            (l7.id for l7 in l3_balancer_pb.spec.real_servers.balancers),
            processed_locations)

        if in_progress_locations:
            # even if we updated all specs, we need to wait until all L7 balancers have finished deploying
            desc = u'Configuring tunnels in L7 {} in {} "{}"'.format(
                strutils.cardinalize(u'balancer', len(in_progress_locations)),
                strutils.cardinalize(u'location', len(in_progress_locations)),
                quote_join_sorted(in_progress_locations))
            return Result(
                description=desc,
                content_pb=model_pb2.NamespaceOperationOrder.OrderFeedback.L7LocationsInProgress(
                    locations=sorted(in_progress_locations)
                ),
                next_state=cls.state
            )
        if not balancer_to_process:
            # all done
            return None

        location, l7_balancer_pb = balancer_to_process
        order_processors.UpdatingL7ContainerSpec.update_container_spec(
            ctx, l3_balancer_pb=l3_balancer_pb, l7_balancer_pb=l7_balancer_pb, ip_addresses=ip_addresses)
        processed_locations.add(location)
        operation.context[u'processed_locations'] = sorted(processed_locations)
        return operation.make_result(cls.state)

    @classmethod
    def process(cls, ctx, operation):
        l3_balancer_pb = cls._zk.must_get_l3_balancer(operation.namespace_id, operation.l3_balancer_id)
        result = cls.update_next_l7_balancer(ctx, operation, l3_balancer_pb,
                                             ip_addresses=cls.get_ip_addresses(operation, l3_balancer_pb))
        if result is None:
            # all done
            return operation.make_result(next_state=cls.next_state)
        else:
            # ready to process next balancer (or wait until it's idle)
            return result


class UpdatingL3Spec(AddIpToL3BalancerProcessor):
    state = State.UPDATING_L3_SPEC
    next_state = State.WAITING_FOR_L3_ACTIVATION
    cancelled_state = State.CANCELLING

    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _dao = inject.attr(dao.IDao)  # type: dao.Dao
    _l3mgr_client = inject.attr(l3mgrclient.IL3MgrClient)  # type: l3mgrclient.L3MgrClient

    @classmethod
    def process(cls, ctx, operation):
        l3_balancer_pb = cls._zk.must_get_l3_balancer(operation.namespace_id, operation.l3_balancer_id)
        ip = operation.context[u'ip_address']
        vs_order_pb = operation.pb.order.content.add_ip_address_to_l3_balancer
        for vs_pb in l3_balancer.make_vs_spec_pbs_from_vs_order_pb(vs_order_pb, ip):
            spec_vs_pb = l3_balancer_pb.spec.virtual_servers.add()
            spec_vs_pb.CopyFrom(vs_pb)
        cls._dao.update_l3_balancer(operation.namespace_id,
                                    operation.l3_balancer_id,
                                    comment=u'Added virtual servers for IP {}'.format(ip),
                                    login=NANNY_ROBOT_LOGIN,
                                    version=l3_balancer_pb.meta.version,
                                    updated_spec_pb=l3_balancer_pb.spec)
        return operation.make_result(cls.next_state)


class WaitingForL3Activation(AddIpToL3BalancerProcessor):
    state = State.WAITING_FOR_L3_ACTIVATION
    next_state = State.FINISHED
    cancelled_state = State.CANCELLING

    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    @classmethod
    def is_l3_updated(cls, ctx, operation):
        l3_balancer_pb = cls._zk.must_get_l3_balancer(operation.namespace_id, operation.l3_balancer_id)
        current_ver = version.L3BalancerVersion.from_pb(l3_balancer_pb)
        l3_balancer_state_pb = cls._cache.get_l3_balancer_state(operation.namespace_id, operation.l3_balancer_id)
        if l3_balancer_state_pb is None:
            return False
        vectors = l3_balancer.L3BalancerStateHandler(l3_balancer_state_pb).generate_vectors()
        ctx.log.info(u'current_ver: %s', current_ver)
        ctx.log.info(u'vectors.active.l3_balancer_version: %s', vectors.active.l3_balancer_version)
        ctx.log.info(u'rv: %s', vectors.active.l3_balancer_version >= current_ver)
        return vectors.active.l3_balancer_version >= current_ver

    @classmethod
    def process(cls, ctx, operation):
        if not cls.is_l3_updated(ctx, operation):
            return operation.make_result(cls.state)
        for op_pb in objects.NamespaceOperation.zk.update(operation.namespace_id, operation.id):
            op_pb.spec.incomplete = False
        return operation.make_result(cls.next_state)


class Cancelling(AddIpToL3BalancerProcessor):
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    @classmethod
    def process(cls, ctx, operation):
        for op_pb in objects.NamespaceOperation.zk.update(operation.namespace_id, operation.id):
            op_pb.spec.incomplete = False
        # TODO: clean up
        return operation.make_result(cls.next_state)
