import enum
import inject
import six
from boltons import strutils

from awacs.lib import l3mgrclient, racktables
from awacs.lib.order_processor.model import WithOrder, FeedbackMessage, BaseStandaloneProcessor
from awacs.model import zk, objects, cache, dao
from awacs.model.l3_balancer import l3mgr, l3_balancer
from awacs.model.namespace.operations import op_add_ip_address_to_l3_balancer
from awacs.model.util import NANNY_ROBOT_LOGIN
from infra.awacs.proto import model_pb2


class State(enum.Enum):
    LOCKING_L3MGR_SERVICE = 1
    UPDATING_L7_CONTAINER_SPEC = 2
    UPDATING_L3_SPEC = 3
    WAITING_FOR_L3_ACTIVATION = 4
    FINISHED = 5
    CANCELLING = 6
    CANCELLED = 7


class ImportVsFromL3MgrOp(WithOrder):
    __slots__ = ()

    name = strutils.under2camel(objects.NamespaceOperation.IMPORT_VS_FROM_L3MGR)
    states = State

    state_descriptions = {
        State.LOCKING_L3MGR_SERVICE: u'Locking L3 balancer in L3 Manager',
        State.UPDATING_L7_CONTAINER_SPEC: u'Updating tunnels in L7 balancers',
        State.UPDATING_L3_SPEC: u'Adding virtual servers to L3 balancer',
        State.WAITING_FOR_L3_ACTIVATION: u'Waiting for L3 balancer to finish deploying',
    }

    def zk_update(self):
        return objects.NamespaceOperation.zk.update(namespace_id=self.namespace_id,
                                                    op_id=self.id,
                                                    pb=self.pb)

    @property
    def l3_balancer_id(self):
        return self.pb.order.content.import_virtual_servers_from_l3mgr.l3_balancer_id

    @property
    def enforce_configs(self):
        return self.pb.order.content.import_virtual_servers_from_l3mgr.enforce_configs

    @staticmethod
    def get_processors():
        return (
            LockingL3MgrService,
            UpdatingL7ContainerSpec,
            UpdatingL3Spec,
            WaitingForL3Activation,
            Cancelling,
        )


class ImportVsFromL3MgrProcessor(BaseStandaloneProcessor):
    state = None
    next_state = None
    cancelled_state = None

    @classmethod
    def process(cls, ctx, operation):
        """
        :type ctx: context.OpCtx
        :type operation: ImportVsFromL3MgrOp
        :rtype: State | FeedbackMessage
        """
        raise NotImplementedError


class LockingL3MgrService(ImportVsFromL3MgrProcessor):
    state = State.LOCKING_L3MGR_SERVICE
    next_state = State.UPDATING_L7_CONTAINER_SPEC
    cancelled_state = State.CANCELLING

    _l3mgr_client = inject.attr(l3mgrclient.IL3MgrClient)  # type: l3mgrclient.L3MgrClient
    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    @classmethod
    def process(cls, _, operation):
        l3_balancer_pb = cls._cache.must_get_l3_balancer(operation.namespace_id, operation.l3_balancer_id)
        meta = l3_balancer.make_l3mgr_balancer_meta(operation.namespace_id, operation.l3_balancer_id, locked=True)
        cls._l3mgr_client.update_meta(svc_id=l3_balancer_pb.spec.l3mgr_service_id, data=meta)
        return operation.make_result(cls.next_state)


class UpdatingL7ContainerSpec(ImportVsFromL3MgrProcessor):
    state = State.UPDATING_L7_CONTAINER_SPEC
    next_state = State.UPDATING_L3_SPEC
    cancelled_state = State.CANCELLING

    _cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache
    _dao = inject.attr(dao.IDao)  # type: dao.Dao
    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _l3mgr_client = inject.attr(l3mgrclient.IL3MgrClient)  # type: l3mgrclient.L3MgrClient

    @classmethod
    def get_ip_addresses(cls, operation):
        l3_balancer_pb = cls._cache.must_get_l3_balancer(operation.namespace_id, operation.l3_balancer_id)
        svc_id = l3_balancer_pb.spec.l3mgr_service_id
        latest_config = l3mgr.ServiceConfig.latest_from_api(cls._l3mgr_client, svc_id)
        rv = set()
        for vs_id in latest_config.vs_ids:
            vs = l3mgr.VirtualServer.from_api(cls._l3mgr_client, svc_id, vs_id)
            rv.add(vs.ip)
        return rv

    @classmethod
    def process(cls, ctx, operation):
        l3_balancer_pb = cls._zk.must_get_l3_balancer(operation.namespace_id, operation.l3_balancer_id)
        result = op_add_ip_address_to_l3_balancer.UpdatingL7ContainerSpec.update_next_l7_balancer(
            ctx, operation, l3_balancer_pb, ip_addresses=cls.get_ip_addresses(operation))
        if result is None:
            # all done
            return operation.make_result(next_state=cls.next_state)
        else:
            # ready to process next balancer (or wait until it's idle)
            return result.translate_state(State)


class UpdatingL3Spec(ImportVsFromL3MgrProcessor):
    state = State.UPDATING_L3_SPEC
    next_state = State.WAITING_FOR_L3_ACTIVATION
    cancelled_state = None

    _zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    _dao = inject.attr(dao.IDao)  # type: dao.Dao
    _l3mgr_client = inject.attr(l3mgrclient.IL3MgrClient)  # type: l3mgrclient.L3MgrClient
    _rt_client = inject.attr(racktables.IRacktablesClient)  # type: racktables.RacktablesClient

    @classmethod
    def import_vs_from_l3mgr(cls, virtual_servers, l3_balancer_pb):
        del l3_balancer_pb.spec.virtual_servers[:]  # avoid duplication in case of partial/incorrect migrations
        for vs in six.itervalues(virtual_servers.virtual_servers):
            if cls._rt_client.is_public_ip(vs.ip):  # temporary solution until TRAFFIC-12333
                traffic_type = model_pb2.L3BalancerSpec.VirtualServer.TT_EXTERNAL
            else:
                traffic_type = model_pb2.L3BalancerSpec.VirtualServer.TT_INTERNAL
            if vs.port == 443:
                vs.config[l3mgr.VSConfigKey.check_type] = u'SSL_GET'
            else:
                vs.config[l3mgr.VSConfigKey.check_type] = u'HTTP_GET'
            vs_pb = l3_balancer_pb.spec.virtual_servers.add()
            vs_pb.CopyFrom(vs.to_pb(traffic_type))
        return l3_balancer_pb

    @classmethod
    def process(cls, ctx, operation):
        l3_balancer_pb = cls._zk.must_get_l3_balancer(operation.namespace_id, operation.l3_balancer_id)
        svc_id = l3_balancer_pb.spec.l3mgr_service_id

        latest_config = l3mgr.ServiceConfig.latest_from_api(cls._l3mgr_client, svc_id)
        virtual_servers = l3mgr.VirtualServers.from_api(cls._l3mgr_client, svc_id, latest_config.vs_ids,
                                                        allow_l3mgr_configs_mismatch=operation.enforce_configs)
        l3_balancer_pb = cls.import_vs_from_l3mgr(virtual_servers, l3_balancer_pb)
        l3_balancer_pb.spec.config_management_mode = model_pb2.L3BalancerSpec.MODE_REAL_AND_VIRTUAL_SERVERS
        l3_balancer_pb.spec.ctl_version = 2
        l3_balancer_pb.spec.enforce_configs = operation.enforce_configs
        cls._dao.update_l3_balancer(operation.namespace_id,
                                    operation.l3_balancer_id,
                                    comment=u'Imported virtual servers from L3 Manager',
                                    login=NANNY_ROBOT_LOGIN,
                                    version=l3_balancer_pb.meta.version,
                                    updated_spec_pb=l3_balancer_pb.spec)
        return operation.make_result(cls.next_state)


class WaitingForL3Activation(ImportVsFromL3MgrProcessor):
    state = State.WAITING_FOR_L3_ACTIVATION
    next_state = State.FINISHED
    cancelled_state = None

    @classmethod
    def process(cls, ctx, operation):
        if not op_add_ip_address_to_l3_balancer.WaitingForL3Activation.is_l3_updated(ctx, operation):
            return operation.make_result(cls.state)
        for op_pb in objects.NamespaceOperation.zk.update(operation.namespace_id, operation.id):
            op_pb.spec.incomplete = False
        return operation.make_result(cls.next_state)


class Cancelling(ImportVsFromL3MgrProcessor):
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    @classmethod
    def process(cls, ctx, operation):
        for op_pb in objects.NamespaceOperation.zk.update(operation.namespace_id, operation.id):
            op_pb.spec.incomplete = False
        # TODO: clean up
        return operation.make_result(cls.next_state)
