import enum
import inject
import six
from abc import ABCMeta
from sepelib.core import config

from awacs.lib import nannyclient, staffclient
from awacs.lib.order_processor.model import BaseProcessor, WithOrder, cancel_order, is_spec_complete
from awacs.lib.vectors.cacheutil import get_backend_versions, get_endpoint_set_versions
from awacs.lib.yamlparser.wrappers_util import dump_uem_pb
from awacs.lib.strutils import flatten_full_id2
from awacs.model import dao, zk, util, cache, alerting
from awacs.model.balancer.discoverer import NamespaceProxy as BalancerNamespaceProxy
from awacs.model.balancer.order import util as balancer_util
from awacs.model.balancer.stateholder import BalancerStateHolder
from awacs.model.balancer.vector import BackendVersion
from awacs.model.balancer.generator import get_included_full_backend_ids_from_holder
from awacs.model.dns_records.dns_record import get_dns_record_versions, DnsRecordStateHandler
from awacs.model.errors import NotFoundError
from awacs.model.validation import validate_and_parse_yaml_upstream_config
from infra.awacs.proto import model_pb2, modules_pb2
from awacs.web.validation.balancer import validate_balancer_location
from awacs.web.validation.namespace import apply_l3_only_layout_constraints, apply_external_layout_constraints
from awacs.wrappers import l7upstreammacro
from awacs.wrappers.base import ValidationCtx

DEFAULT_UPSTREAM_ID = 'default'

DEFAULT_UPSTREAM_YML = '''l7_upstream_macro:
  version: {}
  id: default
  matcher:
    any: true
  flat_scheme:
    balancer:
      attempts: 2
      fast_attempts: 2
      max_reattempts_share: 0.15
      max_pessimized_endpoints_share: 0.2
      retry_http_responses:
        codes: [5xx]
      backend_timeout: 10s
      connect_timeout: 70ms
    backend_ids: [%s]
    on_error:
      static:
        status: 504
        content: "Service unavailable"'''.format(l7upstreammacro.LATEST_VERSION)


class State(enum.Enum):
    START = 1
    VALIDATING_AWACS_NAMESPACE = 8
    ACTIVATING_NANNY_SERVICES = 9
    FINISHED = 10
    CREATING_DNS_RECORD = 13
    VALIDATING_DNS_RECORD = 14
    CREATING_BALANCERS = 16
    WAITING_FOR_BALANCERS = 17
    FINALIZING = 19
    ENABLING_NAMESPACE_ALERTING = 20
    CREATING_UPSTREAMS = 21
    CREATING_BACKENDS = 22
    CREATING_CERTS = 23
    WAITING_FOR_CERTS = 24
    UNPAUSING_BALANCER_CONFIG_UPDATES = 25
    WAITING_FOR_BALANCERS_TO_BE_IN_PROGRESS = 26
    WAITING_FOR_BALANCERS_TO_ALLOCATE = 27
    CREATING_DOMAINS = 28
    WAITING_FOR_DOMAINS = 29
    CANCELLING = 30
    CANCELLED = 40


def get_namespace_order_processors():
    return (
        Start,
        CreatingBalancers,
        WaitingForBalancersToAllocate,
        WaitingForBalancers,
        CreatingBackends,
        CreatingUpstreams,
        CreatingCerts,
        CreatingDomains,
        WaitingForDomains,
        WaitingForCerts,
        ValidatingNamespace,
        UnpausingBalancerConfigUpdates,
        WaitingForBalancersToBeInProgress,
        ActivatingNannyServices,
        CreatingDnsRecord,
        ValidatingDnsRecord,
        EnablingNamespaceAlerting,
        Finalizing,
        Cancelling,
    )


class OrderType(enum.Enum):
    EASY_MODE = 0
    EASY_MODE_WITH_DOMAINS = 1
    EMPTY = 2


class NamespaceOrder(WithOrder):
    __slots__ = ()

    zk = inject.attr(zk.IZkStorage)  # type: zk.ZkStorage
    dao = inject.attr(dao.IDao)  # type: dao.Dao
    name = u'Namespace'
    states = State

    namespace_id = None

    def zk_update(self):
        return self.zk.update_namespace(namespace_id=self.id)

    def dao_update(self, updated_spec_pb, updated_annotations=None, comment=u'Updated namespace spec'):
        return self.dao.update_namespace(
            namespace_id=self.id,
            version=self.pb.meta.version,
            login=util.NANNY_ROBOT_LOGIN,
            comment=comment,
            updated_spec_pb=updated_spec_pb,
            updated_annotations=updated_annotations,
        )

    @property
    def order_type(self):
        if self.pb.order.content.flow_type == model_pb2.NamespaceOrder.Content.QUICK_START:
            return OrderType.EASY_MODE_WITH_DOMAINS
        if self.pb.order.content.flow_type == model_pb2.NamespaceOrder.Content.YP_LITE:
            return OrderType.EASY_MODE
        if self.pb.order.content.flow_type == model_pb2.NamespaceOrder.Content.EMPTY:
            return OrderType.EMPTY
        raise RuntimeError(u'unexpected order type')


class NamespaceOrderProcessor(six.with_metaclass(ABCMeta, BaseProcessor)):
    __slots__ = (u'meta_pb', u'order_pb',)

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def __init__(self, entity):
        super(NamespaceOrderProcessor, self).__init__(entity)
        self.entity = self.entity  # type: NamespaceOrder
        self.meta_pb = self.entity.pb.meta  # type: model_pb2.NamespaceMeta
        self.order_pb = self.entity.pb.order.content  # type: model_pb2.NamespaceOrder.Content

    def _set_auth(self, meta_pb):
        meta_pb.auth.type = meta_pb.auth.STAFF
        meta_pb.auth.staff.owners.logins.extend(self.meta_pb.auth.staff.owners.logins)
        meta_pb.auth.staff.owners.group_ids.extend(self.meta_pb.auth.staff.owners.group_ids)


class Start(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.START
    next_state = State.CREATING_BALANCERS
    cancelled_state = State.CANCELLING

    def process(self, ctx):
        if self.order_pb.flow_type == self.order_pb.GENCFG:
            raise RuntimeError(u'Gencfg balancers are not supported')

        if self.order_pb.flow_type not in (
            self.order_pb.YP_LITE,
            self.order_pb.QUICK_START,
            self.order_pb.EMPTY,
        ):
            raise RuntimeError(u'unexpected flow_type')
        if self.order_pb.flow_type == self.order_pb.QUICK_START:
            if not self.order_pb.certificate_order_content.common_name:
                raise RuntimeError(u'QUICK_START namespace order must contain cert order')
        if not self.order_pb.alerting_simple_settings.notify_staff_group_id:
            raise RuntimeError(u'"notify_staff_group_id" is required to enable alerting')
        return self.next_state


class CreatingBalancers(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.CREATING_BALANCERS
    next_state = State.WAITING_FOR_BALANCERS_TO_ALLOCATE
    cancelled_state = State.CANCELLING

    def _make_meta_pb_template(self):
        meta_pb_template = model_pb2.BalancerMeta(
            namespace_id=self.entity.id,
        )
        self._set_auth(meta_pb_template)
        meta_pb_template.transport_paused.value = True
        meta_pb_template.transport_paused.mtime.GetCurrentTime()
        meta_pb_template.transport_paused.comment = u'Paused until namespace is ready'
        meta_pb_template.transport_paused.author = util.NANNY_ROBOT_LOGIN
        return meta_pb_template

    def _make_meta_pb(self, meta_pb_template, location):
        meta_pb = model_pb2.BalancerMeta()
        meta_pb.CopyFrom(meta_pb_template)
        meta_pb.id = balancer_util.make_awacs_balancer_id(self.entity.id, location)
        if self.entity.pb.order.content.cloud_type == model_pb2.CT_AZURE:
            meta_pb.location.type = meta_pb.location.AZURE_CLUSTER
            meta_pb.location.azure_cluster = location.upper()
        else:
            meta_pb.location.type = meta_pb.location.YP_CLUSTER
            meta_pb.location.yp_cluster = location.upper()
        return meta_pb

    def _make_order_content_pb_template(self):
        pb_template = model_pb2.BalancerOrder.Content()
        if self.entity.order_type == OrderType.EASY_MODE:
            pb_template.mode = pb_template.YP_LITE
        elif self.entity.order_type == OrderType.EASY_MODE_WITH_DOMAINS:
            pb_template.mode = pb_template.EASY_MODE_WITH_DOMAINS
        pb_template.cloud_type = self.order_pb.cloud_type
        pb_template.resource_group_label = self.order_pb.resource_group_label

        pb_template.abc_service_id = self.meta_pb.abc_service_id
        pb_template.cert_id = self.order_pb.certificate_order_content.common_name

        pb_template.activate_balancer = False
        pb_template.wait_for_approval_after_allocation = True
        pb_template.instance_tags.CopyFrom(self.order_pb.instance_tags)
        return pb_template

    def _make_order_content_pb(self, order_content_pb_template, location):
        order_content_pb = model_pb2.BalancerOrder.Content()
        order_content_pb.CopyFrom(order_content_pb_template)
        req = order_content_pb.allocation_request
        req.location = location
        ns_allocation_pb = self.order_pb.yp_lite_allocation_request
        req.nanny_service_id_slug = ns_allocation_pb.nanny_service_id_slug
        req.network_macro = ns_allocation_pb.network_macro
        req.type = ns_allocation_pb.type
        req.preset.type = ns_allocation_pb.preset.type
        req.preset.instances_count = ns_allocation_pb.preset.instances_count
        order_content_pb.do_not_create_user_endpoint_set = self.order_pb.do_not_create_user_endpoint_sets
        return order_content_pb

    def _create_balancers(self):
        order_content_pb_template = self._make_order_content_pb_template()
        meta_pb_template = self._make_meta_pb_template()
        for location in self.order_pb.yp_lite_allocation_request.locations:
            meta_pb = self._make_meta_pb(meta_pb_template, location)
            validate_balancer_location(meta_pb.location, meta_pb.id)
            order_content_pb = self._make_order_content_pb(order_content_pb_template, location)

            rev_index_pb = model_pb2.RevisionGraphIndex()
            included_backend_ids = []
            for backend_id in self.order_pb.backends:
                full_id = (meta_pb.namespace_id, backend_id)
                included_backend_ids.append(flatten_full_id2(full_id))
            rev_index_pb.included_backend_ids.extend(included_backend_ids)

            self.entity.dao.create_balancer_if_missing(meta_pb, order_content_pb, rev_index_pb=rev_index_pb)
            self.entity.context[u'completed_balancer_ids'][meta_pb.id] = False

    def process(self, ctx):
        self.entity.context[u'completed_balancer_ids'] = {}
        if self.entity.order_type != OrderType.EMPTY:
            self._create_balancers()

        return self.next_state


class WaitingForBalancersToAllocate(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_BALANCERS_TO_ALLOCATE
    next_state = State.WAITING_FOR_BALANCERS
    cancelled_state = State.CANCELLING

    def _wait_for_balancers_to_allocate(self):
        for balancer_id in six.iterkeys(self.entity.context[u'completed_balancer_ids']):
            balancer_pb = self.cache.get_balancer(self.entity.id, balancer_id)
            if not balancer_pb:
                return self.state
            if balancer_pb.order.approval.after_allocation:
                continue
            balancer_needs_approval = balancer_pb.order.content.wait_for_approval_after_allocation
            balancer_state = balancer_pb.order.progress.state.id
            if balancer_needs_approval and balancer_state != u'WAITING_FOR_APPROVAL_AFTER_ALLOCATION':
                return self.state
        for balancer_id in six.iterkeys(self.entity.context[u'completed_balancer_ids']):
            balancer_pb = self.cache.get_balancer(self.entity.id, balancer_id)
            if balancer_pb.order.approval.after_allocation:
                continue
            if balancer_pb.order.content.wait_for_approval_after_allocation:
                approval_pb = model_pb2.BalancerOrder.Approval()
                approval_pb.after_allocation = True
                self.entity.dao.update_balancer(
                    namespace_id=self.entity.id,
                    balancer_id=balancer_id,
                    version=balancer_pb.meta.version,
                    login=util.NANNY_ROBOT_LOGIN,
                    comment='Approved progress after allocation',
                    updated_approval_pb=approval_pb,
                )

        return self.next_state

    def process(self, ctx):
        if self.entity.order_type != OrderType.EMPTY:
            return self._wait_for_balancers_to_allocate()

        return self.next_state


class WaitingForBalancers(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_BALANCERS
    next_state = State.CREATING_BACKENDS
    cancelled_state = None

    def _wait_for_balancers(self):
        all_ready = True
        for balancer_id in six.iterkeys(self.entity.context[u'completed_balancer_ids']):
            if self.entity.context[u'completed_balancer_ids'][balancer_id]:
                continue
            balancer_pb = self.cache.get_balancer(self.entity.id, balancer_id)
            if balancer_pb and is_spec_complete(balancer_pb):
                self.entity.context[u'completed_balancer_ids'][balancer_id] = True
            else:
                all_ready = False
        if not all_ready:
            return self.state
        return self.next_state

    def process(self, ctx):
        if self.entity.order_type != OrderType.EMPTY:
            return self._wait_for_balancers()
        return self.next_state


class CreatingBackends(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.CREATING_BACKENDS
    next_state = State.CREATING_UPSTREAMS
    cancelled_state = None

    def _create_backends(self):
        for i, (backend_id, selector_pb) in enumerate(sorted(self.order_pb.backends.items())):
            backend_meta_pb = model_pb2.BackendMeta(
                id=backend_id,
                namespace_id=self.entity.id
            )
            self._set_auth(backend_meta_pb)
            backend_spec_pb = model_pb2.BackendSpec()
            backend_spec_pb.selector.CopyFrom(selector_pb)
            backend_pb = self.entity.dao.create_backend_if_missing(backend_meta_pb, backend_spec_pb)
            self.entity.context[u'completed_backend_ids'][backend_id] = True

            if backend_pb.spec.selector.type == model_pb2.BackendSelector.MANUAL:
                endpoint_set_meta_pb = model_pb2.EndpointSetMeta(
                    id=backend_id,
                    namespace_id=self.entity.id
                )
                self._set_auth(endpoint_set_meta_pb)
                endpoint_set_meta_pb.backend_versions.append(backend_pb.meta.version)
                endpoint_set_spec_pb = model_pb2.EndpointSetSpec()
                endpoint_set_spec_pb.instances.add(
                    host=u'please-change-me',
                    port=31337 + i,
                    weight=1,
                    ipv6_addr=u'::1'
                )
                self.entity.dao.create_endpoint_set_if_missing(endpoint_set_meta_pb, endpoint_set_spec_pb)

    def _create_backends_for_domains(self):
        for endpoint_set in self.order_pb.endpoint_sets:
            cluster = endpoint_set.cluster.lower()
            es_id = endpoint_set.id
            backend_id = '{}_{}'.format(es_id, cluster).replace('.', '_').lower()
            meta_pb = model_pb2.BackendMeta(namespace_id=self.entity.id, id=backend_id)
            self._set_auth(meta_pb)

            spec_pb = model_pb2.BackendSpec()
            spec_pb.selector.type = model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD
            spec_pb.selector.yp_endpoint_sets.add(cluster=cluster, endpoint_set_id=es_id)

            self.entity.dao.create_backend_if_missing(meta_pb, spec_pb)
            self.entity.context[u'completed_backend_ids'][backend_id] = True

    def process(self, ctx):
        self.entity.context[u'completed_backend_ids'] = {}
        if self.entity.order_type == OrderType.EASY_MODE_WITH_DOMAINS:
            self._create_backends_for_domains()
        elif self.entity.order_type == OrderType.EASY_MODE:
            self._create_backends()
        return self.next_state


class CreatingUpstreams(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.CREATING_UPSTREAMS
    next_state = State.CREATING_CERTS
    next_state_creating_domains = State.CREATING_DOMAINS
    cancelled_state = None

    def _make_upstream_spec_for_domain(self, upstream_id):
        """
        :type upstream_id: six.text_type
        :rtype: model_pb2.UpstreamSpec
        """
        from awacs.wrappers import l7upstreammacro  # TODO: fix circular imports in wrappers/main.py

        spec_pb = model_pb2.UpstreamSpec(type=model_pb2.YANDEX_BALANCER)
        config_pb = spec_pb.yandex_balancer
        config_pb.mode = model_pb2.YandexBalancerUpstreamSpec.EASY_MODE2
        l7_um_pb = config_pb.config.l7_upstream_macro  # type: modules_pb2.L7UpstreamMacro
        l7_um_pb.id = upstream_id
        l7_um_pb.version = six.text_type(l7upstreammacro.LATEST_VERSION)
        l7_um_pb.monitoring.uuid = upstream_id
        l7_um_pb.matcher.path_re = '(/.*)?'

        dc_balancer = l7_um_pb.by_dc_scheme.dc_balancer
        dc_balancer.weights_section_id = 'bygeo'
        dc_balancer.method = modules_pb2.L7UpstreamMacro.DcBalancerSettings.LOCAL_THEN_BY_DC_WEIGHT

        balancer = l7_um_pb.by_dc_scheme.balancer
        balancer.do_not_retry_http_responses = True
        balancer.retry_non_idempotent.value = False
        balancer.max_reattempts_share = 0.15
        balancer.max_pessimized_endpoints_share = 0.2
        balancer.attempts = 2
        balancer.fast_attempts = 2
        balancer.connect_timeout = '70ms'
        balancer.backend_timeout = '10s'

        dcs = {}
        for endpoint_set in self.order_pb.endpoint_sets:
            dc_name = endpoint_set.cluster.lower()
            if dc_name in dcs:
                dc = dcs[dc_name]
            else:
                dcs[dc_name] = dc = l7_um_pb.by_dc_scheme.dcs.add(name=dc_name)
            backend_id = '{}_{}'.format(endpoint_set.id, endpoint_set.cluster).replace('.', '_').lower()
            dc.backend_ids.append(backend_id)
        dc_balancer.attempts = min(2, len(l7_um_pb.by_dc_scheme.dcs))

        static_on_error = l7_um_pb.by_dc_scheme.on_error.static
        static_on_error.status = 504
        static_on_error.content = 'Service unavailable'

        spec_pb.yandex_balancer.yaml = dump_uem_pb(spec_pb.yandex_balancer.config)
        spec_pb.labels[u'order'] = u'10000000'
        return spec_pb

    def _make_upstream_spec(self):
        spec_pb = model_pb2.UpstreamSpec()
        yaml_config = DEFAULT_UPSTREAM_YML % ', '.join(six.iterkeys(self.order_pb.backends))
        spec_pb.yandex_balancer.yaml = yaml_config
        spec_pb.yandex_balancer.mode = spec_pb.yandex_balancer.EASY_MODE2
        spec_pb.labels[u'order'] = u'99999999'
        return spec_pb

    def process(self, ctx):
        self.entity.context[u'completed_upstream_ids'] = {}

        if self.entity.order_type == OrderType.EMPTY:
            return self.next_state

        if self.entity.order_type == OrderType.EASY_MODE:
            upstream_id = DEFAULT_UPSTREAM_ID
            spec_pb = self._make_upstream_spec()
        else:
            upstream_id = self.order_pb.certificate_order_content.common_name.replace('.', '_').lower()
            spec_pb = self._make_upstream_spec_for_domain(upstream_id)

        meta_pb = model_pb2.UpstreamMeta(
            id=upstream_id,
            namespace_id=self.entity.id
        )
        self._set_auth(meta_pb)

        validation_ctx = ValidationCtx.create_ctx_with_config_type_upstream(namespace_pb=self.cache.must_get_namespace(self.entity.id),
                                                                            full_upstream_id=(self.entity.id, upstream_id),
                                                                            upstream_spec_pb=spec_pb)
        holder = validate_and_parse_yaml_upstream_config(namespace_id=meta_pb.namespace_id,
                                                         upstream_id=meta_pb.id,
                                                         spec_pb=spec_pb,
                                                         ctx=validation_ctx)

        included_full_backend_ids = get_included_full_backend_ids_from_holder(meta_pb.namespace_id, holder)
        rev_index_pb = model_pb2.RevisionGraphIndex()
        rev_index_pb.included_backend_ids.extend(flatten_full_id2(f_id) for f_id in included_full_backend_ids)

        self.entity.dao.create_upstream_if_missing(meta_pb, spec_pb, rev_index_pb)

        self.entity.context[u'completed_upstream_ids'][meta_pb.id] = True

        if self.entity.order_type == OrderType.EASY_MODE_WITH_DOMAINS:
            # note: domain creation includes cert creation
            # and is mutually exclusive with CreatingCerts
            return self.next_state_creating_domains
        return self.next_state


class CreatingCerts(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.CREATING_CERTS
    next_state = State.WAITING_FOR_CERTS
    cancelled_state = None

    def process(self, ctx):
        self.entity.context[u'completed_cert_ids'] = {}
        if self.order_pb.certificate_order_content.common_name:
            meta_pb = model_pb2.CertificateMeta(
                id=self.order_pb.certificate_order_content.common_name,
                namespace_id=self.entity.id,
            )
            self._set_auth(meta_pb)
            self.entity.dao.create_cert_if_missing(meta_pb, self.order_pb.certificate_order_content)
            self.entity.context[u'completed_cert_ids'][meta_pb.id] = False

        return self.next_state


class WaitingForCerts(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_CERTS
    next_state = State.VALIDATING_AWACS_NAMESPACE
    cancelled_state = None

    def process(self, ctx):
        all_ready = True
        for cert_id in six.iterkeys(self.entity.context[u'completed_cert_ids']):
            if self.entity.context[u'completed_cert_ids'][cert_id]:
                continue
            cert_pb = self.cache.get_cert(self.entity.id, cert_id)
            if cert_pb and is_spec_complete(cert_pb):
                self.entity.context[u'completed_cert_ids'][cert_id] = True
            else:
                all_ready = False
        if not all_ready:
            return self.state
        return self.next_state


class CreatingDomains(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.CREATING_DOMAINS
    next_state = State.WAITING_FOR_DOMAINS
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def _make_domain_order(self, upstream_id):
        """
        :rtype: model_pb2.DomainOrder
        """
        order_pb = model_pb2.DomainOrder()
        order_pb.content.fqdns.append(self.order_pb.certificate_order_content.common_name)
        order_pb.content.fqdns.extend(self.order_pb.certificate_order_content.subject_alternative_names)
        order_pb.content.protocol = model_pb2.DomainSpec.Config.HTTP_AND_HTTPS

        order_pb.content.cert_order.content.ca_name = self.order_pb.certificate_order_content.ca_name
        order_pb.content.cert_order.content.common_name = self.order_pb.certificate_order_content.common_name
        order_pb.content.cert_order.content.subject_alternative_names.extend(
            self.order_pb.certificate_order_content.subject_alternative_names)
        order_pb.content.cert_order.content.abc_service_id = self.meta_pb.abc_service_id
        order_pb.content.cert_order.content.public_key_algorithm_id = 'rsa'

        order_pb.content.include_upstreams.type = modules_pb2.BY_ID
        order_pb.content.include_upstreams.ids.append(upstream_id)
        return order_pb

    def _make_domain_meta(self, domain_id):
        """
        :type domain_id: six.text_type
        :rtype: model_pb2.DomainMeta
        """
        return model_pb2.DomainMeta(id=domain_id, namespace_id=self.entity.id,
                                    author=self.entity.pb.meta.author)

    def process(self, ctx):
        self.entity.context['completed_domain_ids'] = {}
        domain_id = self.order_pb.certificate_order_content.common_name.lower()
        upstream_id = domain_id.replace('.', '_')
        order_pb = self._make_domain_order(upstream_id)
        meta_pb = self._make_domain_meta(domain_id)
        self.entity.dao.create_domain_if_missing(meta_pb=meta_pb,
                                                 order_content_pb=order_pb.content,
                                                 login=self.entity.pb.meta.author)
        self.entity.context['completed_domain_ids'][meta_pb.id] = False
        return self.next_state


class WaitingForDomains(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_DOMAINS
    next_state = State.VALIDATING_AWACS_NAMESPACE
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def process(self, ctx):
        all_ready = True
        for domain_id in six.iterkeys(self.entity.context['completed_domain_ids']):
            if self.entity.context['completed_domain_ids'][domain_id]:
                continue
            domain_pb = self.cache.get_domain(self.entity.id, domain_id)
            if domain_pb and not domain_pb.spec.incomplete:
                self.entity.context['completed_domain_ids'][domain_id] = True
            else:
                all_ready = False
        if not all_ready:
            return self.state
        return self.next_state


def is_namespace_ready(ctx, entity, required_state):
    assert required_state in (u'valid', u'in_progress')
    ns_proxy = BalancerNamespaceProxy(entity.id)
    c = cache.IAwacsCache.instance()

    balancer_versions = {}
    upstream_versions = {}
    backend_versions = {}
    domain_versions = {}
    sd_backend_full_ids = set()
    endpoint_set_versions = {}
    for balancer_id in six.iterkeys(entity.context[u'completed_balancer_ids']):
        full_balancer_id = (entity.id, balancer_id)
        balancer_versions[full_balancer_id] = ns_proxy.must_get_latest_balancer_version(full_balancer_id)
    for upstream_id in six.iterkeys(entity.context.get(u'completed_upstream_ids', {})):
        full_upstream_id = (entity.id, upstream_id)
        upstream_versions[full_upstream_id] = ns_proxy.must_get_latest_upstream_version(full_upstream_id)
    for domain_id in six.iterkeys(entity.context.get(u'completed_domain_ids', {})):
        full_domain_id = (entity.id, domain_id)
        domain_versions[full_domain_id] = ns_proxy.must_get_latest_domain_version(full_domain_id)
    for backend_id in six.iterkeys(entity.context.get(u'completed_backend_ids', {})):
        full_backend_id = (entity.id, backend_id)
        backend_pb = c.must_get_backend(*full_backend_id)
        if backend_pb.spec.selector.type == model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD:
            sd_backend_full_ids.add(full_backend_id)
        backend_version = BackendVersion.from_pb(backend_pb)
        backend_versions[full_backend_id] = backend_version
        try:
            endpoint_set_versions[full_backend_id] = ns_proxy.must_get_latest_endpoint_set_version(full_backend_id)
        except NotFoundError:
            pass

    backends_to_be_resolved = set(backend_versions.keys()) - sd_backend_full_ids
    resolved_backends = set(endpoint_set_versions.keys())
    all_backends_resolved = True
    entity.context[u'resolved_backend_ids'] = {}
    for backend_id in backends_to_be_resolved:
        resolved = backend_id in resolved_backends
        entity.context[u'resolved_backend_ids'][backend_id[1]] = resolved
        if not resolved:
            all_backends_resolved = False
    if all_backends_resolved:
        ctx.log.debug(u'All backends are resolved')
    else:
        short_backend_uids = [b for b, r in six.iteritems(entity.context[u'resolved_backend_ids']) if not r]
        ctx.log.debug(u'Unresolved backends: %s', u', '.join(short_backend_uids))
        return False

    ok_balancer_versions = set()
    ok_upstream_versions = set()
    ok_domain_versions = set()
    ok_backend_versions = set()
    ok_endpoint_set_versions = set()

    for i, balancer_id in enumerate(six.iterkeys(entity.context[u'completed_balancer_ids'])):
        balancer_state_pb = c.must_get_balancer_state(namespace_id=entity.id,
                                                      balancer_id=balancer_id)
        h = BalancerStateHolder(namespace_id=entity.id,
                                balancer_id=balancer_id,
                                balancer_state_pb=balancer_state_pb)
        if required_state == u'valid':
            vector = h.valid_vector
        elif required_state == u'in_progress':
            vector = h.in_progress_vector
        else:
            raise RuntimeError(u"Unknown required_state {}".format(required_state))
        ok_balancer_versions.add(vector.balancer_version)
        if i == 0:
            ok_upstream_versions = set(vector.upstream_versions.values())
            ok_domain_versions = set(vector.domain_versions.values())
            ok_backend_versions = set(vector.backend_versions.values())
            ok_endpoint_set_versions = set(vector.endpoint_set_versions.values())
        else:
            ok_upstream_versions &= set(vector.upstream_versions.values())
            ok_domain_versions &= set(vector.domain_versions.values())
            ok_backend_versions &= set(vector.backend_versions.values())
            ok_endpoint_set_versions &= set(vector.endpoint_set_versions.values())

    balancers_are_ready = set(balancer_versions.values()) <= ok_balancer_versions
    if balancers_are_ready:
        ctx.log.debug(u'All balancers are %s', required_state)
    else:
        ctx.log.debug(u'Not all balancers are %s', required_state)
        ctx.log.debug(u'Expected balancer versions: %s', set(balancer_versions.values()))
        ctx.log.debug(u'Actual balancer versions: %s', ok_balancer_versions)

    upstreams_are_ready = set(upstream_versions.values()) <= ok_upstream_versions
    if upstreams_are_ready:
        ctx.log.debug(u'All upstreams are %s', required_state)
    else:
        ctx.log.debug(u'Not all upstreams are %s', required_state)
        ctx.log.debug(u'Expected upstreams versions: %s', set(upstream_versions.values()))
        ctx.log.debug(u'Actual upstreams versions: %s', ok_upstream_versions)

    domains_are_ready = set(domain_versions.values()) <= ok_domain_versions
    if domains_are_ready:
        ctx.log.debug(u'All domains are %s', required_state)
    else:
        ctx.log.debug(u'Not all domains are %s', required_state)
        ctx.log.debug(u'Expected domains versions: %s', set(domain_versions.values()))
        ctx.log.debug(u'Actual domains versions: %s', ok_domain_versions)

    backends_are_ready = set(backend_versions.values()) <= ok_backend_versions
    if backends_are_ready:
        ctx.log.debug(u'All backends are %s', required_state)
    else:
        ctx.log.debug(u'Not all backends are %s', required_state)
        ctx.log.debug(u'Expected backends versions: %s', set(backend_versions.values()))
        ctx.log.debug(u'Actual backends versions: %s', ok_backend_versions)

    endpoint_sets_are_ready = set(endpoint_set_versions.values()) <= ok_endpoint_set_versions
    if endpoint_sets_are_ready:
        ctx.log.debug(u'All endpoint sets are %s', required_state)
    else:
        ctx.log.debug(u'Not all endpoint sets are %s', required_state)
        ctx.log.debug(u'Expected endpoint sets versions: %s', set(endpoint_set_versions.values()))
        ctx.log.debug(u'Actual endpoint sets versions: %s', ok_endpoint_set_versions)

    result = (
        balancers_are_ready
        and domains_are_ready
        and upstreams_are_ready
        and backends_are_ready
        and endpoint_sets_are_ready
    )
    if not result:
        ctx.log.debug(u'Not everything inside balancer is ready')
    return result


class ValidatingNamespace(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.VALIDATING_AWACS_NAMESPACE
    next_state = State.UNPAUSING_BALANCER_CONFIG_UPDATES
    cancelled_state = None

    def _is_namespace_valid(self, ctx):
        return is_namespace_ready(ctx, self.entity, u'valid')

    def process(self, ctx):
        if not self._is_namespace_valid(ctx):
            return self.state
        return self.next_state


class UnpausingBalancerConfigUpdates(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.UNPAUSING_BALANCER_CONFIG_UPDATES
    next_state = State.WAITING_FOR_BALANCERS_TO_BE_IN_PROGRESS
    cancelled_state = None

    def process(self, ctx):
        updated_transport_paused_pb = model_pb2.PausedCondition(
            value=False,
            author=util.NANNY_ROBOT_LOGIN,
            comment=u'Unpause balancer config updates'
        )
        updated_transport_paused_pb.mtime.GetCurrentTime()

        for balancer_id in six.iterkeys(self.entity.context[u'completed_balancer_ids']):
            balancer_pb = self.cache.must_get_balancer(self.entity.id, balancer_id)
            balancer_version = balancer_pb.meta.version
            if balancer_pb.meta.transport_paused.value:
                self.entity.dao.update_balancer(
                    namespace_id=self.entity.id,
                    balancer_id=balancer_id,
                    version=balancer_version,
                    comment=u'Unpause balancer config updates',
                    login=util.NANNY_ROBOT_LOGIN,
                    updated_transport_paused_pb=updated_transport_paused_pb
                )
        return self.next_state


class WaitingForBalancersToBeInProgress(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.WAITING_FOR_BALANCERS_TO_BE_IN_PROGRESS
    next_state = State.ACTIVATING_NANNY_SERVICES
    cancelled_state = None

    def _is_namespace_in_progress(self, ctx):
        return is_namespace_ready(ctx, self.entity, u'in_progress')

    def process(self, ctx):
        if not self._is_namespace_in_progress(ctx):
            return self.state
        return self.next_state


class ActivatingNannyServices(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.ACTIVATING_NANNY_SERVICES
    next_state = State.CREATING_DNS_RECORD
    cancelled_state = None

    nanny_client = inject.attr(nannyclient.INannyClient)  # type: nannyclient.NannyClient

    def _activate_balancer_snapshot(self, balancer_id):
        balancer_state_pb = self.cache.must_get_balancer_state(namespace_id=self.entity.id,
                                                               balancer_id=balancer_id)
        statuses = balancer_state_pb.balancer.statuses
        if not statuses:
            return False
        snapshots = statuses[-1].in_progress.meta.nanny_static_file.snapshots
        if not snapshots:
            return False
        snapshot_pb = snapshots[-1]
        self.entity.context[u'balancer_snapshots_info'][balancer_id] = (snapshot_pb.service_id, snapshot_pb.snapshot_id)
        self.nanny_client.set_snapshot_state(
            service_id=snapshot_pb.service_id,
            snapshot_id=snapshot_pb.snapshot_id,
            state=u'ACTIVE',
            comment=u'Activating L7 balancer',
            recipe=u'common')
        return True

    def process(self, ctx):
        self.entity.context[u'balancer_snapshots_info'] = {}
        all_activated = True
        for balancer_id in six.iterkeys(self.entity.context[u'completed_balancer_ids']):
            all_activated &= self._activate_balancer_snapshot(balancer_id)
        if not all_activated:
            return self.state
        return self.next_state


class CreatingDnsRecord(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.CREATING_DNS_RECORD
    next_state = State.VALIDATING_DNS_RECORD
    cancelled_state = None

    def _create_dns_record(self):
        dns_req_pb = self.order_pb.dns_record_request
        name_server = self.cache.must_get_name_server(dns_req_pb.default.name_server.namespace_id,
                                                      dns_req_pb.default.name_server.id)
        dns_record_id = u'{hostname}.{domain}'.format(hostname=dns_req_pb.default.zone, domain=name_server.spec.zone)
        meta_pb = model_pb2.DnsRecordMeta(
            id=dns_record_id,
            namespace_id=self.entity.id
        )
        self._set_auth(meta_pb)

        spec_pb = model_pb2.DnsRecordSpec()
        spec_pb.name_server.namespace_id = dns_req_pb.default.name_server.namespace_id
        spec_pb.name_server.id = dns_req_pb.default.name_server.id
        spec_pb.address.zone = dns_req_pb.default.zone
        spec_pb.address.backends.type = spec_pb.address.backends.BALANCERS
        for balancer_id in six.iterkeys(self.entity.context[u'completed_balancer_ids']):
            spec_pb.address.backends.balancers.add(id=balancer_id)

        self.entity.dao.create_dns_record_if_missing(meta_pb, spec_pb)
        self.entity.context[u'completed_dns_record_ids'][meta_pb.id] = True

    def process(self, ctx):
        self.entity.context[u'completed_dns_record_ids'] = {}

        if self.order_pb.dns_record_request.default.zone:
            self._create_dns_record()

        return self.next_state


class ValidatingDnsRecord(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.VALIDATING_DNS_RECORD
    next_state = State.ENABLING_NAMESPACE_ALERTING
    cancelled_state = None

    cache = inject.attr(cache.IAwacsCache)  # type: cache.AwacsCache

    def _is_dns_record_valid(self, ctx):
        backend_versions = get_backend_versions(self.cache, namespace_id=self.entity.id)
        endpoint_set_versions = get_endpoint_set_versions(self.cache, namespace_id=self.entity.id)
        dns_record_versions = get_dns_record_versions(self.cache, namespace_id=self.entity.id)
        dns_record_versions_to_check = set()
        backend_versions_to_check = set()
        endpoint_set_versions_to_check = set()

        backend_ids = self.entity.context[u'completed_balancer_ids']
        for backend_id in six.iterkeys(backend_ids):
            full_backend_id = (self.entity.id, backend_id)
            backend_version = backend_versions.get(full_backend_id)
            es_version = endpoint_set_versions.get(full_backend_id)
            if backend_version is None or es_version is None:
                ctx.log.debug('Backend "%s" is not resolved yet', backend_id)
                return False
            backend_versions_to_check.add(backend_version)
            endpoint_set_versions_to_check.add(es_version)

        ok_dns_record_versions = set()
        ok_backend_versions = set()
        ok_endpoint_set_versions = set()

        for i, dns_record_id in enumerate(six.iterkeys(self.entity.context[u'completed_dns_record_ids'])):
            dns_record_state_pb = self.cache.must_get_dns_record_state(namespace_id=self.entity.id,
                                                                       dns_record_id=dns_record_id)
            dns_record_versions_to_check.add(dns_record_versions.get((self.entity.id, dns_record_id)))
            vector = DnsRecordStateHandler(dns_record_state_pb).generate_vectors().valid
            ok_dns_record_versions.add(vector.dns_record_version)
            if i == 0:
                ok_backend_versions = set(vector.backend_versions.values())
                ok_endpoint_set_versions = set(vector.endpoint_set_versions.values())
            else:
                ok_backend_versions &= set(vector.backend_versions.values())
                ok_endpoint_set_versions &= set(vector.endpoint_set_versions.values())

        dns_records_are_ready = dns_record_versions_to_check <= ok_dns_record_versions
        if dns_records_are_ready:
            ctx.log.debug(u'All DNS records are ready')
        else:
            ctx.log.debug(u'Not all DNS records are ready')
            ctx.log.debug(u'Expected DNS record versions: %s', dns_record_versions_to_check)
            ctx.log.debug(u'Actual DNS record versions: %s', ok_dns_record_versions)

        backends_are_ready = backend_versions_to_check <= ok_backend_versions
        if backends_are_ready:
            ctx.log.debug(u'All backends are ready')
        else:
            ctx.log.debug(u'Not all backends are ready')
            ctx.log.debug(u'Expected backends versions: %s', backend_versions_to_check)
            ctx.log.debug(u'Actual backends versions: %s', ok_backend_versions)

        endpoint_sets_are_ready = endpoint_set_versions_to_check <= ok_endpoint_set_versions
        if endpoint_sets_are_ready:
            ctx.log.debug(u'All endpoint sets are ready')
        else:
            ctx.log.debug(u'Not all endpoint sets are ready')
            ctx.log.debug(u'Expected endpoint sets versions: %s', endpoint_set_versions_to_check)
            ctx.log.debug(u'Actual endpoint sets versions: %s', ok_endpoint_set_versions)

        result = dns_records_are_ready and backends_are_ready and endpoint_sets_are_ready
        if not result:
            ctx.log.debug(u'Not everything inside DNS records is ready')
        return result

    def process(self, ctx):
        if self.order_pb.dns_record_request.default.zone and not self._is_dns_record_valid(ctx):
            return self.state
        return self.next_state


class EnablingNamespaceAlerting(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.ENABLING_NAMESPACE_ALERTING
    next_state = State.FINALIZING
    cancelled_state = None

    def _should_disable_juggler_notifications(self):
        alerting_pb = self.entity.pb.spec.alerting
        if alerting_pb.WhichOneof('notify_rules') != 'notify_rules_disabled':
            return False
        return alerting_pb.notify_rules_disabled

    def process(self, ctx):
        staff_group_id = self.order_pb.alerting_simple_settings.notify_staff_group_id

        alerting.apply_alerting_preset(self.entity.pb.spec, self.entity.pb.spec.preset)

        alerting_pb = self.entity.pb.spec.alerting
        alerting_pb.version = six.text_type(alerting.CURRENT_VERSION)
        alerting_pb.juggler_raw_downtimers.staff_group_ids.append(staff_group_id)

        if not self._should_disable_juggler_notifications():
            alerting.fill_default_notify_rules(alerting_pb, staff_group_ids=[staff_group_id])

        self.entity.dao_update(self.entity.pb.spec, comment=u'Namespace alerting enabled')
        return self.next_state


class Finalizing(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.FINALIZING
    next_state = State.FINISHED
    cancelled_state = None

    def process(self, ctx):
        updated_annotations = None
        self.entity.pb.spec.incomplete = False
        if self.order_pb.project == self.order_pb.PR_DZEN:
            updated_annotations = self.entity.pb.meta.annotations
            updated_annotations['project'] = 'dzen'
            self.entity.pb.spec.object_upper_limits.l3_balancer.value = 0

        if self.entity.order_type == OrderType.EMPTY:
            self.entity.pb.spec.layout_type = self.entity.pb.spec.NS_LAYOUT_L3_ONLY
            apply_l3_only_layout_constraints(self.entity.pb.spec)
        else:
            easy_mode_settings_pb = self.entity.pb.spec.easy_mode_settings
            easy_mode_settings_pb.non_sd_backends_creation_disabled.value = True
            easy_mode_settings_pb.prohibit_explicit_l3_selector.value = True
            easy_mode_settings_pb.prohibit_explicit_dns_selector.value = True
            easy_mode_settings_pb.l7_macro_only.value = True
            any_upstream_mode_whitelist = config.get_value('ns_easy_mode_constraints', {}).get('any_upstream_mode_whitelist', [])
            creator = self.entity.pb.meta.annotations.get('creator')
            if creator not in any_upstream_mode_whitelist:
                easy_mode_settings_pb.l7_upstream_macro_only.value = True

        self.entity.pb.spec.cloud_type = self.entity.pb.order.content.cloud_type
        if self.entity.pb.order.content.cloud_type == model_pb2.CT_AZURE:
            apply_external_layout_constraints(self.entity.pb.spec)

        self.entity.dao_update(self.entity.pb.spec, updated_annotations=updated_annotations)
        return self.next_state


class Cancelling(NamespaceOrderProcessor):
    __slots__ = ()
    state = State.CANCELLING
    next_state = State.CANCELLED
    cancelled_state = None

    def process(self, ctx):
        for location in self.order_pb.yp_lite_allocation_request.locations:
            balancer_id = balancer_util.make_awacs_balancer_id(self.entity.id, location)
            for balancer_pb in self.entity.zk.update_balancer(namespace_id=self.entity.id, balancer_id=balancer_id):
                if not balancer_pb.order.cancelled.value and balancer_pb.spec.incomplete:
                    cancel_order(balancer_pb, author=util.NANNY_ROBOT_LOGIN, comment=u'Cancelled from namespace order')
        self.entity.pb.spec.incomplete = False
        self.entity.dao_update(self.entity.pb.spec, comment=u'Cancelled namespace order')
        return self.next_state
