from collections import defaultdict, namedtuple

import six
from typing import Generator
from six.moves import map
from datetime import datetime

from awacs.lib.vectors.version import L7HeavyConfigVersion
from awacs.lib.models.classes import (
    Descriptor,
    ModelObject,
    ModelZkClient,
    ModelCache,
    ModelMongoClient,
    ZkNodeType,
    IncludeInDaemons,
)
from awacs.model import codecs, util, errors
from infra.awacs.proto import model_pb2


# State


class L7HeavyConfigStateDescriptor(Descriptor):
    zk_prefix = u'l7heavy_config_states'
    zk_node_type = ZkNodeType.DOUBLE_NESTED

    proto_class = model_pb2.L7HeavyConfigState
    codec_class = codecs.StateGenerationCodec

    include_in_daemons = IncludeInDaemons.ALL_EXCEPT_STATUS


class L7HeavyConfigStateZkClient(ModelZkClient):
    desc = L7HeavyConfigStateDescriptor

    @classmethod
    def create(cls, pb):
        """
        :type pb: model_pb2.L7HeavyConfigState
        """
        return cls._create(cls.desc.uid_to_zk_path(pb.namespace_id, pb.l7heavy_config_id), pb)

    @classmethod
    def get(cls, namespace_id, l7hc_id, sync=False):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :type sync: bool
        :rtype: model_pb2.L7HeavyConfigState | None
        """
        return cls._get(cls.desc.uid_to_zk_path(namespace_id, l7hc_id))

    @classmethod
    def must_get(cls, namespace_id, l7hc_id, sync=False):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :type sync: bool
        :rtype: model_pb2.L7HeavyConfigState
        :raises: errors.NotFoundError
        """
        return cls._must_get(cls.desc.uid_to_zk_path(namespace_id, l7hc_id))

    @classmethod
    def update(cls, namespace_id, l7hc_id, pb=None):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :type pb: model_pb2.L7HeavyConfigState | None
        :rtype: Generator[model_pb2.L7HeavyConfigState, None, None]
        :raises: errors.NotFoundError
        """
        if pb is not None:
            assert isinstance(pb, model_pb2.L7HeavyConfigState)
            assert pb.namespace_id == namespace_id
            assert pb.l7heavy_config_id == l7hc_id
        for pb in cls._update(cls.desc.uid_to_zk_path(namespace_id, l7hc_id), pb):
            yield pb

    @classmethod
    def remove(cls, namespace_id, l7hc_id):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        """
        return cls._remove(cls.desc.uid_to_zk_path(namespace_id, l7hc_id))

    @classmethod
    def remove_all(cls, namespace_id):
        """
        :type namespace_id: six.text_type
        """
        return cls._remove_recursive(namespace_id)

    @classmethod
    def cancel_order(cls, namespace_id, l7hc_id, author, comment=u''):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :type author: six.text_type
        :type comment: six.text_type
        """
        return cls._cancel_order(cls.desc.uid_to_zk_path(namespace_id, l7hc_id), author, comment)


class L7HeavyConfigStateCache(ModelCache):
    desc = L7HeavyConfigStateDescriptor

    _cache = {}
    _cache_by_namespace_id = defaultdict(set)

    @classmethod
    def add(cls, zk_path, pb):
        """
        :type zk_path: six.text_type
        :type pb: model_pb2.L7HeavyConfigState
        """
        added = cls._add(zk_path, pb)
        if not added:
            return

        cls._cache_by_namespace_id[pb.namespace_id].add(zk_path)

    @classmethod
    def discard(cls, zk_path):
        """
        :type zk_path: six.text_type
        """
        l7hc_pb = cls._discard(zk_path)
        if not l7hc_pb:
            return

        cls._cache_by_namespace_id[l7hc_pb.namespace_id].discard(zk_path)

    @classmethod
    def get(cls, namespace_id, l7hc_id):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :rtype: model_pb2.L7HeavyConfigState | None
        """
        return cls._cache.get(cls.desc.uid_to_zk_path(namespace_id, l7hc_id))

    @classmethod
    def must_get(cls, namespace_id, l7hc_id):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :rtype: model_pb2.L7HeavyConfigState
        :raises: errors.NotFoundError
        """
        return cls._must_get(cls.desc.uid_to_zk_path(namespace_id, l7hc_id))

    @classmethod
    def count(cls, namespace_id=None):
        """
        :type namespace_id: six.text_type | None
        :rtype: int
        """
        if namespace_id:
            return len(cls._cache_by_namespace_id.get(namespace_id) or frozenset())
        else:
            return len(cls._cache)


class L7HeavyConfigState(ModelObject):
    desc = L7HeavyConfigStateDescriptor
    zk = L7HeavyConfigStateZkClient
    cache = L7HeavyConfigStateCache

    @classmethod
    def create(cls, ns_id, l7hc_id, utcnow):
        """
        :rtype: model_pb2.L7HeavyConfigState
        """
        l7hc_state_pb = model_pb2.L7HeavyConfigState(namespace_id=ns_id, l7heavy_config_id=l7hc_id)
        l7hc_state_pb.ctime.FromDatetime(utcnow)
        try:
            cls.zk.create(l7hc_state_pb)
        except errors.ConflictError as e:
            raise errors.ConflictError('Failed to create L7HeavyConfig state "{}:{}": {}'.format(ns_id, l7hc_id, e))
        return l7hc_state_pb

    @classmethod
    def remove(cls, namespace_id, l7hc_state_id):
        """
        :type namespace_id: six.text_type
        :type l7hc_state_id: six.text_type
        """
        cls.zk.remove(namespace_id, l7hc_state_id)

    @classmethod
    def update(cls):
        raise NotImplementedError


# Config


class L7HeavyConfigDescriptor(Descriptor):
    zk_prefix = u'l7heavy_configs'
    zk_node_type = ZkNodeType.DOUBLE_NESTED

    mongo_revs_column = 'l7heavy_config_revisions'
    rev_proto_class = model_pb2.L7HeavyConfigRevision

    proto_class = model_pb2.L7HeavyConfig
    codec_class = codecs.GenerationCodec

    include_in_daemons = IncludeInDaemons.ALL


class L7HeavyConfigZkClient(ModelZkClient):
    desc = L7HeavyConfigDescriptor

    @classmethod
    def create(cls, pb):
        """
        :type pb: model_pb2.L7HeavyConfig
        """
        return cls._create(cls.desc.uid_to_zk_path(pb.meta.namespace_id, pb.meta.id), pb)

    @classmethod
    def get(cls, namespace_id, l7hc_id, sync=False):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :type sync: bool
        :rtype: model_pb2.L7HeavyConfig | None
        """
        return cls._get(cls.desc.uid_to_zk_path(namespace_id, l7hc_id))

    @classmethod
    def must_get(cls, namespace_id, l7hc_id, sync=False):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :type sync: bool
        :rtype: model_pb2.L7HeavyConfig
        :raises: errors.NotFoundError
        """
        return cls._must_get(cls.desc.uid_to_zk_path(namespace_id, l7hc_id))

    @classmethod
    def update(cls, namespace_id, l7hc_id, pb=None):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :type pb: model_pb2.L7HeavyConfig | None
        :rtype: Generator[model_pb2.L7HeavyConfig, None, None]
        :raises: errors.NotFoundError
        """
        if pb is not None:
            assert isinstance(pb, model_pb2.L7HeavyConfig)
            assert pb.meta.namespace_id == namespace_id
            assert pb.meta.id == l7hc_id
        for pb in cls._update(cls.desc.uid_to_zk_path(namespace_id, l7hc_id), pb):
            yield pb

    @classmethod
    def remove(cls, namespace_id, l7hc_id):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        """
        return cls._remove(cls.desc.uid_to_zk_path(namespace_id, l7hc_id))

    @classmethod
    def remove_all(cls, namespace_id):
        """
        :type namespace_id: six.text_type
        """
        return cls._remove_recursive(namespace_id)

    @classmethod
    def cancel_order(cls, namespace_id, l7hc_id, author, comment=u''):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :type author: six.text_type
        :type comment: six.text_type
        """
        return cls._cancel_order(cls.desc.uid_to_zk_path(namespace_id, l7hc_id), author, comment)


class L7HeavyConfigCache(ModelCache):
    desc = L7HeavyConfigDescriptor

    _cache = {}
    _cache_by_namespace_id = defaultdict(set)

    @classmethod
    def clear(cls):
        cls._cache.clear()
        cls._cache_by_namespace_id.clear()

    @classmethod
    def add(cls, zk_path, pb):
        """
        :type zk_path: six.text_type
        :type pb: model_pb2.L7HeavyConfig
        """
        added = cls._add(zk_path, pb)
        if not added:
            return

        cls._cache_by_namespace_id[pb.meta.namespace_id].add(zk_path)

    @classmethod
    def discard(cls, zk_path):
        """
        :type zk_path: six.text_type
        """
        l7hc_pb = cls._discard(zk_path)
        if not l7hc_pb:
            return
        assert zk_path in cls._cache_by_namespace_id[l7hc_pb.meta.namespace_id]
        cls._cache_by_namespace_id[l7hc_pb.meta.namespace_id].discard(zk_path)

    @classmethod
    def get(cls, namespace_id, l7hc_id):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :rtype: model_pb2.L7HeavyConfig | None
        """
        return cls._cache.get(cls.desc.uid_to_zk_path(namespace_id, l7hc_id))

    @classmethod
    def must_get(cls, namespace_id, l7hc_id):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :rtype: model_pb2.L7HeavyConfig
        :raises: errors.NotFoundError
        """
        return cls._must_get(cls.desc.uid_to_zk_path(namespace_id, l7hc_id))

    @classmethod
    def count(cls, namespace_id=None):
        """
        :type namespace_id: six.text_type | None
        :rtype: int
        """
        if namespace_id:
            return len(cls._cache_by_namespace_id.get(namespace_id) or frozenset())
        else:
            return len(cls._cache)

    @classmethod
    def list(cls, namespace_id=None):
        """
        :type namespace_id: six.text_type | None
        :rtype: List[model_pb2.L7HeavyConfig]
        """
        if namespace_id:
            paths = cls._cache_by_namespace_id.get(namespace_id, set())
        elif cls._cache_by_namespace_id:
            paths = set.union(*six.itervalues(cls._cache_by_namespace_id))
        else:
            paths = set()
        return list(map(cls._cache.get, sorted(paths)))


class L7HeavyConfigMongoClient(ModelMongoClient):
    desc = L7HeavyConfigDescriptor


class L7HeavyConfig(ModelObject):
    desc = L7HeavyConfigDescriptor
    zk = L7HeavyConfigZkClient
    cache = L7HeavyConfigCache
    mongo = L7HeavyConfigMongoClient
    version = L7HeavyConfigVersion
    state = L7HeavyConfigState

    @classmethod
    def create(cls, meta_pb, order_content_pb, login):
        """
        :type meta_pb: model_pb2.L7HeavyConfigMeta
        :type order_content_pb: model_pb2.L7HeavyConfigOrder.Content
        :type login: six.text_type
        :rtype: model_pb2.L7HeavyConfig
        """
        utcnow = datetime.utcnow()
        spec_pb = model_pb2.L7HeavyConfigSpec(incomplete=True)
        l7hc_pb = cls._create_with_rev(meta_pb, login, utcnow=utcnow, order_content_pb=order_content_pb, spec_pb=spec_pb)
        l7hc_state_pb = cls.state.create(meta_pb.namespace_id, meta_pb.id, utcnow)
        return l7hc_pb, l7hc_state_pb

    @classmethod
    def remove(cls, namespace_id, l7hc_id):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        """
        cls.mongo.remove_revs_by_full_id(namespace_id, l7hc_id)
        cls.state.zk.remove(namespace_id, l7hc_id)
        cls.zk.remove(namespace_id, l7hc_id)

    @classmethod
    def update(cls, namespace_id, l7hc_id, version, comment, login, updated_spec_pb=None, updated_transport_paused_pb=None):
        """
        :type namespace_id: six.text_type
        :type l7hc_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_spec_pb: model_pb2.L7HeavyConfigSpec
        :type updated_spec_pb: model_pb2.PausedCondition | None
        :rtype: model_pb2.L7HeavyConfig
        """
        assert updated_transport_paused_pb or updated_spec_pb
        l7hc_pb = None
        new_version = cls._gen_version_id()
        utcnow = datetime.utcnow()
        if updated_spec_pb is not None:
            rev_pb = cls._gen_new_rev(namespace_id, l7hc_id, new_version, comment, login, updated_spec_pb, utcnow=utcnow)
            cls.mongo.save_rev(rev_pb)

        for l7hc_pb in cls.zk.update(namespace_id, l7hc_id):
            if updated_spec_pb is not None:
                if l7hc_pb.meta.version != version:
                    cls.mongo.remove_rev(new_version)
                    raise errors.ConflictError(
                        u'L7Heavy Config modification conflict: assumed version="{}", current="{}"'.format(
                            version, l7hc_pb.meta.version))

                if l7hc_pb.spec.state != l7hc_pb.spec.PRESENT:
                    raise errors.ConflictError('L7HeavyConfig with "spec.state" != PRESENT flag can not be updated')
                if not util.update_spec_in_pb(l7hc_pb, updated_spec_pb, new_version, comment, login, utcnow=utcnow):
                    break
            if updated_transport_paused_pb is not None:
                util.set_condition(l7hc_pb.meta.transport_paused,
                                   updated_condition_pb=updated_transport_paused_pb,
                                   author=login,
                                   utcnow=utcnow)

        return l7hc_pb
