from collections import defaultdict

import six
from typing import Generator, List, Tuple
from six.moves import map

from awacs.lib.models.classes import (
    Descriptor,
    ModelObject,
    ModelZkClient,
    ModelCache,
    ZkNodeType,
    IncludeInDaemons,
)
from awacs.lib.strutils import quote_join_sorted
from awacs.model import codecs, util, errors
from awacs.model.objects.descriptors import L7BalancerDescriptor
from infra.awacs.proto import model_pb2


class NsOpDescriptor(Descriptor):
    zk_prefix = u'namespace_operations'
    zk_node_type = ZkNodeType.DOUBLE_NESTED

    proto_class = model_pb2.NamespaceOperation
    codec_class = codecs.GenerationCodec

    include_in_daemons = IncludeInDaemons.ALL_EXCEPT_RESOLVER


class NsOpZkClient(ModelZkClient):
    desc = NsOpDescriptor

    @classmethod
    def create(cls, pb):
        """
        :type pb: model_pb2.NamespaceOperation
        """
        return cls._create(cls.desc.uid_to_zk_path(pb.meta.namespace_id, pb.meta.id), pb)

    @classmethod
    def get(cls, namespace_id, op_id):
        """
        :type namespace_id: six.text_type
        :type op_id: six.text_type
        :rtype: model_pb2.NamespaceOperation | None
        """
        return cls._get(cls.desc.uid_to_zk_path(namespace_id, op_id))

    @classmethod
    def must_get(cls, namespace_id, op_id):
        """
        :type namespace_id: six.text_type
        :type op_id: six.text_type
        :rtype: model_pb2.NamespaceOperation
        :raises: errors.NotFoundError
        """
        return cls._must_get(cls.desc.uid_to_zk_path(namespace_id, op_id))

    @classmethod
    def update(cls, namespace_id, op_id, pb=None):
        """
        :type namespace_id: six.text_type
        :type op_id: six.text_type
        :type pb: model_pb2.NamespaceOperation | None
        :rtype: Generator[model_pb2.NamespaceOperation, None, None]
        :raises: errors.NotFoundError
        """
        if pb is not None:
            assert isinstance(pb, model_pb2.NamespaceOperation)
            assert pb.meta.namespace_id == namespace_id
            assert pb.meta.id == op_id
        for pb in cls._update(cls.desc.uid_to_zk_path(namespace_id, op_id), pb):
            yield pb

    @classmethod
    def update_pb(cls, pb):
        """
        :type pb: model_pb2.NamespaceOperation
        :rtype: Generator[model_pb2.NamespaceOperation, None, None]
        :raises: errors.NotFoundError
        """
        assert isinstance(pb, model_pb2.NamespaceOperation)
        for pb in cls._update(cls.desc.uid_to_zk_path(pb.meta.namespace_id, pb.meta.id), pb):
            yield pb

    @classmethod
    def remove(cls, namespace_id, op_id):
        """
        :type namespace_id: six.text_type
        :type op_id: six.text_type
        """
        return cls._remove(cls.desc.uid_to_zk_path(namespace_id, op_id))

    @classmethod
    def remove_all(cls, namespace_id):
        """
        :type namespace_id: six.text_type
        """
        return cls._remove_recursive(namespace_id)

    @classmethod
    def cancel_order(cls, namespace_id, op_id, author, comment):
        """
        :type namespace_id: six.text_type
        :type op_id: six.text_type
        :type author: six.text_type
        :type comment: six.text_type
        """
        return cls._cancel_order(cls.desc.uid_to_zk_path(namespace_id, op_id), author, comment)


class NsOpCache(ModelCache):
    desc = NsOpDescriptor

    _cache = {}
    _cache_by_namespace_id = defaultdict(set)

    @classmethod
    def add(cls, zk_path, pb):
        """
        :type zk_path: six.text_type
        :type pb: model_pb2.NamespaceOperation
        """
        added = cls._add(zk_path, pb)
        if added:
            cls._cache_by_namespace_id[pb.meta.namespace_id].add(zk_path)

    @classmethod
    def discard(cls, zk_path):
        """
        :type zk_path: six.text_type
        """
        op_pb = cls._discard(zk_path)
        if op_pb:
            cls._cache_by_namespace_id.pop(op_pb.meta.namespace_id, None)

    @classmethod
    def get(cls, namespace_id, op_id):
        """
        :type namespace_id: six.text_type
        :type op_id: six.text_type
        :rtype: model_pb2.NamespaceOperation | None
        """
        return cls._cache.get(cls.desc.uid_to_zk_path(namespace_id, op_id))

    @classmethod
    def must_get(cls, namespace_id, op_id):
        """
        :type namespace_id: six.text_type
        :type op_id: six.text_type
        :rtype: model_pb2.NamespaceOperation
        :raises: errors.NotFoundError
        """
        return cls._must_get(cls.desc.uid_to_zk_path(namespace_id, op_id))

    @classmethod
    def list(cls, namespace_id=None):
        """
        :type namespace_id: six.text_type | None
        :rtype: List[model_pb2.NamespaceOperation]
        """
        if namespace_id:
            paths = cls._cache_by_namespace_id.get(namespace_id, ())
        else:
            paths = list(cls._cache)
        return list(map(cls._cache.get, sorted(paths)))


class NamespaceOperation(ModelObject):
    desc = NsOpDescriptor
    zk = NsOpZkClient
    cache = NsOpCache

    ADD_IP_TO_L3 = u'add_ip_address_to_l3_balancer'
    IMPORT_VS_FROM_L3MGR = u'import_virtual_servers_from_l3mgr'
    _supported_operations = (ADD_IP_TO_L3, IMPORT_VS_FROM_L3MGR)

    @classmethod
    def get_operation_name(cls, content_pb):
        """
        :type content_pb: model_pb2.NamespaceOperationOrder.Content
        :rtype: six.text_type
        :raises: RuntimeError
        """
        operation = content_pb.WhichOneof('operation')
        if not operation:
            raise RuntimeError(u'Namespace operation not specified. Supported operations: "{}"'.format(
                quote_join_sorted(cls._supported_operations)))
        if operation not in cls._supported_operations:
            raise RuntimeError(u'Namespace operation unknown: "{}". Supported operations: "{}"'.format(
                operation, quote_join_sorted(cls._supported_operations)))
        return operation

    @classmethod
    def get_operation(cls, content_pb):
        """
        :type content_pb: model_pb2.NamespaceOperationOrder.Content
        :rtype: (model_pb2.NamespaceOperationOrder.Content.ImportVirtualServersFromL3mgr |
                 model_pb2.NamespaceOperationOrder.Content.AddIpAddressToL3Balancer)
        :raises: RuntimeError
        """
        operation = cls.get_operation_name(content_pb)
        if operation == cls.ADD_IP_TO_L3:
            return content_pb.add_ip_address_to_l3_balancer
        elif operation == cls.IMPORT_VS_FROM_L3MGR:
            return content_pb.import_virtual_servers_from_l3mgr

    @classmethod
    def _get_affected_object_paths(cls, content_pb):
        """
        :type content_pb: model_pb2.NamespaceOperationOrder.Content
        :rtype: Tuple[six.text_type]
        """
        operation = cls.get_operation_name(content_pb)
        if operation in cls._supported_operations:
            return (L7BalancerDescriptor.zk_prefix,)

    @classmethod
    def lock_affected_objects(cls, content_pb, namespace_id, op_id):
        """
        :type content_pb: model_pb2.NamespaceOperationOrder.Content
        :type namespace_id: six.text_type
        :type op_id: six.text_type
        :raises: KazooTransactionException
        """
        zk_paths = cls._get_affected_object_paths(content_pb)
        cls.zk.lock(zk_paths, namespace_id, op_id)

    @classmethod
    def unlock_affected_objects(cls, content_pb, namespace_id):
        """
        :type content_pb: model_pb2.NamespaceOperationOrder.Content
        :type namespace_id: six.text_type
        """
        zk_paths = cls._get_affected_object_paths(content_pb)
        cls.zk.unlock(zk_paths, namespace_id)

    @classmethod
    def create(cls, meta_pb, order_content_pb, login):
        """
        :type meta_pb: model_pb2.NamespaceOperationMeta
        :type order_content_pb: model_pb2.NamespaceOperationOrder.Content
        :type login: six.text_type
        :rtype: model_pb2.NamespaceOperation
        """
        op_pb = model_pb2.NamespaceOperation(meta=meta_pb)
        util.fill_default_order(meta_pb, order_content_pb, op_pb, login)

        cls.lock_affected_objects(order_content_pb, meta_pb.namespace_id, meta_pb.id)
        cls.zk.create(op_pb)
        return op_pb

    @classmethod
    def remove(cls, op_pb):
        """
        :type op_pb: model_pb2.NamespaceOperation
        """
        cls.unlock_affected_objects(op_pb.order.content, op_pb.meta.namespace_id)
        cls.zk.remove(op_pb.meta.namespace_id, op_pb.meta.id)

    @classmethod
    def update(cls, namespace_id, op_id, version, comment, login, updated_spec_pb):
        """
        :type namespace_id: six.text_type
        :type op_id: six.text_type
        :type version: six.text_type
        :type comment: six.text_type
        :type login: six.text_type
        :type updated_spec_pb: model_pb2.NamespaceOperationSpec
        :rtype: model_pb2.NamespaceOperation
        """
        op_pb = None
        new_version = cls._gen_version_id()

        for op_pb in cls.zk.update(namespace_id, op_id):
            if op_pb.meta.version != version:
                raise errors.ConflictError(
                    u'Namespace operation modification conflict: assumed version="{}", current="{}"'.format(
                        version, op_pb.meta.version))
            if not util.update_spec_in_pb(op_pb, updated_spec_pb, new_version, comment, login):
                break

        return op_pb

    @classmethod
    def cancel_order(cls, op_pb, author, comment=u''):
        return cls.zk.cancel_order(op_pb.meta.namespace_id, op_pb.meta.id, author, comment)
