# coding: utf-8
import collections
import logging

import six

from awacs.lib import OrderedDict
from awacs.model import util, cache
from infra.awacs.proto import model_pb2


log = logging.getLogger()

# https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/GenerateBalancerGraphsPanelLinks/__init__.py?rev=3791243#L76
IGNORED_PRJ_TAGS = frozenset((
    'kernel-test-dev', 'kernel-test-main', 'kernel-stable-new',
    'cplb', 'cplb-menace', 'cplb-saas', 'cplb-dynamic', 'cplb-swat'
))

LVLS = {
    'False': 10,
    'Unknown': 5,
    'Invalidated': 5,
    'True': 0,
}


def _get_worst_condition(fst, snd):
    """
    :type fst: model_pb2.Condition
    :type snd: model_pb2.Condition
    :rtype: model_pb2.Condition
    """
    if LVLS[fst.status] <= LVLS[snd.status]:
        return fst
    else:
        return snd


def _get_yasm_panel_selector(panel_id, balancer_id, tags_pb, locations, signal=None):
    """
    :type balancer_id: six.text_type
    :type tags_pb: model_pb2.ClusterAspectsContent.Tags
    :type locations: list[six.text_type]
    :type signal: six.text_type | None
    :rtype: model_pb2.YasmPanel.Selector
    """
    selector_pb = model_pb2.YasmPanel.Selector()
    selector_pb.id = panel_id
    selector_pb.params.add(name='fqdn', value=balancer_id)
    if tags_pb.itype:
        selector_pb.params.add(name='itype', value=','.join(sorted(tags_pb.itype)))
    if tags_pb.ctype:
        selector_pb.params.add(name='ctype', value=','.join(sorted(tags_pb.ctype)))
    if locations:
        selector_pb.params.add(name='locations', value=','.join(sorted(locations)))
    selector_pb.params.add(name='prj',
                           value=','.join(sorted([tag for tag in tags_pb.prj if tag not in IGNORED_PRJ_TAGS])))
    if signal:
        selector_pb.params.add(name='signal', value=signal)
    return selector_pb


def _get_yasm_panel_url_from_selector(selector_pb):
    """
    :type selector_pb: YasmPanel.Selector
    :rtype: str
    """
    url = 'https://yasm.yandex-team.ru/template/panel/{}/'.format(selector_pb.id)
    for param_pb in selector_pb.params:
        url += '{}={};'.format(param_pb.name, param_pb.value)
    return url


def get_yasm_panel_url(balancer_id, aspects_set_content_pb):
    """
    :type balancer_id: six.text_type
    :type aspects_set_content_pb: model_pb2.BalancerAspectsSetContent
    """
    if not aspects_set_content_pb.HasField('cluster'):
        return '', model_pb2.Condition(status='False', message='awacs has not resolved "cluster" aspects yet')

    if not aspects_set_content_pb.cluster.status.HasField('last_successful_attempt'):
        return '', model_pb2.Condition(status='False', message='awacs has not resolved "cluster" aspects yet')

    if not aspects_set_content_pb.cluster.content.HasField('tags'):
        return '', model_pb2.Condition(status='False', message='Balancer instances are not assigned proper '
                                                               'orthogonal tags.')

    balancer_tags_pb = aspects_set_content_pb.cluster.content.tags
    balancer_locations = aspects_set_content_pb.cluster.content.locations

    selector_pb = _get_yasm_panel_selector('balancer_common_panel', balancer_id, balancer_tags_pb, balancer_locations)
    url = _get_yasm_panel_url_from_selector(selector_pb)

    cond_pb = _get_aspect_condition('cluster', aspects_set_content_pb.cluster)
    return url, cond_pb


def dt_to_str(dt):
    return str(dt.replace(microsecond=0))


def get_balancer_its_location_paths(aspects_set_content_pb):
    """
    :type aspects_set_content_pb: model_pb2.BalancerAspectsSetContent
    """
    if not aspects_set_content_pb.HasField('its'):
        return [], model_pb2.Condition(status='False', message='awacs has not resolved "its" aspects yet')

    if not aspects_set_content_pb.its.status.HasField('last_successful_attempt'):
        return [], model_pb2.Condition(status='False', message='awacs has not resolved "its" aspects yet')

    cond_pb = _get_aspect_condition('its', aspects_set_content_pb.its)
    return list(aspects_set_content_pb.its.content.location_paths), cond_pb


def _merge_occurrences(occurrences_1, occurrences_2):
    rv = list(occurrences_1)
    occurrences_to_skip = set(occurrences_1)
    for occurrence in occurrences_2:
        if occurrence not in occurrences_to_skip:
            rv.append(occurrence)
    return rv


def _format_occurrences(occurrences):
    balancer_ids = []
    upstream_ids = []

    occurrences.sort(key=lambda t__: t__[0])
    for type_, id_ in occurrences:
        if type_ == 'balancer':
            balancer_ids.append(id_)
        elif type_ == 'upstream':
            upstream_ids.append(id_)
    rv = ''
    if balancer_ids:
        if len(balancer_ids) == 1:
            rv += 'balancer {}'.format(balancer_ids[0])
        else:
            rv += 'balancers {}'.format(', '.join(balancer_ids))
    if upstream_ids:
        if balancer_ids:
            rv += ' and '
        if len(upstream_ids) == 1:
            rv += 'upstream {}'.format(upstream_ids[0])
        else:
            rv += 'upstreams {}'.format(', '.join(upstream_ids))
    return rv


def _get_balancer_yasm_default_panels(balancer_id, balancer_tags_pb, balancer_locations):
    """
    :type balancer_id: six.text_type
    :type balancer_tags_pb: model_pb2.ClusterAspectsContent.Tags
    :type balancer_locations: list[str]
    """
    yasm_panel_pbs = []

    for panel_id, desc in (
            ('balancer_portoinst_panel', 'Balancer instances statistics'),
            ('balancer_aux_panel', 'Aux statistics')
    ):
        selector_pb = _get_yasm_panel_selector(
            panel_id, balancer_id, balancer_tags_pb, balancer_locations)
        url = _get_yasm_panel_url_from_selector(selector_pb)
        yasm_panel_pb = model_pb2.YasmPanel(
            id=panel_id.replace('_', '-'),
            url=url,
            selector=selector_pb,
            desc=desc
        )
        yasm_panel_pbs.append(yasm_panel_pb)

    return yasm_panel_pbs


def _get_balancer_yasm_report_uuid_panels(balancer_id, balancer_tags_pb, balancer_locations, config_aspects_content_pb):
    """
    :type balancer_id: six.text_type
    :type balancer_tags_pb: model_pb2.ClusterAspectsContent.Tags
    :type balancer_locations: list[str]
    :type config_aspects_content_pb: model_pb2.ConfigAspectsContent
    """
    yasm_panel_pb_by_report_uuids = OrderedDict()
    report_uuid_occurrences = collections.defaultdict(list)
    yasm_panel_pbs = []

    balancer_config_aspects_pb = config_aspects_content_pb.balancer
    for report_uuid in balancer_config_aspects_pb.report_uuids:
        selector_pb = _get_yasm_panel_selector(
            'balancer_common_panel', balancer_id, balancer_tags_pb, balancer_locations, signal=report_uuid)
        url = _get_yasm_panel_url_from_selector(selector_pb)
        yasm_panel_pb = model_pb2.YasmPanel(
            id=report_uuid,
            url=url,
            selector=selector_pb
        )
        if report_uuid not in yasm_panel_pb_by_report_uuids:
            yasm_panel_pb_by_report_uuids[report_uuid] = yasm_panel_pb
            yasm_panel_pbs.append(yasm_panel_pb)
            report_uuid_occurrences[report_uuid].append(('balancer', balancer_id))

    for upstream_config_aspects_pb in config_aspects_content_pb.upstreams:
        upstream_id = upstream_config_aspects_pb.id
        for report_uuid in upstream_config_aspects_pb.report_uuids:
            selector_pb = _get_yasm_panel_selector(
                'balancer_common_panel', balancer_id, balancer_tags_pb, balancer_locations, signal=report_uuid)
            url = _get_yasm_panel_url_from_selector(selector_pb)
            yasm_panel_pb = model_pb2.YasmPanel(
                id=report_uuid,
                url=url,
                selector=selector_pb
            )
            if report_uuid not in yasm_panel_pb_by_report_uuids:
                yasm_panel_pb_by_report_uuids[report_uuid] = yasm_panel_pb
                yasm_panel_pbs.append(yasm_panel_pb)
                report_uuid_occurrences[report_uuid].append(('upstream', upstream_id))

    return yasm_panel_pbs, yasm_panel_pb_by_report_uuids, report_uuid_occurrences


def _validate_cluster_aspects(aspects_set_content_pb):
    """
    :type aspects_set_content_pb: model_pb2.BalancerAspectsSetContent
    """
    # validate cluster aspects
    if not aspects_set_content_pb.HasField('cluster'):
        return 'awacs has not resolved "cluster" aspects yet'

    if not aspects_set_content_pb.cluster.status.HasField('last_successful_attempt'):
        return 'awacs has not resolved "cluster" aspects yet'

    if not aspects_set_content_pb.cluster.content.HasField('tags'):
        return 'Balancer instances are not assigned proper orthogonal tags.'

    return None


def _validate_config_aspects(aspects_set_content_pb):
    """
    :type aspects_set_content_pb: model_pb2.BalancerAspectsSetContent
    """
    if not aspects_set_content_pb.HasField('config'):
        return 'awacs has not resolved "config" aspects yet'

    if not aspects_set_content_pb.config.status.HasField('last_successful_attempt'):
        return 'awacs has not resolved "config" aspects yet'

    return None


def _get_aspect_condition(aspect_name, aspect_pb):
    """
    :type aspect_name: six.text_type
    :type aspect_pb: model_pb2.ClusterAspects | model_pb2.ItsAspects | model_pb2.ConfigAspects
    :rtype: model_pb2.Condition
    """
    last_attempt_at = aspect_pb.status.last_attempt.finished_at.ToDatetime()
    last_successful_attempt_at = aspect_pb.status.last_successful_attempt.finished_at.ToDatetime()
    if aspect_pb.status.last_attempt.succeeded.status == 'True':
        cond_pb = model_pb2.Condition(status='True')
    else:
        cond_pb = model_pb2.Condition(
            status='Unknown',
            message='Based on the data retrieved by the last successful attempt at {} UTC. '
                    'The last attempt to resolve "{}" aspects finished at {} UTC and '
                    'has not been successful.'.format(dt_to_str(last_successful_attempt_at),
                                                      aspect_name,
                                                      dt_to_str(last_attempt_at)))
    if aspect_pb.status.invalidated:
        cond_pb = model_pb2.Condition(
            status='Invalidated',
            message='"{}" aspects are invalidated. Please wait until they are re-resolved.'.format(aspect_name))
    return cond_pb


def get_balancer_yasm_panels(balancer_id, aspects_set_content_pb):
    """
    :type balancer_id: six.text_type
    :type aspects_set_content_pb: model_pb2.BalancerAspectsSetContent
    """
    err = _validate_cluster_aspects(aspects_set_content_pb)
    if err:
        return [], {}, {}, model_pb2.Condition(status='False', message=err)
    balancer_tags_pb = aspects_set_content_pb.cluster.content.tags
    balancer_locations = aspects_set_content_pb.cluster.content.locations

    err = _validate_config_aspects(aspects_set_content_pb)
    if err:
        return [], {}, {}, model_pb2.Condition(status='False', message=err)
    config_aspects_content_pb = aspects_set_content_pb.config.content

    yasm_panel_pbs = _get_balancer_yasm_default_panels(balancer_id, balancer_tags_pb, balancer_locations)
    report_uuid_yasm_panel_pbs, yasm_panel_pb_by_report_uuids, report_uuid_occurrences = \
        _get_balancer_yasm_report_uuid_panels(balancer_id,
                                              balancer_tags_pb,
                                              balancer_locations,
                                              config_aspects_content_pb)
    yasm_panel_pbs.extend(report_uuid_yasm_panel_pbs)

    cluster_induced_condition = _get_aspect_condition('cluster', aspects_set_content_pb.cluster)
    config_induced_condition = _get_aspect_condition('config', aspects_set_content_pb.config)
    cond_pb = _get_worst_condition(cluster_induced_condition, config_induced_condition)
    return yasm_panel_pbs, yasm_panel_pb_by_report_uuids, report_uuid_occurrences, cond_pb


def get_namespace_its_tabs(balancer_aspects_set_pbs):
    """
    :type balancer_aspects_set_pbs: list[model_pb2.BalancerAspectsSet]
    """
    its_tab_pbs = []

    conditions = []
    for balancer_aspects_set_pb in balancer_aspects_set_pbs:
        balancer_id = balancer_aspects_set_pb.meta.balancer_id
        filled_balancer_aspects_set_pb = util.clone_pb(balancer_aspects_set_pb)
        fill_ui_balancer_aspects(balancer_id, filled_balancer_aspects_set_pb.content)

        balancer_ui_aspects_content_pb = filled_balancer_aspects_set_pb.content.ui.content
        condition_pb = balancer_ui_aspects_content_pb.its_location_paths_condition

        its_tab_pb = model_pb2.NamespaceUiAspectsContent.ItsTab()
        its_tab_pb.id = balancer_id
        its_tab_pb.location_paths.extend(balancer_ui_aspects_content_pb.its_location_paths)
        its_tab_pb.condition.CopyFrom(condition_pb)
        its_tab_pbs.append(its_tab_pb)

        conditions.append((LVLS[condition_pb.status], balancer_id, condition_pb))
    conditions = sorted(conditions)

    it = iter(its_tab_pbs)
    first_its_tab_pb = next(it, None)
    common_location_paths = set(first_its_tab_pb.location_paths) if first_its_tab_pb is not None else set()
    for its_tab_pb in it:
        common_location_paths &= set(its_tab_pb.location_paths)

    if common_location_paths and len(its_tab_pbs) > 1:
        common_its_tab_pb = model_pb2.NamespaceUiAspectsContent.ItsTab()
        common_its_tab_pb.id = 'common'
        common_its_tab_pb.location_paths.extend(sorted(common_location_paths))
        worst_lvl, worst_balancer_id, worst_condition = conditions[-1]
        if worst_lvl > 0:
            common_its_tab_pb.condition.status = worst_condition.status
            common_its_tab_pb.condition.message = '{}: {}'.format(worst_balancer_id, worst_condition.message)
        else:
            common_its_tab_pb.condition.status = 'True'

        for its_tab_pb in its_tab_pbs:
            for path in common_location_paths:
                its_tab_pb.location_paths.remove(path)
        its_tab_pbs.append(common_its_tab_pb)

    return its_tab_pbs, model_pb2.Condition(status='True')


def _selector_to_dict(selector_pb):
    """
    :type selector_pb: model_pb2.YasmPanel.Selector
    :rtype: dict
    """
    params = OrderedDict()
    for param_pb in selector_pb.params:
        params[param_pb.name] = param_pb.value
    return selector_pb.id, params


def _dict_to_selector(selector_id, selector_params):
    """
    :type selector_params: collections.OrderedDcit
    :rtype: model_pb2.YasmPanel.Selector
    """
    selector_pb = model_pb2.YasmPanel.Selector(id=selector_id)
    for name, value in six.iteritems(selector_params):
        selector_pb.params.add(name=name, value=value)
    return selector_pb


def _selectors_equal(selector_1_pb, selector_2_pb, ignored_params=()):
    """
    :type selector_1_pb: model_pb2.YasmPanel.Selector
    :type selector_2_pb: model_pb2.YasmPanel.Selector
    """
    selector_1_id, selector_1_params = _selector_to_dict(selector_1_pb)
    selector_2_id, selector_2_params = _selector_to_dict(selector_2_pb)
    for ignored_param in ignored_params:
        if ignored_param in selector_1_params:
            del selector_1_params[ignored_param]
        if ignored_param in selector_2_params:
            del selector_2_params[ignored_param]
    return selector_1_id == selector_2_id and selector_1_params == selector_2_params


def _set_selector_param_value(selector_pb, name, value):
    """
    :type selector_pb: model_pb2.YasmPanel.Selector
    """
    for param_pb in selector_pb.params:
        if param_pb.name == name:
            param_pb.value = value
            return
    raise ValueError('Param {} not found'.format(name))


def _merge_selectors(selector_1_pb, selector_2_pb):
    """
    :type selector_1_pb: model_pb2.YasmPanel.Selector
    :type selector_2_pb: model_pb2.YasmPanel.Selector
    :rtype: model_pb2.YasmPanel.Selector
    """
    selector_1_id, selector_1_params = _selector_to_dict(selector_1_pb)
    selector_2_id, selector_2_params = _selector_to_dict(selector_2_pb)
    assert selector_1_id == selector_2_id
    merged_selector_pb = model_pb2.YasmPanel.Selector(id=selector_1_id)
    for name, selector_1_value in six.iteritems(selector_1_params):
        assert name in selector_2_params
        selector_2_value = selector_2_params[name]
        if selector_1_value == selector_2_value:
            merged_selector_value = selector_1_value
        else:
            merged_selector_value = ','.join(sorted(set(selector_1_value.split(',')) |
                                                    set(selector_2_value.split(','))))
        merged_selector_pb.params.add(name=name, value=merged_selector_value)
    return merged_selector_pb


def get_namespace_yasm_tabs(namespace_id, balancer_aspects_set_pbs):
    """
    :type namespace_id: str
    :type balancer_aspects_set_pbs: list[model_pb2.BalancerAspectsSet]
    """
    yasm_tab_pbs = []

    common_yasm_panel_by_ids = OrderedDict()
    common_yasm_panels_cond_pb = model_pb2.Condition(status='True')
    common_report_uuid_occurrences = {}
    common_yasm_panel_pb_by_report_uuids = {}

    for i, balancer_aspects_set_pb in enumerate(balancer_aspects_set_pbs):
        balancer_id = balancer_aspects_set_pb.meta.balancer_id
        balancer_aspects_set_content_pb = util.clone_pb(balancer_aspects_set_pb.content)

        yasm_panel_pbs, yasm_panel_pb_by_report_uuids, report_uuid_occurrences, cond_pb = get_balancer_yasm_panels(
            balancer_id, balancer_aspects_set_content_pb)

        report_uuid_by_yasm_panel_ids = OrderedDict()
        for report_uuid, yasm_panel_pb in six.iteritems(yasm_panel_pb_by_report_uuids):
            report_uuid_by_yasm_panel_ids[yasm_panel_pb.id] = report_uuid

        for report_uuid, occurrences in six.iteritems(report_uuid_occurrences):
            yasm_panel_pb = yasm_panel_pb_by_report_uuids[report_uuid]
            yasm_panel_pb.desc = ('Statistics from "report" module with uuid "{}" '
                                  'from {}'.format(report_uuid, _format_occurrences(occurrences)))

        yasm_tab_pb = model_pb2.NamespaceUiAspectsContent.YasmTab()
        yasm_tab_pb.id = balancer_id
        yasm_tab_pb.panels.extend(yasm_panel_pbs)
        yasm_tab_pb.condition.CopyFrom(cond_pb)
        yasm_tab_pbs.append(yasm_tab_pb)

        common_yasm_panels_cond_pb = _get_worst_condition(cond_pb, common_yasm_panels_cond_pb)
        if i == 0:
            for yasm_panel_pb in yasm_panel_pbs:
                common_yasm_panel = util.clone_pb(yasm_panel_pb)
                common_yasm_panel_by_ids[yasm_panel_pb.id] = common_yasm_panel
                if yasm_panel_pb.id in report_uuid_by_yasm_panel_ids:
                    report_uuid = report_uuid_by_yasm_panel_ids[yasm_panel_pb.id]
                    common_yasm_panel_pb_by_report_uuids[report_uuid] = common_yasm_panel
        else:
            for yasm_panel_pb in yasm_panel_pbs:
                if yasm_panel_pb.id not in common_yasm_panel_by_ids:
                    continue
                common_yasm_panel = common_yasm_panel_by_ids[yasm_panel_pb.id]
                if _selectors_equal(yasm_panel_pb.selector, common_yasm_panel.selector,
                                    ignored_params=['fqdn', 'ctype', 'locations']):
                    merged_selector_pb = _merge_selectors(yasm_panel_pb.selector, common_yasm_panel.selector)
                    common_yasm_panel.url = _get_yasm_panel_url_from_selector(merged_selector_pb)
                    common_yasm_panel.selector.CopyFrom(merged_selector_pb)
                else:
                    del common_yasm_panel_by_ids[yasm_panel_pb.id]
        for report_uuid, occurrences in six.iteritems(report_uuid_occurrences):
            if report_uuid not in common_report_uuid_occurrences:
                common_report_uuid_occurrences[report_uuid] = occurrences
            else:
                common_report_uuid_occurrences[report_uuid] = _merge_occurrences(
                    common_report_uuid_occurrences[report_uuid], occurrences)

    if common_yasm_panel_by_ids:
        for report_uuid, occurrences in six.iteritems(common_report_uuid_occurrences):
            if report_uuid not in common_yasm_panel_pb_by_report_uuids:
                continue
            yasm_panel_pb = common_yasm_panel_pb_by_report_uuids[report_uuid]
            yasm_panel_pb.desc = ('Statistics from "report" module with uuid "{}" '
                                  'from {}'.format(report_uuid, _format_occurrences(occurrences)))

        common_yasm_tab_pb = model_pb2.NamespaceUiAspectsContent.YasmTab()
        common_yasm_tab_pb.id = 'common'
        common_yasm_tab_pb.panels.extend(six.itervalues(common_yasm_panel_by_ids))
        for panel_pb in common_yasm_tab_pb.panels:
            _set_selector_param_value(panel_pb.selector, 'fqdn', namespace_id)
            panel_pb.url = _get_yasm_panel_url_from_selector(panel_pb.selector)
        common_yasm_tab_pb.condition.CopyFrom(common_yasm_panels_cond_pb)
        yasm_tab_pbs.insert(0, common_yasm_tab_pb)
    return yasm_tab_pbs, model_pb2.Condition(status='True')


def fill_ui_balancer_aspects(balancer_id, balancer_aspects_set_content_pb):
    """
    :type balancer_id: six.text_type
    :type balancer_aspects_set_content_pb: model_pb2.BalancerAspectsSetContent
    """
    ui_aspects_content_pb = balancer_aspects_set_content_pb.ui.content
    ui_aspects_content_pb.Clear()

    url, cond_pb = get_yasm_panel_url(balancer_id, balancer_aspects_set_content_pb)
    ui_aspects_content_pb.yasm_panel_url = url
    ui_aspects_content_pb.yasm_panel_url_condition.CopyFrom(cond_pb)

    yasm_panel_pbs, yasm_panel_pb_by_report_uuids, report_uuid_occurrences, cond_pb = get_balancer_yasm_panels(
        balancer_id, balancer_aspects_set_content_pb)
    for report_uuid, occurrences in six.iteritems(report_uuid_occurrences):
        yasm_panel_pb = yasm_panel_pb_by_report_uuids[report_uuid]
        yasm_panel_pb.desc = ('Statistics from "report" module with uuid "{}" '
                              'from {}'.format(report_uuid, _format_occurrences(occurrences)))
    ui_aspects_content_pb.yasm_panels.extend(yasm_panel_pbs)
    ui_aspects_content_pb.yasm_panels_condition.CopyFrom(cond_pb)

    its_location_paths, cond_pb = get_balancer_its_location_paths(balancer_aspects_set_content_pb)
    ui_aspects_content_pb.its_location_paths.extend(its_location_paths)
    ui_aspects_content_pb.its_location_paths_condition.CopyFrom(cond_pb)


def fill_ui_namespace_aspects(namespace_id, namespace_aspects_set_content_pb):
    """
    :type namespace_id: str
    :type namespace_aspects_set_content_pb: model_pb2.NamespaceAspectsSetContent
    """
    ui_aspects_content_pb = namespace_aspects_set_content_pb.ui.content
    ui_aspects_content_pb.Clear()

    balancer_aspects_set_pbs = cache.IAwacsCache.instance().list_all_balancer_aspects_sets(namespace_id=namespace_id)

    its_tabs, condition = get_namespace_its_tabs(balancer_aspects_set_pbs)
    ui_aspects_content_pb.its_tabs.extend(its_tabs)
    ui_aspects_content_pb.its_tabs_condition.CopyFrom(condition)

    yasm_tabs, condition = get_namespace_yasm_tabs(namespace_id, balancer_aspects_set_pbs)
    ui_aspects_content_pb.yasm_tabs.extend(yasm_tabs)
    ui_aspects_content_pb.yasm_tabs_condition.CopyFrom(condition)

    graph_json = namespace_aspects_set_content_pb.graph.content.inclusion_graph_json
    condition = namespace_aspects_set_content_pb.graph.status.last_successful_attempt.succeeded
    ui_aspects_content_pb.inclusion_graph_json = graph_json
    ui_aspects_content_pb.inclusion_graph_json_condition.CopyFrom(condition)
