import json
import struct

import binascii
import six
import zlib
from datetime import datetime
from sepelib.core import config as appconfig

from awacs.lib.strutils import flatten_full_id2
from awacs.model import external_clusters
from infra.awacs.proto import model_pb2


NANNY_ROBOT_LOGIN = 'nanny-robot'
SECONDS_IN_DAY = 60 * 60 * 24
AWACS_L7HEAVY_OWNER_NAME = 'AWACS'
COMMON_L7HEAVY_GROUP_ID = 'common'

LARGE_BALANCER_IDS = None
LARGE_NAMESPACE_IDS = None
YP_CLUSTERS = ['SAS', 'MAN', 'VLA', 'MYT', 'IVA', 'MAN_PRE', 'TEST_SAS']
GENCFG_DATA_CENTERS = ['SAS', 'MAN', 'VLA', 'MYT', 'IVA']
BALANCER_LOCATIONS = ['SAS', 'MAN', 'VLA', 'MYT', 'IVA', 'XDC', 'UNKNOWN']

SYSTEM_ENDPOINT_SET_ID_TEMPLATE = 'awacs-{nanny_service_id}'
NAMESPACE_LINK_TEMPLATE = 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{namespace_id}/show/'
DEV_NAMESPACE_LINK_TEMPLATE = 'https://dev-nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{namespace_id}/show/'

WEIGHTS_KNOB_ID = 'cplb_balancer_load_switch'
ITS_VALUE_PATH_TEMPLATE = 'balancer/{}/common/common/' + WEIGHTS_KNOB_ID

YP_CLUSTERS_TO_BALANCER_LOCATIONS = {
    'SAS': 'SAS',
    'MAN': 'MAN',
    'VLA': 'VLA',
    'MYT': 'MYT',
    'IVA': 'IVA',
    'MAN_PRE': 'UNKNOWN',
    'TEST_SAS': 'UNKNOWN',
}

unset = object()


def yp_cluster_to_balancer_location(yp_cluster, default=unset):
    if default is not unset:
        return YP_CLUSTERS_TO_BALANCER_LOCATIONS.get(yp_cluster, default)
    else:
        if yp_cluster not in YP_CLUSTERS_TO_BALANCER_LOCATIONS:
            raise ValueError('unknown YP cluster "{}"'.format(yp_cluster))
        return YP_CLUSTERS_TO_BALANCER_LOCATIONS[yp_cluster]


GENCFG_DCS_TO_BALANCER_LOCATIONS = {
    'SAS': 'SAS',
    'MAN': 'MAN',
    'VLA': 'VLA',
    'MYT': 'MYT',
    'IVA': 'IVA',
}


def gencfg_dc_to_balancer_location(gencfg_dc, default=unset):
    if default is not unset:
        return GENCFG_DCS_TO_BALANCER_LOCATIONS.get(gencfg_dc, default)
    else:
        if gencfg_dc not in GENCFG_DCS_TO_BALANCER_LOCATIONS:
            raise ValueError('unknown gencfg DC "{}"'.format(gencfg_dc))
        return GENCFG_DCS_TO_BALANCER_LOCATIONS[gencfg_dc]


def get_balancer_yp_cluster(balancer_pb):
    if balancer_pb.meta.location.type == balancer_pb.meta.location.YP_CLUSTER:
        return balancer_pb.meta.location.yp_cluster
    elif balancer_pb.meta.location.type == balancer_pb.meta.location.AZURE_CLUSTER:
        return external_clusters.AZURE_CLUSTERS_BY_NAME[balancer_pb.meta.location.azure_cluster].yp_cluster
    elif balancer_pb.meta.location.type == balancer_pb.meta.location.GENCFG_DC:
        return None
    else:
        raise AssertionError('Unknown balancer location type')


def get_balancer_location(balancer_pb, logger=None):
    if balancer_pb.meta.location.type in (balancer_pb.meta.location.YP_CLUSTER, balancer_pb.meta.location.AZURE_CLUSTER):
        yp_cluster = get_balancer_yp_cluster(balancer_pb).upper()
        balancer_location = yp_cluster_to_balancer_location(yp_cluster, default=None)
        if balancer_location is None:
            if logger is not None:
                logger.warn('Unknown YP cluster "{}"?'.format(balancer_pb.meta.location.yp_cluster))
            balancer_location = 'UNKNOWN'
    elif balancer_pb.meta.location.type == balancer_pb.meta.location.GENCFG_DC:
        balancer_location = gencfg_dc_to_balancer_location(balancer_pb.meta.location.gencfg_dc.upper(),
                                                           default=None)
        if balancer_location is None:
            if logger is not None:
                logger.warn('Unknown gencfg DC "{}"?'.format(balancer_pb.meta.location.gencfg_dc))
            balancer_location = 'UNKNOWN'
    else:
        if logger is not None:
            logger.warn('XDC balancer found')
        balancer_location = 'XDC'
    return balancer_location


def is_large_namespace(namespace_id):
    global LARGE_NAMESPACE_IDS
    if LARGE_NAMESPACE_IDS is None:
        LARGE_NAMESPACE_IDS = frozenset(appconfig.get_value('run.large_namespace_ids', []))
    return namespace_id in LARGE_NAMESPACE_IDS


def is_large_balancer(namespace_id, balancer_id):
    global LARGE_BALANCER_IDS
    if LARGE_BALANCER_IDS is None:
        LARGE_BALANCER_IDS = frozenset((ns, b) for (ns, b) in appconfig.get_value('run.large_balancer_ids', []))
    return is_large_namespace(namespace_id) or (namespace_id, balancer_id) in LARGE_BALANCER_IDS


def is_upstream_internal(upstream_id):
    return upstream_id.startswith('_')


def int_to_hex_bytes(i):
    return binascii.hexlify(struct.pack('>I', i))


def crc32(data, crc=0):
    """
    :type data: six.binary_type
    :type crc: int
    :rtype: int
    """
    # https://docs.python.org/3/library/zlib.html#zlib.crc32
    # > To generate the same numeric value across all Python versions and
    # > platforms, use crc32(data) & 0xffffffff.
    return zlib.crc32(data, crc) & 0xffffffff


class Diff(object):
    __slots__ = ('updated', 'added', 'removed')

    def __init__(self, updated=frozenset(), added=frozenset(), removed=frozenset()):
        """
        :type updated: set
        :type added: set
        :type removed: set
        """
        self.updated = updated
        self.added = added
        self.removed = removed


def version_to_str(version):
    if version is None:
        return 'NONE'
    else:
        rv = version.version[:8]
        if version.deleted:
            rv = 'DEL@' + rv
        return rv


def versions_dict_to_str(d):
    """
    :type d: dict[(six.text_type, six.text_type),
                  Union[BalancerVersion | DomainVersion | UpstreamVersion | BackendVersion |
                  EndpointSetVersion | KnobVersion | DnsRecordVersion]
                 ]
    :rtype: six.text_type
    """
    return json.dumps({flatten_full_id2(full_id): version_to_str(version)
                       for full_id, version in six.iteritems(d)})


def clone_pb(pb):
    cloned_pb = type(pb)()
    cloned_pb.CopyFrom(pb)
    return cloned_pb


def find_rev_status_by_id(status_pbs, revision_id):
    """
    :type status_pbs: list[UpstreamRevisionStatusPerBalancer | BackendRevisionStatusPerBalancer |
                           EndpointSetRevisionStatusPerBalancer | KnobRevisionStatusPerBalancer |
                           CertificateRevisionStatusPerBalancer | DomainRevisionStatusPerBalancer]
    :type revision_id: six.text_type | six.text_type
    :rtype: UpstreamRevisionStatusPerBalancer | DomainRevisionStatusPerBalancer | BackendRevisionStatusPerBalancer |
            EndpointSetRevisionStatusPerBalancer | KnobRevisionStatusPerBalancer |
            CertificateRevisionStatusPerBalancer
    """
    for status_pb in status_pbs:
        if status_pb.id == revision_id:
            return status_pb


def find_rev_status_by_revision_id(status_pbs, revision_id):
    for status_pb in status_pbs:
        if status_pb.revision_id == revision_id:
            return status_pb


def newer(curr_version, version_to_check):
    """
    :type curr_version: Version
    :type version_to_check: Version
    :rtype: Version
    """
    if curr_version is None:
        return version_to_check
    else:
        if curr_version.ctime < version_to_check.ctime:
            return version_to_check
        else:
            return curr_version


def omit_duplicate_items(items):
    rv = []
    seen = set()
    for item in items:
        if item not in seen:
            rv.append(item)
            seen.add(item)
    return rv


def omit_duplicate_items_from_auth(auth_pb):
    owners_pb = auth_pb.staff.owners
    logins = omit_duplicate_items(owners_pb.logins)
    if len(logins) != len(owners_pb.logins):
        owners_pb.ClearField('logins')
        owners_pb.logins.extend(logins)

    group_ids = omit_duplicate_items(owners_pb.group_ids)
    if len(group_ids) != len(owners_pb.group_ids):
        owners_pb.ClearField('group_ids')
        owners_pb.group_ids.extend(group_ids)


def set_condition(condition_pb, updated_condition_pb, author, utcnow):
    """
    :type condition_pb: model_pb2.BoolCondition | model_pb2.TimedBoolCondition | model_pb2.PausedCondition
    :type updated_condition_pb: model_pb2.BoolCondition | model_pb2.TimedBoolCondition | model_pb2.PausedCondition
    """
    if isinstance(condition_pb, (model_pb2.BoolCondition, model_pb2.PausedCondition)):
        need_update = condition_pb.value != updated_condition_pb.value
    elif isinstance(condition_pb, model_pb2.TimedBoolCondition):
        # It is hard to get and set nanos in UI, so sometimes it will set it to zero.
        # Let's compare only seconds to avoid author/mtime changes
        need_update = (condition_pb.value != updated_condition_pb.value or
                       condition_pb.not_before.seconds != updated_condition_pb.not_before.seconds or
                       condition_pb.not_after.seconds != updated_condition_pb.not_after.seconds)
    else:
        raise AssertionError()

    if not need_update:
        return

    condition_pb.Clear()
    condition_pb.MergeFrom(updated_condition_pb)
    condition_pb.mtime.FromDatetime(utcnow)
    condition_pb.author = author


def check_condition(condition_pb, utcnow):
    """
    :type condition_pb: model_pb2.BoolCondition | model_pb2.TimedBoolCondition | model_pb2.PausedCondition
    :rtype: bool
    """
    if not condition_pb.value:
        return False
    if isinstance(condition_pb, model_pb2.TimedBoolCondition):
        if utcnow < condition_pb.not_before.ToDatetime():
            return False
        if condition_pb.not_after.seconds and utcnow > condition_pb.not_after.ToDatetime():
            # Consider 0 as +inf
            return False
    return True


def make_system_endpoint_set_id(nanny_service_id):
    return SYSTEM_ENDPOINT_SET_ID_TEMPLATE.format(nanny_service_id=nanny_service_id)


def get_diff_between_dicts(d1, d2):
    """
    :return: a sorted list of items from d1 that are different from d2, and a similar list for d2
    """
    values_1 = {}
    values_2 = {}
    for k, _ in set(d1.items()).symmetric_difference(set(d2.items())):
        values_1[k] = d1.get(k)
        values_2[k] = d2.get(k)
    return sorted(values_1.items()), sorted(values_2.items())


def fill_default_order(meta_pb, order_content_pb, target_pb, login):
    utcnow = datetime.utcnow()
    target_pb.meta.CopyFrom(meta_pb)
    target_pb.order.content.CopyFrom(order_content_pb)
    target_pb.meta.ctime.FromDatetime(utcnow)
    target_pb.meta.mtime.FromDatetime(utcnow)
    target_pb.meta.author = login
    target_pb.spec.incomplete = True


def update_spec_in_pb(target_pb, spec_pb, new_version, comment, login, utcnow=None):
    if target_pb.spec == spec_pb:
        return False
    if utcnow is None:
        utcnow = datetime.utcnow()
    target_pb.spec.CopyFrom(spec_pb)
    target_pb.meta.version = new_version
    target_pb.meta.comment = comment
    target_pb.meta.author = login
    target_pb.meta.mtime.FromDatetime(utcnow)
    return True


def removeprefix(text, prefix):
    """
    str.removeprefix() from python3.9

    :type text: six.text_type
    :type prefix: six.text_type
    :rtype: six.text_type
    """
    if text.startswith(prefix):
        return text[len(prefix):]
    return text


def get_namespace_link(namespace_id):
    if appconfig.get_value('run.production', False):
        return NAMESPACE_LINK_TEMPLATE.format(namespace_id=namespace_id)
    return DEV_NAMESPACE_LINK_TEMPLATE.format(namespace_id=namespace_id)
