# coding: utf-8
import re
import logging
from six.moves.urllib import parse as urlparse

import six
from sepelib.core import config as appconfig

from awacs import yamlparser
from awacs.lib.startrekclient import IStartrekClient
from awacs.model import errors, cache
from awacs.model.balancer.modes import L7FastBalancerModeValidator
from infra.awacs.proto import modules_pb2, model_pb2
from awacs.wrappers.base import Holder, ANY_MODULE, Chain, DEFAULT_CTX, ValidationCtx
from awacs.wrappers.errors import ValidationError as ConfigValidationError
from awacs.wrappers.l7upstreammacro import L7UpstreamMacro
from awacs.wrappers.l7macro import L7Macro
from awacs.wrappers.main import (RegexpSection, RegexpPathSection, PrefixPathRouterSection,
                                 RegexpHostSection, L7FastUpstreamMacro, L7FastSitemapUpstreamMacro)


log = logging.getLogger(__name__)

_SECTION_TYPES = (RegexpSection, RegexpPathSection, PrefixPathRouterSection, RegexpHostSection)
L7_FAST_ROUTE_PATTERN = '^/[a-z0-9%][a-z0-9-/%]+$'
L7_FAST_ROUTE_MAX_LENGTH = 100


def _parse_config_yaml(yaml, field_name):
    """
    :param six.text_type yaml:
    :param six.text_type field_name:
    :rtype: modules_pb2.Holder
    """
    try:
        return yamlparser.parse(modules_pb2.Holder, yaml, ensure_ascii=True)
    except yamlparser.Error as e:
        raise errors.ValidationError('{}: {}'.format(field_name, e))


def get_yaml(config_pb, dump_fn=yamlparser.dump):
    """
    :type config_pb: modules_pb2.Holder
    :rtype: str
    """
    assert isinstance(config_pb, modules_pb2.Holder)
    yml = dump_fn(config_pb)
    parsed_dumped_config_pb = yamlparser.parse(modules_pb2.Holder, yml)
    if config_pb != parsed_dumped_config_pb:
        msg = ('awacs encountered an error during protobuf to yaml conversion. '
               'Please report this error to https://st.yandex-team.ru/SWAT-4140. '
               'Sorry for the inconvenience.')
        log.warn(msg)
        raise errors.InternalError(msg)
    return yml


def validate_balancer_config(holder_pb, mode, ctx=DEFAULT_CTX, field_name='spec.yandex_balancer.config'):
    """
    :type holder_pb: model_pb2.Holder
    :type mode: model_pb2.YandexBalancerSpec.FULL_MODE |
          model_pb2.YandexBalancerSpec.EASY_MODE
    :type field_name: six.text_type
    :type ctx: ValidationCtx
    :rtype awacs.wrappers.base.Holder
    :raises: errors.ValidationError
    """
    try:
        h = Holder(holder_pb)
        h.validate(ctx=ctx)
    except ConfigValidationError as e:
        if field_name:
            raise errors.ValidationError('{}: {}'.format(field_name, six.text_type(e)))
        else:
            raise errors.ValidationError(six.text_type(e))
    if mode == model_pb2.YandexBalancerSpec.EASY_MODE:
        validate_easy_mode_balancer_config_structure(h, ctx=ctx, field_name=field_name)
    elif mode == model_pb2.YandexBalancerSpec.FULL_MODE:
        validate_full_mode_balancer_config_structure(h, ctx=ctx, field_name=field_name)
    else:
        raise errors.ValidationError('Unknown mode: {}'.format(mode))
    return h


def validate_and_parse_yaml_balancer_config(spec_pb, full_balancer_id, namespace_pb):
    """
    :type spec_pb: model_pb2.BalancerSpec
    :type full_balancer_id: (six.text_type, six.text_type)
    :type namespace_pb: model_pb2.Namespace
    :rtype awacs.wrappers.base.Holder
    """
    if spec_pb.type != model_pb2.YANDEX_BALANCER:
        raise errors.ValidationError('spec.type must be YANDEX_BALANCER')

    yandex_pb = spec_pb.yandex_balancer
    if not yandex_pb.HasField('config') and not yandex_pb.yaml:
        raise errors.ValidationError('spec.yandex_balancer: "config" or "yaml" '
                                     'must be specified')

    if yandex_pb.HasField('config') and yandex_pb.template_engine:
        raise errors.ValidationError('spec.yandex_balancer: "config" and "template_engine" '
                                     'fields are mutually exclusive')

    if yandex_pb.yaml:
        if yandex_pb.template_engine == model_pb2.NONE:
            config_pb = _parse_config_yaml(yandex_pb.yaml, 'spec.yandex_balancer.yaml')
            yandex_pb.config.CopyFrom(config_pb)
        elif yandex_pb.template_engine == model_pb2.DYNAMIC_JINJA2:
            raise errors.ValidationError('Template engine DYNAMIC_JINJA2 is not supported anymore')
    elif yandex_pb.HasField('config') and yandex_pb.template_engine == model_pb2.NONE:
        yandex_pb.yaml = get_yaml(yandex_pb.config)

    if yandex_pb.HasField('config'):
        ctx = ValidationCtx.create_ctx_with_config_type_balancer(namespace_pb=namespace_pb,
                                                                 full_balancer_id=full_balancer_id,
                                                                 balancer_spec_pb=spec_pb,
                                                                 )
        return validate_balancer_config(
            yandex_pb.config,
            mode=yandex_pb.mode,
            ctx=ctx,
            field_name='spec.yandex_balancer.yaml' if yandex_pb.yaml else 'spec.yandex_balancer.config')


def validate_full_mode_config_structure(upstream_id, config, field_name, ctx=DEFAULT_CTX):
    """
    :type upstream_id: str
    :type config: awacs.wrappers.base.Holder
    :type field_name: six.text_type
    :type ctx: ValidationCtx
    :raises: errors.ValidationError
    """
    if isinstance(config.module, L7UpstreamMacro):
        raise errors.ValidationError(
            u'"{}": full_mode: config must not start with l7_upstream_macro'.format(field_name))


def validate_easy_mode2_config_structure(upstream_id, config, field_name, ctx=DEFAULT_CTX):
    """
    :type upstream_id: str
    :type config: awacs.wrappers.base.Holder
    :type field_name: six.text_type
    :type ctx: ValidationCtx
    :raises: errors.ValidationError
    """
    if not isinstance(config.module, L7UpstreamMacro):
        raise errors.ValidationError(
            u'"{}": easy_mode: config must start with l7_upstream_macro, '
            u'not {}'.format(field_name, config.module_name))

    if config.module.pb.id != upstream_id:
        raise errors.ValidationError(
            u'"{}": easy_mode: l7_upstream_macro.id must be equal '
            u'to upstream id ("{}")'.format(field_name, upstream_id))


def validate_l7_fast_route(route):
    if not re.match(L7_FAST_ROUTE_PATTERN, route):
        raise errors.ValidationError('must be matched by {}'.format(L7_FAST_ROUTE_PATTERN))
    if '//' in route:
        raise errors.ValidationError('must not contain two slashes in a row')
    if len(route) >= L7_FAST_ROUTE_MAX_LENGTH:
        raise errors.ValidationError(
            'must be of length less than or equal to {} characters'.format(L7_FAST_ROUTE_MAX_LENGTH))


def validate_easy_mode_balancer_config_structure(config, field_name, ctx=DEFAULT_CTX):
    """
    :type config: awacs.wrappers.base.Holder
    :type field_name: six.text_type
    :type ctx: ValidationCtx
    :raises: errors.ValidationError
    """
    if not isinstance(config.module, L7Macro):
        raise errors.ValidationError('"{}": easy_mode: config must start with l7_macro, '
                                     'not {}'.format(field_name, config.module_name))


def validate_full_mode_balancer_config_structure(config, field_name, ctx=DEFAULT_CTX):
    """
    :type config: awacs.wrappers.base.Holder
    :type field_name: six.text_type
    :type ctx: ValidationCtx
    :raises: errors.ValidationError
    """
    if isinstance(config.module, L7Macro):
        raise errors.ValidationError('"{}": easy_mode must be selected if config starts with '
                                     'l7_macro'.format(field_name))


def validate_l7_fast_mode_config_structure(upstream_id, mode, config, field_name, ctx=DEFAULT_CTX):
    """
    :type upstream_id: str
    :param mode: model_pb2.YandexBalancerUpstreamSpec.L7_FAST_MODE |
                 model_pb2.YandexBalancerUpstreamSpec.L7_FAST_SITEMAP_MODE
    :type config: awacs.wrappers.base.Holder
    :type field_name: six.text_type
    :type ctx: ValidationCtx
    :raises: errors.ValidationError
    """
    try:
        if not isinstance(config.module, PrefixPathRouterSection):
            raise errors.ValidationError('l7-fast: {} is not allowed, '
                                         'please use prefix_path_router_section'.format(config.module_name))

        module = config.module
        try:
            validate_l7_fast_route(config.module.pb.route)
        except errors.ValidationError as e:
            raise errors.ValidationError('l7-fast: prefix_path_router_section\'s route {}'.format(e))

        next_module = next(module.nested.walk_chain())
        if mode == model_pb2.YandexBalancerUpstreamSpec.L7_FAST_MODE:
            expected_module_cls = L7FastUpstreamMacro
            expected_module_name = 'l7_fast_upstream_macro'
        elif mode == model_pb2.YandexBalancerUpstreamSpec.L7_FAST_SITEMAP_MODE:
            expected_module_cls = L7FastSitemapUpstreamMacro
            expected_module_name = 'l7_fast_sitemap_upstream_macro'
        else:
            raise AssertionError('unexpected mode')

        if not next_module or not isinstance(next_module, expected_module_cls):
            raise errors.ValidationError('l7-fast: only {} is allowed'.format(expected_module_name))
        if not re.match(expected_module_cls.ID_PATTERN, upstream_id):
            raise errors.ValidationError('l7-fast: upstream id must be matched '
                                         'by {}'.format(expected_module_cls.ID_PATTERN))
        if next_module.pb.id != upstream_id:
            raise errors.ValidationError(
                'l7-fast: {} id must be the same as upstream id'.format(expected_module_name))

    except errors.ValidationError as e:
        raise errors.ValidationError('"{}": {}'.format(field_name, six.text_type(e)))


def _validate_upstream_config(upstream_id, config, ctx=DEFAULT_CTX):
    """
    :type upstream_id: six.text_type
    :type config: awacs.wrappers.base.Holder
    :type ctx: ValidationCtx
    :raises: awacs.wrappers.errors.ValidationError
    """
    preceding_modules = [ANY_MODULE]
    if config.module:
        if isinstance(config.module, _SECTION_TYPES):
            config.module.validate(key=upstream_id, ctx=ctx, preceding_modules=preceding_modules)
            return
    else:
        fst = config.chain.modules[0]
        if isinstance(fst.module, _SECTION_TYPES):
            if not fst.module.nested:
                fst.module.nested = Chain(config.chain.modules[1:])
            fst.module.validate(key=upstream_id, ctx=ctx, preceding_modules=preceding_modules)
            return
    config.validate(ctx=ctx, preceding_modules=preceding_modules)


def validate_l7_fast_secaudit(upstream_spec_pb, comment):
    """
    :type upstream_spec_pb: model_pb2.UpstreamSpec
    :type comment: six.text_type
    """
    startrek_client = IStartrekClient.instance()
    secaudit_ticket_ids = re.findall(r'SECAUDIT-\d+', comment)
    if not secaudit_ticket_ids:
        raise errors.ValidationError('Please specify a valid SECAUDIT ticket identifier in the comment')
    elif len(secaudit_ticket_ids) > 1:
        raise errors.ValidationError('Please specify a single SECAUDIT ticket identifier in the comment')
    else:
        secaudit_ticket_id = secaudit_ticket_ids[0]
    try:
        ticket = startrek_client.issues[secaudit_ticket_id]
    except Exception:
        log.exception('Failed to get %s', secaudit_ticket_id)
        raise errors.ValidationError(
            'Failed to read {} from Startrek. Please make sure it exists or try again '
            'later. If the error persists, contact the support.'.format(secaudit_ticket_id))
    else:
        upstream_route = L7FastBalancerModeValidator.get_l7_fast_mode_upstream_route(upstream_spec_pb)

        status = ticket.status.key
        production_url = getattr(ticket, 'productionURL', None)
        resolution = ticket.resolution.key if ticket.resolution else None

        try:
            result = urlparse.urlparse(production_url)
        except Exception:
            is_matched = False
        else:
            production_path = result.path
            ticket_routes = [
                production_path.strip('/').split('/', 1)[0],  # legacy logic from around 2014
                production_path.strip('/'),  # more sensible logic required in SWAT-5770
            ]
            # AWACS-726
            upstream_route_with_slash = upstream_route.strip('/') + '/'
            production_path_with_slash = production_path.strip('/') + '/'
            is_matched = upstream_route in ticket_routes or upstream_route_with_slash.startswith(production_path_with_slash)
        if status == 'approved':
            if not is_matched:
                raise errors.ValidationError(
                    '{} is closed, but its production URL ({}) '
                    'does not match upstream route ({})'.format(secaudit_ticket_id, production_url, upstream_route))
        elif status == 'closed':
            if resolution == 'fixed':
                if not is_matched:
                    raise errors.ValidationError(
                        '{} is closed, but its production URL ({}) '
                        'does not match upstream route ({})'.format(secaudit_ticket_id, production_url, upstream_route))
            else:
                raise errors.ValidationError(
                    '{} is closed, but not fixed (resolved as {})'.format(secaudit_ticket_id, resolution))
        else:
            raise errors.ValidationError('{} is not closed'.format(secaudit_ticket_id))


def validate_upstream_config(upstream_id, holder_pb, ctx=DEFAULT_CTX, field_name='spec.yandex_balancer.config'):
    """
    :type upstream_id: six.text_type
    :type holder_pb: model_pb2.Holder
    :type field_name: six.text_type
    :type ctx: ValidationCtx
    :rtype: Holder
    :raises: errors.ValidationError
    """
    holder = Holder(holder_pb)
    try:
        _validate_upstream_config(upstream_id, holder, ctx=ctx)
    except ConfigValidationError as e:
        raise errors.ValidationError('"{}": {}'.format(field_name, six.text_type(e)))
    else:
        return holder


def validate_and_parse_yaml_upstream_config(
        namespace_id, upstream_id, spec_pb, curr_spec_pb=None, ctx=DEFAULT_CTX,
        comment=''):
    """
    :type namespace_id: six.text_type
    :type upstream_id: six.text_type
    :type spec_pb: model_pb2.UpstreamSpec
    :type curr_spec_pb: model_pb2.UpstreamSpec | None
    :type comment: six.text_type
    :type ctx: ValidationCtx
    :rtype awacs.wrappers.base.Holder
    :raises: errors.ValidationError
    """
    l7_fast_namespaces = appconfig.get_value('run.l7_fast_namespaces', default=[])
    is_namespace_l7_fast = False
    is_namespace_l7_fast_production = False
    for item in l7_fast_namespaces:
        if item['id'] == namespace_id:
            is_namespace_l7_fast = True
            if item['production']:
                is_namespace_l7_fast_production = True

    if spec_pb.type != model_pb2.YANDEX_BALANCER:
        raise errors.ValidationError('"spec.type" must be YANDEX_BALANCER')
    y_b_pb = spec_pb.yandex_balancer

    if (is_namespace_l7_fast and y_b_pb.mode not in (y_b_pb.L7_FAST_MODE, y_b_pb.L7_FAST_SITEMAP_MODE) and
            upstream_id not in L7FastBalancerModeValidator.SPECIAL_UPSTREAM_IDS):
        raise errors.ValidationError('"spec.yandex_balancer.mode": must be L7_FAST_MODE or L7_FAST_SITEMAP_MODE')

    if y_b_pb.mode in (y_b_pb.FULL_MODE, y_b_pb.EASY_MODE2):
        if y_b_pb.yaml:
            field_name = 'spec.yandex_balancer.yaml'
            y_b_pb.config.CopyFrom(_parse_config_yaml(y_b_pb.yaml, 'spec.yandex_balancer.yaml'))
        elif y_b_pb.HasField('config'):
            field_name = 'spec.yandex_balancer.config'
            y_b_pb.yaml = get_yaml(y_b_pb.config)
        else:
            raise errors.ValidationError('either "spec.yandex_balancer.config" or "spec.yandex_balancer.yaml" '
                                         'must be specified')
    elif y_b_pb.mode in (y_b_pb.L7_FAST_MODE, y_b_pb.L7_FAST_SITEMAP_MODE):
        if y_b_pb.HasField('config') and y_b_pb.yaml:
            raise errors.ValidationError('"spec.yandex_balancer": at most one of '
                                         '"config" or "yaml" must be specified when '
                                         'L7_FAST_MODE or L7_FAST_SITEMAP_MODE is on')
        elif y_b_pb.HasField('config'):
            field_name = 'spec.yandex_balancer.config'
            y_b_pb.yaml = get_yaml(y_b_pb.config)
        elif y_b_pb.yaml:
            field_name = 'spec.yandex_balancer.yaml'
            y_b_pb.config.CopyFrom(_parse_config_yaml(y_b_pb.yaml, 'spec.yandex_balancer.yaml'))
        else:
            raise errors.ValidationError('either "spec.yandex_balancer.config" or "spec.yandex_balancer.yaml" '
                                         'must be specified')
    else:
        raise errors.ValidationError('"spec.yandex_balancer.type": specified type is not supported')

    holder = validate_upstream_config(upstream_id, y_b_pb.config, ctx=ctx, field_name=field_name)
    if y_b_pb.mode == y_b_pb.FULL_MODE:
        validate_full_mode_config_structure(upstream_id, holder, ctx=ctx, field_name=field_name)
    if y_b_pb.mode == y_b_pb.EASY_MODE2:
        validate_easy_mode2_config_structure(upstream_id, holder, ctx=ctx, field_name=field_name)
    elif y_b_pb.mode in (y_b_pb.L7_FAST_MODE, y_b_pb.L7_FAST_SITEMAP_MODE):
        validate_l7_fast_mode_config_structure(upstream_id, y_b_pb.mode, holder,
                                               ctx=ctx, field_name=field_name)
        if curr_spec_pb is None:
            # upstream is being created
            if is_namespace_l7_fast_production and y_b_pb.mode == y_b_pb.L7_FAST_MODE:
                validate_l7_fast_secaudit(spec_pb, comment)
        else:
            try:
                curr_holder = validate_upstream_config(upstream_id, curr_spec_pb.yandex_balancer.config, ctx=ctx)
                validate_l7_fast_mode_config_structure(upstream_id, y_b_pb.mode, curr_holder,
                                                       ctx=ctx, field_name=field_name)
            except errors.ValidationError:
                log.warn('Unexpected validation error: current upstream config is not valid -- '
                         'skipping L7-fast route check, please investigate this warning!', exc_info=True)
            else:
                # after all the validation we know that both {holder,curr_holder}.module are PrefixPathRouterSection
                assert isinstance(holder.module, PrefixPathRouterSection)
                assert isinstance(curr_holder.module, PrefixPathRouterSection)
                if holder.module.pb.route != curr_holder.module.pb.route:
                    raise errors.ValidationError('l7-fast: prefix_path_router_section\'s route can not be changed')
    return holder
