import inject

from awacs.lib.zk_storage import NodeAlreadyExistsError, ZkStorageClient, path_with_prefix
from awacs.model import errors, objects
from infra.awacs.proto import model_pb2


class IZkStorage(object):
    @classmethod
    def instance(cls):
        """
        :rtype: ZkStorage
        """
        return inject.instance(cls)


class ZkStorage(IZkStorage):
    def __init__(self, zk_client, prefix=''):
        """
        :param awacs.lib.zookeeper_client.ZookeeperClient zk_client:
        :param six.text_type prefix:
        """
        self._prefix = prefix
        self.namespace_storage = self._get_zk_client(zk_client, objects.NamespaceDescriptor)
        self.component_storage = self._get_zk_client(zk_client, objects.ComponentDescriptor)
        self.knob_storage = self._get_zk_client(zk_client, objects.KnobDescriptor)
        self.balancer_storage = self._get_zk_client(zk_client, objects.L7BalancerDescriptor)
        self.balancer_state_storage = self._get_zk_client(zk_client, objects.L7BalancerStateDescriptor)
        self.balancer_aspects_set_storage = self._get_zk_client(zk_client, objects.L7BalancerAspectSetDescriptor)
        self.namespace_aspects_set_storage = self._get_zk_client(zk_client, objects.NamespaceAspectSetDescriptor)
        self.l3_balancer_storage = self._get_zk_client(zk_client, objects.L3BalancerDescriptor)
        self.l3_balancer_state_storage = self._get_zk_client(zk_client, objects.L3BalancerStateDescriptor)
        self.upstream_storage = self._get_zk_client(zk_client, objects.UpstreamDescriptor)
        self.domain_storage = self._get_zk_client(zk_client, objects.DomainDescriptor)
        self.domain_operation_storage = self._get_zk_client(zk_client, objects.DomainOperationDescriptor)
        self.balancer_operation_storage = self._get_zk_client(zk_client, objects.L7BalancerOperationDescriptor)
        self.backend_storage = self._get_zk_client(zk_client, objects.BackendDescriptor)
        self.endpoint_set_storage = self._get_zk_client(zk_client, objects.EndpointSetDescriptor)
        self.dns_record_storage = self._get_zk_client(zk_client, objects.DnsRecordDescriptor)
        self.dns_record_operation_storage = self._get_zk_client(zk_client, objects.DnsRecordOperationDescriptor)
        self.dns_record_state_storage = self._get_zk_client(zk_client, objects.DnsRecordStateDescriptor)
        self.name_server_storage = self._get_zk_client(zk_client, objects.NameServerDescriptor)
        self.cert_storage = self._get_zk_client(zk_client, objects.CertDescriptor)
        self.cert_renewal_storage = self._get_zk_client(zk_client, objects.CertRenewalDescriptor)

    def _get_zk_client(self, zk_client, descriptor):
        return ZkStorageClient(zk_client, self._path_with_prefix(descriptor.zk_prefix), descriptor.codec)

    def _path_with_prefix(self, path):
        if self._prefix:
            return path_with_prefix(self._prefix, path)
        else:
            return path

    def does_component_exist(self, component_type, version, sync=False):
        """
        :type component_type: six.text_type
        :type version: six.text_type
        :param bool sync:
        :rtype: bool
        """
        zk_path = objects.ComponentDescriptor.uid_to_zk_path(component_type, version)
        if sync:
            self.component_storage.sync(zk_path)
        return self.component_storage.exists(zk_path)

    def get_component(self, component_type, version, sync=False):
        """
        :type component_type: model_pb2.ComponentMeta.Type
        :type version: six.text_type
        :param bool sync:
        :rtype: model_pb2.Component | None
        """
        component_type_str = model_pb2.ComponentMeta.Type.Name(component_type)
        zk_path = objects.ComponentDescriptor.uid_to_zk_path(component_type_str, version)
        if sync:
            self.component_storage.sync(zk_path)
        return self.component_storage.get(zk_path)

    def must_get_component(self, component_type, version, sync=False):
        """
        :type component_type: model_pb2.ComponentMeta.Type
        :type version: six.text_type
        :param bool sync:
        :rtype: model_pb2.Component
        :raises: errors.NotFoundError
        """
        component_pb = self.get_component(component_type, version, sync=sync)
        if component_pb is None:
            component_type_str = model_pb2.ComponentMeta.Type.Name(component_type)
            raise errors.NotFoundError('Component "{}" of version "{}" does not exist'.format(component_type_str, version))
        return component_pb

    def draft_component(self, component_type, version, component_pb):
        """
        :type component_type: six.text_type
        :type version: six.text_type
        :type component_pb: model_pb2.Component
        :rtype: model_pb2.Component
        :raises: errors.ConflictError
        """
        assert isinstance(component_pb, model_pb2.Component)
        assert component_pb.meta.type == model_pb2.ComponentMeta.Type.Value(component_type)
        assert component_pb.meta.version == version
        zk_path = objects.ComponentDescriptor.uid_to_zk_path(component_type, version)
        try:
            return self.component_storage.create(zk_path, component_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Component "{}" of version "{}" already exists'.format(component_type, version))

    def update_component(self, component_type, version, component_pb=None):
        """
        :type component_type: six.text_type
        :type version: six.text_type
        :type component_pb: model_pb2.Component | None
        :rtype: list[model_pb2.Component]
        :raises: errors.NotFoundError
        """
        assert component_pb is None or (isinstance(component_pb, model_pb2.Component) and
                                        component_pb.meta.type == model_pb2.ComponentMeta.Type.Value(component_type) and
                                        component_pb.meta.version == version)
        zk_path = objects.ComponentDescriptor.uid_to_zk_path(component_type, version)
        for component_pb in self.component_storage.guaranteed_update(zk_path, obj=component_pb):
            if component_pb is None:
                raise errors.NotFoundError('Component "{}" of version "{}" does not exist'.format(component_type,
                                                                                                  version))
            yield component_pb

    def remove_component(self, component_type, version):
        """
        :type component_type: six.text_type
        :type version: six.text_type
        """
        zk_path = objects.ComponentDescriptor.uid_to_zk_path(component_type, version)
        self.component_storage.remove(zk_path)

    def does_namespace_exist(self, namespace_id):
        """
        :type namespace_id: six.text_type
        :rtype: bool
        """
        zk_path = objects.NamespaceDescriptor.uid_to_zk_path(namespace_id)
        return self.namespace_storage.exists(zk_path)

    def get_namespace(self, namespace_id, sync=False):
        """
        :type namespace_id: six.text_type
        :param bool sync:
        :rtype: model_pb2.Namespace | None
        """
        zk_path = objects.NamespaceDescriptor.uid_to_zk_path(namespace_id)
        if sync:
            self.namespace_storage.sync(zk_path)
        return self.namespace_storage.get(zk_path)

    def must_get_namespace(self, namespace_id, sync=False):
        """
        :type namespace_id: six.text_type
        :param bool sync:
        :rtype: model_pb2.Namespace
        :raises: errors.NotFoundError
        """
        namespace_pb = self.get_namespace(namespace_id, sync=sync)
        if namespace_pb is None:
            raise errors.NotFoundError('Namespace "{}" does not exist'.format(namespace_id))
        return namespace_pb

    def create_namespace(self, namespace_id, namespace_pb):
        """
        :type namespace_id: six.text_type
        :type namespace_pb: model_pb2.Namespace | None
        :rtype: model_pb2.Namespace
        :raises: errors.ConflictError
        """
        assert isinstance(namespace_pb, model_pb2.Namespace)
        assert namespace_id == namespace_pb.meta.id
        zk_path = objects.NamespaceDescriptor.uid_to_zk_path(namespace_id)
        try:
            return self.namespace_storage.create(zk_path, namespace_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Namespace "{}" already exists'.format(namespace_id))

    def update_namespace(self, namespace_id, namespace_pb=None):
        """
        :type namespace_id: six.text_type
        :type namespace_pb: model_pb2.Namespace | None
        :rtype: list[model_pb2.Namespace]
        :raises: errors.NotFoundError
        """
        assert namespace_pb is None or isinstance(namespace_pb, model_pb2.Namespace)
        assert namespace_pb is None or namespace_id == namespace_pb.meta.id
        zk_path = objects.NamespaceDescriptor.uid_to_zk_path(namespace_id)
        for namespace_pb in self.namespace_storage.guaranteed_update(zk_path, obj=namespace_pb):
            if namespace_pb is None:
                raise errors.NotFoundError('Namespace "{}" does not exist'.format(namespace_id))
            yield namespace_pb

    def remove_namespace(self, namespace_id):
        """
        :type namespace_id: six.text_type
        """
        zk_path = objects.NamespaceDescriptor.uid_to_zk_path(namespace_id)
        self.namespace_storage.remove(zk_path)

    def create_namespace_aspects_set(self, namespace_id, namespace_aspects_set_pb):
        """
        :type namespace_id: six.text_type
        :type namespace_aspects_set_pb: model_pb2.NamespaceAspectsSet
        :rtype: model_pb2.NamespaceAspectsSet
        :raises: errors.ConflictError
        """
        assert isinstance(namespace_aspects_set_pb, model_pb2.NamespaceAspectsSet)
        assert namespace_aspects_set_pb.meta.namespace_id == namespace_id
        zk_path = objects.NamespaceAspectSetDescriptor.uid_to_zk_path(namespace_id)
        try:
            return self.namespace_aspects_set_storage.create(zk_path, namespace_aspects_set_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Namespace aspects set "{}" already exists'.format(namespace_id))

    def update_namespace_aspects_set(self, namespace_id, namespace_aspects_set_pb=None):
        """
        :type namespace_id: six.text_type
        :type namespace_aspects_set_pb: model_pb2.NamespaceAspectsSet | None
        :rtype: list[model_pb2.NamespaceAspectsSet]
        :raises: errors.NotFoundError
        """
        assert namespace_aspects_set_pb is None or isinstance(namespace_aspects_set_pb, model_pb2.NamespaceAspectsSet)
        assert namespace_aspects_set_pb is None or namespace_id == namespace_aspects_set_pb.meta.namespace_id
        zk_path = objects.NamespaceAspectSetDescriptor.uid_to_zk_path(namespace_id)
        for namespace_aspects_set_pb in self.namespace_aspects_set_storage.guaranteed_update(
                zk_path, obj=namespace_aspects_set_pb):
            if namespace_aspects_set_pb is None:
                raise errors.NotFoundError('Namespace aspects set "{}" does not exist'.format(namespace_id))
            yield namespace_aspects_set_pb

    def remove_namespace_aspects_set(self, namespace_id):
        """
        :type namespace_id: six.text_type
        """
        zk_path = objects.NamespaceAspectSetDescriptor.uid_to_zk_path(namespace_id)
        self.namespace_aspects_set_storage.remove(zk_path)

    def get_knob(self, namespace_id, knob_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type knob_id:
        :param bool sync:
        :rtype: model_pb2.Knob | None
        """
        zk_path = objects.KnobDescriptor.uid_to_zk_path(namespace_id, knob_id)
        if sync:
            self.knob_storage.sync(zk_path)
        return self.knob_storage.get(zk_path)

    def must_get_knob(self, namespace_id, knob_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type knob_id:
        :param bool sync:
        :rtype: model_pb2.Knob
        :raises: errors.NotFoundError
        """
        knob_pb = self.get_knob(namespace_id, knob_id, sync=sync)
        if knob_pb is None:
            raise errors.NotFoundError('Knob "{}:{}" does not exist'.format(namespace_id, knob_id))
        return knob_pb

    def create_knob(self, namespace_id, knob_id, knob_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type knob_id:
        :type knob_pb: model_pb2.Knob
        :rtype: model_pb2.Knob
        :raises: errors.ConflictError
        """
        assert isinstance(knob_pb, model_pb2.Knob)
        assert namespace_id == knob_pb.meta.namespace_id
        assert knob_id == knob_pb.meta.id
        zk_path = objects.KnobDescriptor.uid_to_zk_path(namespace_id, knob_id)
        try:
            return self.knob_storage.create(zk_path, knob_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Knob "{}:{}" already exists'.format(namespace_id, knob_id))

    def update_knob(self, namespace_id, knob_id, knob_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type knob_id:
        :type knob_pb: model_pb2.Knob | None
        :rtype: list[model_pb2.Knob]
        :raises: errors.NotFoundError
        """
        assert knob_pb is None or isinstance(knob_pb, model_pb2.Knob)
        assert knob_pb is None or namespace_id == knob_pb.meta.namespace_id
        assert knob_pb is None or knob_id == knob_pb.meta.id
        zk_path = objects.KnobDescriptor.uid_to_zk_path(namespace_id, knob_id)
        for knob_pb in self.knob_storage.guaranteed_update(zk_path, obj=knob_pb):
            if knob_pb is None:
                raise errors.NotFoundError('Knob "{}:{}" does not exist'.format(namespace_id, knob_id))
            yield knob_pb

    def remove_knob(self, namespace_id, knob_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type knob_id:
        """
        zk_path = objects.KnobDescriptor.uid_to_zk_path(namespace_id, knob_id)
        self.knob_storage.remove(zk_path)

    def remove_namespace_knobs(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.knob_storage.remove(namespace_id, recursive=True)

    def does_knob_exist(self, namespace_id, knob_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type knob_id:
        :rtype: bool
        """
        zk_path = objects.KnobDescriptor.uid_to_zk_path(namespace_id, knob_id)
        return self.knob_storage.exists(zk_path)

    def get_l3_balancer(self, namespace_id, l3_balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type l3_balancer_id:
        :param bool sync:
        :rtype: model_pb2.L3Balancer | None
        """
        zk_path = objects.L3BalancerDescriptor.uid_to_zk_path(namespace_id, l3_balancer_id)
        if sync:
            self.l3_balancer_storage.sync(zk_path)
        return self.l3_balancer_storage.get(zk_path)

    def must_get_l3_balancer(self, namespace_id, l3_balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type l3_balancer_id:
        :param bool sync:
        :rtype: model_pb2.L3Balancer
        :raises: errors.NotFoundError
        """
        l3_balancer_pb = self.get_l3_balancer(namespace_id, l3_balancer_id, sync=sync)
        if l3_balancer_pb is None:
            raise errors.NotFoundError('L3 balancer "{}:{}" does not exist'.format(namespace_id, l3_balancer_id))
        return l3_balancer_pb

    def create_l3_balancer(self, namespace_id, l3_balancer_id, l3_balancer_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type l3_balancer_id:
        :type l3_balancer_pb: model_pb2.L3Balancer
        :rtype: model_pb2.L3Balancer
        :raises: errors.ConflictError
        """
        assert isinstance(l3_balancer_pb, model_pb2.L3Balancer)
        assert namespace_id == l3_balancer_pb.meta.namespace_id
        assert l3_balancer_id == l3_balancer_pb.meta.id
        zk_path = objects.L3BalancerDescriptor.uid_to_zk_path(namespace_id, l3_balancer_id)
        try:
            return self.l3_balancer_storage.create(zk_path, l3_balancer_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('L3 balancer "{}:{}" already exists'.format(namespace_id, l3_balancer_id))

    def update_l3_balancer(self, namespace_id, l3_balancer_id, l3_balancer_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type l3_balancer_id:
        :type l3_balancer_pb: model_pb2.L3Balancer | None
        :rtype: list[model_pb2.L3Balancer]
        :raises: errors.NotFoundError
        """
        assert l3_balancer_pb is None or isinstance(l3_balancer_pb, model_pb2.L3Balancer)
        assert l3_balancer_pb is None or namespace_id == l3_balancer_pb.meta.namespace_id
        assert l3_balancer_pb is None or l3_balancer_id == l3_balancer_pb.meta.id
        zk_path = objects.L3BalancerDescriptor.uid_to_zk_path(namespace_id, l3_balancer_id)
        for l3_balancer_pb in self.l3_balancer_storage.guaranteed_update(zk_path, obj=l3_balancer_pb):
            if l3_balancer_pb is None:
                raise errors.NotFoundError('L3 balancer "{}:{}" does not exist'.format(namespace_id, l3_balancer_id))
            yield l3_balancer_pb

    def remove_l3_balancer(self, namespace_id, l3_balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type l3_balancer_id:
        """
        zk_path = objects.L3BalancerDescriptor.uid_to_zk_path(namespace_id, l3_balancer_id)
        self.l3_balancer_storage.remove(zk_path)

    def remove_namespace_l3_balancers(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.l3_balancer_storage.remove(namespace_id, recursive=True)

    def does_l3_balancer_exist(self, namespace_id, l3_balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type l3_balancer_id:
        :rtype: bool
        """
        zk_path = objects.L3BalancerDescriptor.uid_to_zk_path(namespace_id, l3_balancer_id)
        return self.l3_balancer_storage.exists(zk_path)

    def create_name_server(self, namespace_id, name_server_id, name_server_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type name_server_id:
        :type name_server_pb: model_pb2.NameServer
        :rtype: model_pb2.NameServer
        :raises: errors.ConflictError
        """
        assert isinstance(name_server_pb, model_pb2.NameServer)
        assert namespace_id == name_server_pb.meta.namespace_id
        assert name_server_id == name_server_pb.meta.id
        zk_path = objects.NameServerDescriptor.uid_to_zk_path(namespace_id, name_server_id)
        try:
            return self.name_server_storage.create(zk_path, name_server_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Name Server "{}:{}" already exists'.format(namespace_id, name_server_id))

    def does_name_server_exist(self, namespace_id, name_server_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type name_server_id:
        :rtype: bool
        """
        zk_path = objects.NameServerDescriptor.uid_to_zk_path(namespace_id, name_server_id)
        return self.name_server_storage.exists(zk_path)

    def get_name_server(self, namespace_id, name_server_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type name_server_id:
        :param bool sync:
        :rtype: model_pb2.NameServer | None
        """
        zk_path = objects.NameServerDescriptor.uid_to_zk_path(namespace_id, name_server_id)
        if sync:
            self.name_server_storage.sync(zk_path)
        return self.name_server_storage.get(zk_path)

    def must_get_name_server(self, namespace_id, name_server_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type name_server_id:
        :param bool sync:
        :rtype: model_pb2.NameServer
        :raises: errors.NotFoundError
        """
        name_server_pb = self.get_name_server(namespace_id, name_server_id, sync=sync)
        if name_server_pb is None:
            raise errors.NotFoundError('Name Server "{}:{}" does not exist'.format(namespace_id, name_server_id))
        return name_server_pb

    def does_dns_record_exist(self, namespace_id, dns_record_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        :rtype: bool
        """
        zk_path = objects.DnsRecordDescriptor.uid_to_zk_path(namespace_id, dns_record_id)
        return self.dns_record_storage.exists(zk_path)

    def create_dns_record(self, namespace_id, dns_record_id, dns_record_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        :type dns_record_pb: model_pb2.DnsRecord
        :rtype: model_pb2.DnsRecord
        :raises: errors.ConflictError
        """
        assert isinstance(dns_record_pb, model_pb2.DnsRecord)
        assert namespace_id == dns_record_pb.meta.namespace_id
        assert dns_record_id == dns_record_pb.meta.id
        zk_path = objects.DnsRecordDescriptor.uid_to_zk_path(namespace_id, dns_record_id)
        try:
            return self.dns_record_storage.create(zk_path, dns_record_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('DNS record "{}:{}" already exists'.format(namespace_id, dns_record_id))

    def get_dns_record(self, namespace_id, dns_record_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        :param bool sync:
        :rtype: model_pb2.DnsRecord | None
        """
        zk_path = objects.DnsRecordDescriptor.uid_to_zk_path(namespace_id, dns_record_id)
        if sync:
            self.dns_record_storage.sync(zk_path)
        return self.dns_record_storage.get(zk_path)

    def must_get_dns_record(self, namespace_id, dns_record_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        :param bool sync:
        :rtype: model_pb2.DnsRecord
        :raises: errors.NotFoundError
        """
        dns_record_pb = self.get_dns_record(namespace_id, dns_record_id, sync=sync)
        if dns_record_pb is None:
            raise errors.NotFoundError('DNS record "{}:{}" does not exist'.format(namespace_id, dns_record_id))
        return dns_record_pb

    def update_dns_record(self, namespace_id, dns_record_id, dns_record_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        :type dns_record_pb: model_pb2.DnsRecord | None
        :rtype: list[model_pb2.DnsRecord]
        :raises: errors.NotFoundError
        """
        assert dns_record_pb is None or isinstance(dns_record_pb, model_pb2.DnsRecord)
        assert dns_record_pb is None or namespace_id == dns_record_pb.meta.namespace_id
        assert dns_record_pb is None or dns_record_id == dns_record_pb.meta.id
        zk_path = objects.DnsRecordDescriptor.uid_to_zk_path(namespace_id, dns_record_id)
        for dns_record_pb in self.dns_record_storage.guaranteed_update(zk_path, obj=dns_record_pb):
            if dns_record_pb is None:
                raise errors.NotFoundError('DNS record "{}:{}" does not exist'.format(namespace_id, dns_record_id))
            yield dns_record_pb

    def remove_dns_record(self, namespace_id, dns_record_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        """
        zk_path = objects.DnsRecordDescriptor.uid_to_zk_path(namespace_id, dns_record_id)
        self.dns_record_storage.remove(zk_path)

    def does_dns_record_operation_exist(self, namespace_id, dns_record_op_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_op_id:
        :rtype: bool
        """
        zk_path = objects.DnsRecordOperationDescriptor.uid_to_zk_path(namespace_id, dns_record_op_id)
        return self.dns_record_operation_storage.exists(zk_path)

    def create_dns_record_operation(self, namespace_id, dns_record_op_id, dns_record_op_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_op_id:
        :type dns_record_op_pb: model_pb2.DnsRecordOperation
        :rtype: model_pb2.DnsRecordOperation
        :raises: errors.ConflictError
        """
        assert isinstance(dns_record_op_pb, model_pb2.DnsRecordOperation)
        assert namespace_id == dns_record_op_pb.meta.namespace_id
        assert dns_record_op_id == dns_record_op_pb.meta.id
        zk_path = objects.DnsRecordDescriptor.uid_to_zk_path(namespace_id, dns_record_op_id)
        try:
            return self.dns_record_operation_storage.create(zk_path, dns_record_op_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('DNS record operation "{}:{}" already exists'.format(
                namespace_id, dns_record_op_id))

    def get_dns_record_operation(self, namespace_id, dns_record_op_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_op_id:
        :param bool sync:
        :rtype: model_pb2.DnsRecordOperation | None
        """
        zk_path = objects.DnsRecordOperationDescriptor.uid_to_zk_path(namespace_id, dns_record_op_id)
        if sync:
            self.dns_record_operation_storage.sync(zk_path)
        return self.dns_record_operation_storage.get(zk_path)

    def must_get_dns_record_operation(self, namespace_id, dns_record_op_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_op_id:
        :param bool sync:
        :rtype: model_pb2.DnsRecordOperation
        :raises: errors.NotFoundError
        """
        dns_record_op_pb = self.get_dns_record_operation(namespace_id, dns_record_op_id, sync=sync)
        if dns_record_op_pb is None:
            raise errors.NotFoundError('DNS record operation "{}:{}" does not exist'.format(
                namespace_id, dns_record_op_id))
        return dns_record_op_pb

    def update_dns_record_operation(self, namespace_id, dns_record_op_id, dns_record_op_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_op_id:
        :type dns_record_op_pb: model_pb2.DnsRecordOperation | None
        :rtype: list[model_pb2.DnsRecordOperation]
        :raises: errors.NotFoundError
        """
        assert dns_record_op_pb is None or isinstance(dns_record_op_pb, model_pb2.DnsRecordOperation)
        assert dns_record_op_pb is None or namespace_id == dns_record_op_pb.meta.namespace_id
        assert dns_record_op_pb is None or dns_record_op_id == dns_record_op_pb.meta.id
        zk_path = objects.DnsRecordOperationDescriptor.uid_to_zk_path(namespace_id, dns_record_op_id)
        for dns_record_op_pb in self.dns_record_operation_storage.guaranteed_update(zk_path, obj=dns_record_op_pb):
            if dns_record_op_pb is None:
                raise errors.NotFoundError('DNS record operation "{}:{}" does not exist'.format(
                    namespace_id, dns_record_op_id))
            yield dns_record_op_pb

    def remove_dns_record_operation(self, namespace_id, dns_record_op_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_op_id:
        """
        zk_path = objects.DnsRecordOperationDescriptor.uid_to_zk_path(namespace_id, dns_record_op_id)
        self.dns_record_operation_storage.remove(zk_path)

    def remove_namespace_dns_record_operations(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.dns_record_operation_storage.remove(namespace_id)

    def does_cert_exist(self, namespace_id, cert_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_id:
        :rtype: bool
        """
        zk_path = objects.CertDescriptor.uid_to_zk_path(namespace_id, cert_id)
        return self.cert_storage.exists(zk_path)

    def create_cert(self, namespace_id, cert_id, cert_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_id:
        :type cert_pb: model_pb2.Certificate
        :rtype: model_pb2.Certificate
        :raises: errors.ConflictError
        """
        assert isinstance(cert_pb, model_pb2.Certificate)
        assert namespace_id == cert_pb.meta.namespace_id
        assert cert_id == cert_pb.meta.id
        zk_path = objects.CertDescriptor.uid_to_zk_path(namespace_id, cert_id)
        try:
            return self.cert_storage.create(zk_path, cert_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Certificate "{}:{}" already exists'.format(namespace_id, cert_id))

    def get_cert(self, namespace_id, cert_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_id:
        :param bool sync:
        :rtype: model_pb2.Certificate | None
        """
        zk_path = objects.CertDescriptor.uid_to_zk_path(namespace_id, cert_id)
        if sync:
            self.cert_storage.sync(zk_path)
        return self.cert_storage.get(zk_path)

    def must_get_cert(self, namespace_id, cert_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_id:
        :param bool sync:
        :rtype: model_pb2.Certificate
        :raises: errors.NotFoundError
        """
        cert_pb = self.get_cert(namespace_id, cert_id, sync=sync)
        if cert_pb is None:
            raise errors.NotFoundError('Certificate "{}:{}" does not exist'.format(namespace_id, cert_id))
        return cert_pb

    def update_cert(self, namespace_id, cert_id, cert_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_id:
        :type cert_pb: model_pb2.Certificate | None
        :rtype: list[model_pb2.Certificate]
        :raises: errors.NotFoundError
        """
        assert cert_pb is None or isinstance(cert_pb, model_pb2.Certificate)
        assert cert_pb is None or namespace_id == cert_pb.meta.namespace_id
        assert cert_pb is None or cert_id == cert_pb.meta.id
        zk_path = objects.CertDescriptor.uid_to_zk_path(namespace_id, cert_id)
        for c_pb in self.cert_storage.guaranteed_update(zk_path, obj=cert_pb):
            if c_pb is None:
                raise errors.NotFoundError('Certificate "{}:{}" does not exist'.format(namespace_id, cert_id))
            yield c_pb

    def remove_cert(self, namespace_id, cert_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_id:
        """
        zk_path = objects.CertDescriptor.uid_to_zk_path(namespace_id, cert_id)
        self.cert_storage.remove(zk_path)

    def does_cert_renewal_exist(self, namespace_id, cert_renewal_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_renewal_id:
        :rtype: bool
        """
        zk_path = objects.CertRenewalDescriptor.uid_to_zk_path(namespace_id, cert_renewal_id)
        return self.cert_renewal_storage.exists(zk_path)

    def create_cert_renewal(self, namespace_id, cert_renewal_id, cert_renewal_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_renewal_id:
        :type cert_renewal_pb: model_pb2.CertificateRenewal
        :rtype: model_pb2.CertificateRenewal
        :raises: errors.ConflictError
        """
        assert isinstance(cert_renewal_pb, model_pb2.CertificateRenewal)
        assert namespace_id == cert_renewal_pb.meta.namespace_id
        assert cert_renewal_id == cert_renewal_pb.meta.id
        zk_path = objects.CertRenewalDescriptor.uid_to_zk_path(namespace_id, cert_renewal_id)
        try:
            return self.cert_renewal_storage.create(zk_path, cert_renewal_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('CertificateRenewal "{}:{}" already exists'.format(
                namespace_id, cert_renewal_id))

    def get_cert_renewal(self, namespace_id, cert_renewal_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_renewal_id:
        :param bool sync:
        :rtype: model_pb2.CertificateRenewal | None
        """
        zk_path = objects.CertRenewalDescriptor.uid_to_zk_path(namespace_id, cert_renewal_id)
        if sync:
            self.cert_renewal_storage.sync(zk_path)
        return self.cert_renewal_storage.get(zk_path)

    def must_get_cert_renewal(self, namespace_id, cert_renewal_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_renewal_id:
        :param bool sync:
        :rtype: model_pb2.CertificateRenewal
        :raises: errors.NotFoundError
        """
        cert_renewal_pb = self.get_cert_renewal(namespace_id, cert_renewal_id, sync=sync)
        if cert_renewal_pb is None:
            raise errors.NotFoundError('Certificate renewal "{}:{}" does not exist'.format(namespace_id,
                                                                                           cert_renewal_id))
        return cert_renewal_pb

    def update_cert_renewal(self, namespace_id, cert_renewal_id, cert_renewal_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_renewal_id:
        :type cert_renewal_pb: model_pb2.CertificateRenewal | None
        :rtype: list[model_pb2.CertificateRenewal]
        :raises: errors.NotFoundError
        """
        assert cert_renewal_pb is None or isinstance(cert_renewal_pb, model_pb2.CertificateRenewal)
        assert cert_renewal_pb is None or namespace_id == cert_renewal_pb.meta.namespace_id
        assert cert_renewal_pb is None or cert_renewal_id == cert_renewal_pb.meta.id
        zk_path = objects.CertRenewalDescriptor.uid_to_zk_path(namespace_id, cert_renewal_id)
        for c_pb in self.cert_renewal_storage.guaranteed_update(zk_path, obj=cert_renewal_pb):
            if c_pb is None:
                raise errors.NotFoundError('CertificateRenewal "{}:{}" does not exist'.format(
                    namespace_id, cert_renewal_id))
            yield c_pb

    def remove_cert_renewal(self, namespace_id, cert_renewal_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type cert_renewal_id:
        """
        zk_path = objects.CertRenewalDescriptor.uid_to_zk_path(namespace_id, cert_renewal_id)
        self.cert_renewal_storage.remove(zk_path)

    def remove_namespace_dns_records(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        zk_path = objects.DnsRecordDescriptor.uid_to_zk_path(namespace_id)
        self.dns_record_storage.remove(zk_path, recursive=True)

    def get_l3_balancer_state(self, namespace_id, l3_balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type l3_balancer_id:
        :param bool sync:
        :rtype: model_pb2.L3BalancerState | None
        """
        zk_path = objects.L3BalancerStateDescriptor.uid_to_zk_path(namespace_id, l3_balancer_id)
        if sync:
            self.l3_balancer_state_storage.sync(zk_path)
        return self.l3_balancer_state_storage.get(zk_path)

    def must_get_l3_balancer_state(self, namespace_id, l3_balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type l3_balancer_id:
        :param bool sync:
        :rtype: model_pb2.L3BalancerState
        :raises: errors.NotFoundError
        """
        l3_balancer_state_pb = self.get_l3_balancer_state(namespace_id, l3_balancer_id, sync=sync)
        if l3_balancer_state_pb is None:
            raise errors.NotFoundError('L3 balancer state "{}:{}" does not exist'.format(namespace_id, l3_balancer_id))
        return l3_balancer_state_pb

    def sync_l3_balancer_states(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.l3_balancer_state_storage.sync(namespace_id)

    def create_l3_balancer_state(self, namespace_id, l3_balancer_id, l3_balancer_state_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type l3_balancer_id:
        :type l3_balancer_state_pb: model_pb2.L3BalancerState
        :rtype: model_pb2.L3BalancerState
        :raises: errors.ConflictError
        """
        assert isinstance(l3_balancer_state_pb, model_pb2.L3BalancerState)
        assert l3_balancer_state_pb.l3_balancer_id == l3_balancer_id
        assert l3_balancer_state_pb.namespace_id == namespace_id
        zk_path = objects.L3BalancerStateDescriptor.uid_to_zk_path(namespace_id, l3_balancer_id)
        try:
            return self.l3_balancer_state_storage.create(zk_path, l3_balancer_state_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('L3 balancer state "{}:{}" already exists'.format(namespace_id, l3_balancer_id))

    def update_l3_balancer_state(self, namespace_id, l3_balancer_id, l3_balancer_state_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type l3_balancer_id:
        :type l3_balancer_state_pb: model_pb2.L3BalancerState | None
        :rtype: list[model_pb2.L3BalancerState]
        :raises: errors.NotFoundError
        """
        assert l3_balancer_state_pb is None or isinstance(l3_balancer_state_pb, model_pb2.L3BalancerState)
        assert l3_balancer_state_pb is None or l3_balancer_state_pb.l3_balancer_id == l3_balancer_id
        assert l3_balancer_state_pb is None or l3_balancer_state_pb.namespace_id == namespace_id
        zk_path = objects.L3BalancerStateDescriptor.uid_to_zk_path(namespace_id, l3_balancer_id)
        for l3_balancer_state_pb in self.l3_balancer_state_storage.guaranteed_update(zk_path,
                                                                                     obj=l3_balancer_state_pb):
            if l3_balancer_state_pb is None:
                raise errors.NotFoundError(
                    'L3 balancer state "{}:{}" does not exist'.format(namespace_id, l3_balancer_id))
            yield l3_balancer_state_pb

    def remove_l3_balancer_state(self, namespace_id, l3_balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type l3_balancer_id:
        """
        zk_path = objects.L3BalancerStateDescriptor.uid_to_zk_path(namespace_id, l3_balancer_id)
        self.l3_balancer_state_storage.remove(zk_path)

    def remove_namespace_l3_balancer_states(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.l3_balancer_state_storage.remove(namespace_id, recursive=True)

    def get_dns_record_state(self, namespace_id, dns_record_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        :param bool sync:
        :rtype: model_pb2.DnsRecordState | None
        """
        zk_path = objects.DnsRecordStateDescriptor.uid_to_zk_path(namespace_id, dns_record_id)
        if sync:
            self.dns_record_state_storage.sync(zk_path)
        return self.dns_record_state_storage.get(zk_path)

    def must_get_dns_record_state(self, namespace_id, dns_record_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        :param bool sync:
        :rtype: model_pb2.DnsRecordState
        :raises: errors.NotFoundError
        """
        dns_record_state_pb = self.get_dns_record_state(namespace_id, dns_record_id, sync=sync)
        if dns_record_state_pb is None:
            raise errors.NotFoundError('DNS Record state "{}:{}" does not exist'.format(namespace_id, dns_record_id))
        return dns_record_state_pb

    def sync_dns_record_states(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.dns_record_state_storage.sync(namespace_id)

    def create_dns_record_state(self, namespace_id, dns_record_id, dns_record_state_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        :type dns_record_state_pb: model_pb2.DnsRecordState
        :rtype: model_pb2.DnsRecordState
        :raises: errors.ConflictError
        """
        assert isinstance(dns_record_state_pb, model_pb2.DnsRecordState)
        assert dns_record_state_pb.dns_record_id == dns_record_id
        assert dns_record_state_pb.namespace_id == namespace_id
        zk_path = objects.DnsRecordStateDescriptor.uid_to_zk_path(namespace_id, dns_record_id)
        try:
            return self.dns_record_state_storage.create(zk_path, dns_record_state_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('DNS Record state "{}:{}" already exists'.format(namespace_id, dns_record_id))

    def update_dns_record_state(self, namespace_id, dns_record_id, dns_record_state_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        :type dns_record_state_pb: model_pb2.DnsRecordState | None
        :rtype: list[model_pb2.DnsRecordState]
        :raises: errors.NotFoundError
        """
        assert dns_record_state_pb is None or isinstance(dns_record_state_pb, model_pb2.DnsRecordState)
        assert dns_record_state_pb is None or dns_record_state_pb.dns_record_id == dns_record_id
        assert dns_record_state_pb is None or dns_record_state_pb.namespace_id == namespace_id
        zk_path = objects.DnsRecordStateDescriptor.uid_to_zk_path(namespace_id, dns_record_id)
        for dns_record_state_pb in self.dns_record_state_storage.guaranteed_update(zk_path,
                                                                                   obj=dns_record_state_pb):
            if dns_record_state_pb is None:
                raise errors.NotFoundError(
                    'DNS Record state "{}:{}" does not exist'.format(namespace_id, dns_record_id))
            yield dns_record_state_pb

    def remove_dns_record_state(self, namespace_id, dns_record_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type dns_record_id:
        """
        zk_path = objects.DnsRecordStateDescriptor.uid_to_zk_path(namespace_id, dns_record_id)
        self.dns_record_state_storage.remove(zk_path)

    def remove_namespace_dns_record_states(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        zk_path = objects.DnsRecordStateDescriptor.uid_to_zk_path(namespace_id)
        self.dns_record_state_storage.remove(zk_path, recursive=True)

    def get_balancer(self, namespace_id, balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :param bool sync:
        :rtype: model_pb2.Balancer | None
        """
        zk_path = objects.L7BalancerDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        if sync:
            self.balancer_storage.sync(zk_path)
        return self.balancer_storage.get(zk_path)

    def must_get_balancer(self, namespace_id, balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :param bool sync:
        :rtype: model_pb2.Balancer
        :raises: errors.NotFoundError
        """
        balancer_pb = self.get_balancer(namespace_id, balancer_id, sync=sync)
        if balancer_pb is None:
            raise errors.NotFoundError('Balancer "{}:{}" does not exist'.format(namespace_id, balancer_id))
        return balancer_pb

    def create_balancer(self, namespace_id, balancer_id, balancer_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :type balancer_pb: model_pb2.Balancer
        :rtype: model_pb2.Balancer
        :raises: errors.ConflictError
        """
        assert isinstance(balancer_pb, model_pb2.Balancer)
        assert balancer_pb.meta.id == balancer_id
        assert balancer_pb.meta.namespace_id == namespace_id
        zk_path = objects.L7BalancerDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        try:
            return self.balancer_storage.create(zk_path, balancer_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Balancer "{}:{}" already exists'.format(namespace_id, balancer_id))

    def update_balancer(self, namespace_id, balancer_id, balancer_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :type balancer_pb: model_pb2.Balancer | None
        :rtype: list[model_pb2.Balancer]
        :raises: errors.NotFoundError
        """
        assert balancer_pb is None or isinstance(balancer_pb, model_pb2.Balancer)
        assert balancer_pb is None or balancer_id == balancer_pb.meta.id
        assert balancer_pb is None or namespace_id == balancer_pb.meta.namespace_id
        zk_path = objects.L7BalancerDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        for balancer_pb in self.balancer_storage.guaranteed_update(zk_path, obj=balancer_pb):
            if balancer_pb is None:
                raise errors.NotFoundError('Balancer "{}:{}" does not exist'.format(namespace_id, balancer_id))
            yield balancer_pb

    def remove_balancer(self, namespace_id, balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        """
        zk_path = objects.L7BalancerDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        self.balancer_storage.remove(zk_path)

    def create_balancer_aspects_set(self, namespace_id, balancer_id, balancer_aspects_set_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :type balancer_aspects_set_pb: model_pb2.BalancerAspectsSet
        :rtype: model_pb2.BalancerAspectsSet
        :raises: errors.ConflictError
        """
        assert isinstance(balancer_aspects_set_pb, model_pb2.BalancerAspectsSet)
        assert balancer_aspects_set_pb.meta.balancer_id == balancer_id
        assert balancer_aspects_set_pb.meta.namespace_id == namespace_id
        zk_path = objects.L7BalancerAspectSetDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        try:
            return self.balancer_aspects_set_storage.create(zk_path, balancer_aspects_set_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Balancer aspects set "{}:{}" already exists'.format(namespace_id, balancer_id))

    def update_balancer_aspects_set(self, namespace_id, balancer_id, balancer_aspects_set_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :type balancer_aspects_set_pb: model_pb2.BalancerAspectsSet | None
        :rtype: list[model_pb2.BalancerAspectsSet]
        :raises: errors.NotFoundError
        """
        assert balancer_aspects_set_pb is None or isinstance(balancer_aspects_set_pb, model_pb2.BalancerAspectsSet)
        assert balancer_aspects_set_pb is None or balancer_id == balancer_aspects_set_pb.meta.balancer_id
        assert balancer_aspects_set_pb is None or namespace_id == balancer_aspects_set_pb.meta.namespace_id
        zk_path = objects.L7BalancerAspectSetDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        for balancer_aspects_set_pb in self.balancer_aspects_set_storage.guaranteed_update(
                zk_path, obj=balancer_aspects_set_pb):
            if balancer_aspects_set_pb is None:
                raise errors.NotFoundError(
                    'Balancer aspects set "{}:{}" does not exist'.format(namespace_id, balancer_id))
            yield balancer_aspects_set_pb

    def remove_balancer_aspects_set(self, namespace_id, balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        """
        zk_path = objects.L7BalancerAspectSetDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        self.balancer_aspects_set_storage.remove(zk_path)

    def remove_namespace_balancers(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.balancer_storage.remove(namespace_id, recursive=True)

    def remove_namespace_balancer_operations(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.balancer_operation_storage.remove(namespace_id, recursive=True)

    def remove_namespace_balancer_aspect_sets(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.balancer_aspects_set_storage.remove(namespace_id, recursive=True)

    def get_balancer_aspects_set(self, namespace_id, balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :param bool sync:
        :rtype: model_pb2.BalancerAspectsSet | None
        """
        zk_path = objects.L7BalancerAspectSetDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        if sync:
            self.balancer_aspects_set_storage.sync(zk_path)
        return self.balancer_aspects_set_storage.get(zk_path)

    def must_get_balancer_aspects_set(self, namespace_id, balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :param bool sync:
        :rtype: model_pb2.BalancerAspectsSet
        :raises: errors.NotFoundError
        """
        balancer_aspects_set_pb = self.get_balancer_aspects_set(namespace_id, balancer_id, sync=sync)
        if balancer_aspects_set_pb is None:
            raise errors.NotFoundError('Balancer aspects set "{}:{}" does not exist'.format(namespace_id, balancer_id))
        return balancer_aspects_set_pb

    def does_balancer_exist(self, namespace_id, balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :param bool sync:
        :rtype: bool
        """
        zk_path = objects.L7BalancerDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        if sync:
            self.balancer_storage.sync(zk_path)
        return self.balancer_storage.exists(zk_path)

    def get_balancer_state(self, namespace_id, balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :param bool sync:
        :rtype: model_pb2.BalancerState | None
        """
        zk_path = objects.L7BalancerStateDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        if sync:
            self.balancer_state_storage.sync(zk_path)
        return self.balancer_state_storage.get(zk_path)

    def must_get_balancer_state(self, namespace_id, balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :param bool sync:
        :rtype: model_pb2.BalancerState
        :raises: errors.NotFoundError
        """
        balancer_state_pb = self.get_balancer_state(namespace_id, balancer_id, sync=sync)
        if balancer_state_pb is None:
            raise errors.NotFoundError('Balancer state "{}:{}" does not exist'.format(namespace_id, balancer_id))
        return balancer_state_pb

    def sync_balancer_states(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.balancer_state_storage.sync(namespace_id)

    def create_balancer_state(self, namespace_id, balancer_id, balancer_state_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :type balancer_state_pb: model_pb2.BalancerState
        :rtype: model_pb2.BalancerState
        :raises: errors.ConflictError
        """
        assert isinstance(balancer_state_pb, model_pb2.BalancerState)
        assert balancer_state_pb.namespace_id == namespace_id
        assert balancer_state_pb.balancer_id == balancer_id
        zk_path = objects.L7BalancerStateDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        try:
            return self.balancer_state_storage.create(zk_path, balancer_state_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Balancer state "{}:{}" already exists'.format(namespace_id, balancer_id))

    def update_balancer_state(self, namespace_id, balancer_id, balancer_state_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :type balancer_state_pb: model_pb2.BalancerState | None
        :rtype: list[model_pb2.BalancerState]
        :raises: errors.NotFoundError
        """
        assert balancer_state_pb is None or isinstance(balancer_state_pb, model_pb2.BalancerState)
        assert balancer_state_pb is None or balancer_state_pb.namespace_id == namespace_id
        assert balancer_state_pb is None or balancer_state_pb.balancer_id == balancer_id
        zk_path = objects.L7BalancerStateDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        for balancer_state_pb in self.balancer_state_storage.guaranteed_update(zk_path, obj=balancer_state_pb):
            if balancer_state_pb is None:
                raise errors.NotFoundError('Balancer state "{}:{}" does not exist'.format(namespace_id, balancer_id))
            yield balancer_state_pb

    def remove_balancer_state(self, namespace_id, balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        """
        zk_path = objects.L7BalancerStateDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        self.balancer_state_storage.remove(zk_path)

    def remove_namespace_balancer_states(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.balancer_state_storage.remove(namespace_id, recursive=True)

    def get_upstream(self, namespace_id, upstream_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type upstream_id:
        :param bool sync:
        :rtype: model_pb2.Upstream | None
        """
        zk_path = objects.UpstreamDescriptor.uid_to_zk_path(namespace_id, upstream_id)
        if sync:
            self.upstream_storage.sync(zk_path)
        return self.upstream_storage.get(zk_path)

    def must_get_upstream(self, namespace_id, upstream_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type upstream_id:
        :param bool sync:
        :rtype: model_pb2.Upstream
        :raises: errors.NotFoundError
        """
        upstream_pb = self.get_upstream(namespace_id, upstream_id, sync=sync)
        if upstream_pb is None:
            raise errors.NotFoundError('Upstream "{}:{}" does not exist'.format(namespace_id, upstream_id))
        return upstream_pb

    def create_upstream(self, namespace_id, upstream_id, upstream_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type upstream_id:
        :type upstream_pb: model_pb2.Upstream
        :rtype: model_pb2.Upstream
        :raises: errors.ConflictError
        """
        assert isinstance(upstream_pb, model_pb2.Upstream)
        assert upstream_pb.meta.id == upstream_id
        assert upstream_pb.meta.namespace_id == namespace_id
        zk_path = objects.UpstreamDescriptor.uid_to_zk_path(namespace_id, upstream_id)
        try:
            return self.upstream_storage.create(zk_path, upstream_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Upstream "{}:{}" already exists'.format(namespace_id, upstream_id))

    def update_upstream(self, namespace_id, upstream_id, upstream_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type upstream_id:
        :type upstream_pb: model_pb2.Upstream | None
        :rtype: list[model_pb2.Upstream]
        :raises: errors.NotFoundError
        """
        assert upstream_pb is None or isinstance(upstream_pb, model_pb2.Upstream)
        assert upstream_pb is None or upstream_pb.meta.id == upstream_id
        assert upstream_pb is None or upstream_pb.meta.namespace_id == namespace_id
        zk_path = objects.UpstreamDescriptor.uid_to_zk_path(namespace_id, upstream_id)
        for upstream_pb in self.upstream_storage.guaranteed_update(zk_path, obj=upstream_pb):
            if upstream_pb is None:
                raise errors.NotFoundError('Upstream "{}:{}" does not exist'.format(namespace_id, upstream_id))
            yield upstream_pb

    def remove_upstream(self, namespace_id, upstream_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type upstream_id:
        """
        zk_path = objects.UpstreamDescriptor.uid_to_zk_path(namespace_id, upstream_id)
        self.upstream_storage.remove(zk_path)

    def remove_namespace_upstreams(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.upstream_storage.remove(namespace_id, recursive=True)

    def does_upstream_exist(self, namespace_id, upstream_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type upstream_id:
        :rtype: bool
        """
        zk_path = objects.UpstreamDescriptor.uid_to_zk_path(namespace_id, upstream_id)
        return self.upstream_storage.exists(zk_path)

    def get_domain(self, namespace_id, domain_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        :param bool sync:
        :rtype: model_pb2.Domain | None
        """
        zk_path = objects.DomainDescriptor.uid_to_zk_path(namespace_id, domain_id)
        if sync:
            self.domain_storage.sync(zk_path)
        return self.domain_storage.get(zk_path)

    def must_get_domain(self, namespace_id, domain_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        :param bool sync:
        :rtype: model_pb2.Domain
        :raises: errors.NotFoundError
        """
        domain_pb = self.get_domain(namespace_id, domain_id, sync=sync)
        if domain_pb is None:
            raise errors.NotFoundError('Domain "{}:{}" does not exist'.format(namespace_id, domain_id))
        return domain_pb

    def create_domain(self, namespace_id, domain_id, domain_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        :type domain_pb: model_pb2.Domain
        :rtype: model_pb2.Domain
        :raises: errors.ConflictError
        """
        assert isinstance(domain_pb, model_pb2.Domain)
        assert domain_pb.meta.id == domain_id
        assert domain_pb.meta.namespace_id == namespace_id
        zk_path = objects.DomainDescriptor.uid_to_zk_path(namespace_id, domain_id)
        try:
            return self.domain_storage.create(zk_path, domain_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Domain "{}:{}" already exists'.format(namespace_id, domain_id))

    def update_domain(self, namespace_id, domain_id, domain_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        :type domain_pb: model_pb2.Domain | None
        :rtype: list[model_pb2.Domain]
        :raises: errors.NotFoundError
        """
        assert domain_pb is None or isinstance(domain_pb, model_pb2.Domain)
        assert domain_pb is None or domain_pb.meta.id == domain_id
        assert domain_pb is None or domain_pb.meta.namespace_id == namespace_id
        zk_path = objects.DomainDescriptor.uid_to_zk_path(namespace_id, domain_id)
        for domain_pb in self.domain_storage.guaranteed_update(zk_path, obj=domain_pb):
            if domain_pb is None:
                raise errors.NotFoundError('Domain "{}:{}" does not exist'.format(namespace_id, domain_id))
            yield domain_pb

    def update_domain_operation(self, namespace_id, domain_id, domain_operation_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        :type domain_operation_pb: model_pb2.DomainOperation | None
        :rtype: list[model_pb2.DomainOperation]
        :raises: errors.NotFoundError
        """
        assert domain_operation_pb is None or isinstance(domain_operation_pb, model_pb2.DomainOperation)
        assert domain_operation_pb is None or domain_operation_pb.meta.id == domain_id
        assert domain_operation_pb is None or domain_operation_pb.meta.namespace_id == namespace_id
        zk_path = objects.DomainOperationDescriptor.uid_to_zk_path(namespace_id, domain_id)
        for op_pb in self.domain_operation_storage.guaranteed_update(zk_path, obj=domain_operation_pb):
            if op_pb is None:
                raise errors.NotFoundError('Domain operation "{}:{}" does not exist'.format(namespace_id, domain_id))
            yield op_pb

    def remove_domain(self, namespace_id, domain_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        """
        zk_path = objects.DomainDescriptor.uid_to_zk_path(namespace_id, domain_id)
        self.domain_storage.remove(zk_path)

    def get_domain_operation(self, namespace_id, domain_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        :param bool sync:
        :rtype: model_pb2.DomainOperation | None
        """
        zk_path = objects.DomainOperationDescriptor.uid_to_zk_path(namespace_id, domain_id)
        if sync:
            self.domain_operation_storage.sync(zk_path)
        return self.domain_operation_storage.get(zk_path)

    def must_get_domain_operation(self, namespace_id, domain_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        :param bool sync:
        :rtype: model_pb2.DomainOperation
        """
        domain_op_pb = self.get_domain_operation(namespace_id, domain_id, sync=sync)
        if domain_op_pb is None:
            raise errors.NotFoundError('Domain operation "{}:{}" does not exist'.format(namespace_id, domain_id))
        return domain_op_pb

    def create_domain_operation(self, namespace_id, domain_id, domain_operation_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        :type domain_operation_pb: model_pb2.DomainOperation
        :rtype: model_pb2.DomainOperation
        :raises: errors.ConflictError
        """
        assert isinstance(domain_operation_pb, model_pb2.DomainOperation)
        assert domain_operation_pb.meta.id == domain_id
        assert domain_operation_pb.meta.namespace_id == namespace_id
        zk_path = objects.DomainOperationDescriptor.uid_to_zk_path(namespace_id, domain_id)
        try:
            return self.domain_operation_storage.create(zk_path, domain_operation_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Domain operation "{}:{}" already exists'.format(namespace_id, domain_id))

    def remove_domain_operation(self, namespace_id, domain_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        """
        zk_path = objects.DomainOperationDescriptor.uid_to_zk_path(namespace_id, domain_id)
        self.domain_operation_storage.remove(zk_path)

    def remove_namespace_domains(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.domain_storage.remove(namespace_id, recursive=True)

    def remove_namespace_domain_operations(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.domain_operation_storage.remove(namespace_id, recursive=True)

    def does_domain_exist(self, namespace_id, domain_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        :rtype: bool
        """
        zk_path = objects.DomainDescriptor.uid_to_zk_path(namespace_id, domain_id)
        return self.domain_storage.exists(zk_path)

    def does_domain_operation_exist(self, namespace_id, domain_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type domain_id:
        :rtype: bool
        """
        zk_path = objects.DomainOperationDescriptor.uid_to_zk_path(namespace_id, domain_id)
        return self.domain_operation_storage.exists(zk_path)

    def update_balancer_operation(self, namespace_id, balancer_id, balancer_operation_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :type balancer_operation_pb: model_pb2.BalancerOperation | None
        :rtype: list[model_pb2.BalancerOperation]
        :raises: errors.NotFoundError
        """
        assert balancer_operation_pb is None or isinstance(balancer_operation_pb, model_pb2.BalancerOperation)
        assert balancer_operation_pb is None or balancer_operation_pb.meta.id == balancer_id
        assert balancer_operation_pb is None or balancer_operation_pb.meta.namespace_id == namespace_id
        zk_path = objects.L7BalancerOperationDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        for m_pb in self.balancer_operation_storage.guaranteed_update(zk_path, obj=balancer_operation_pb):
            if m_pb is None:
                raise errors.NotFoundError('Balancer operation "{}:{}" does not exist'.format(namespace_id,
                                                                                              balancer_id))
            yield m_pb

    def get_balancer_operation(self, namespace_id, balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :param bool sync:
        :rtype: model_pb2.BalancerOperation | None
        """
        zk_path = objects.L7BalancerOperationDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        if sync:
            self.balancer_operation_storage.sync(zk_path)
        return self.balancer_operation_storage.get(zk_path)

    def must_get_balancer_operation(self, namespace_id, balancer_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :param bool sync:
        :rtype: model_pb2.BalancerOperation
        """
        op_pb = self.get_balancer_operation(namespace_id, balancer_id, sync=sync)
        if op_pb is None:
            raise errors.NotFoundError('Balancer operation "{}:{}" does not exist'.format(namespace_id, balancer_id))
        return op_pb

    def create_balancer_operation(self, namespace_id, balancer_id, balancer_operation_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        :type balancer_operation_pb: model_pb2.BalancerOperation
        :rtype: model_pb2.BalancerOperation
        :raises: errors.ConflictError
        """
        assert isinstance(balancer_operation_pb, model_pb2.BalancerOperation)
        assert balancer_operation_pb.meta.id == balancer_id
        assert balancer_operation_pb.meta.namespace_id == namespace_id
        zk_path = objects.L7BalancerOperationDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        try:
            return self.balancer_operation_storage.create(zk_path, balancer_operation_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Balancer operation "{}:{}" already exists'.format(namespace_id, balancer_id))

    def remove_balancer_operation(self, namespace_id, balancer_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type balancer_id:
        """
        zk_path = objects.L7BalancerOperationDescriptor.uid_to_zk_path(namespace_id, balancer_id)
        self.balancer_operation_storage.remove(zk_path)

    def sync_backends(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.backend_storage.sync(namespace_id)

    def get_backend(self, namespace_id, backend_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type backend_id:
        :param bool sync:
        :rtype: model_pb2.Backend | None
        """
        zk_path = objects.BackendDescriptor.uid_to_zk_path(namespace_id, backend_id)
        if sync:
            self.backend_storage.sync(zk_path)
        return self.backend_storage.get(zk_path)

    def must_get_backend(self, namespace_id, backend_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type backend_id:
        :param bool sync:
        :rtype: model_pb2.Backend
        :raises: errors.NotFoundError
        """
        backend_pb = self.get_backend(namespace_id, backend_id, sync=sync)
        if backend_pb is None:
            raise errors.NotFoundError('Backend "{}:{}" does not exist'.format(namespace_id, backend_id))
        return backend_pb

    def create_backend(self, namespace_id, backend_id, backend_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type backend_id:
        :type backend_pb: model_pb2.Backend
        :rtype: model_pb2.Backend
        :raises: errors.ConflictError
        """
        assert isinstance(backend_pb, model_pb2.Backend)
        assert backend_pb.meta.id == backend_id
        assert backend_pb.meta.namespace_id == namespace_id
        zk_path = objects.BackendDescriptor.uid_to_zk_path(namespace_id, backend_id)
        try:
            return self.backend_storage.create(zk_path, backend_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Backend "{}:{}" already exists'.format(namespace_id, backend_id))

    def update_backend(self, namespace_id, backend_id, backend_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type backend_id:
        :type backend_pb: model_pb2.Backend | None
        :rtype: list[model_pb2.Backend]
        :raises: errors.NotFoundError
        """
        assert backend_pb is None or isinstance(backend_pb, model_pb2.Backend)
        assert backend_pb is None or backend_pb.meta.id == backend_id
        assert backend_pb is None or backend_pb.meta.namespace_id == namespace_id
        zk_path = objects.BackendDescriptor.uid_to_zk_path(namespace_id, backend_id)
        for backend_pb in self.backend_storage.guaranteed_update(zk_path, obj=backend_pb):
            if backend_pb is None:
                raise errors.NotFoundError('Backend "{}:{}" does not exist'.format(namespace_id, backend_id))
            yield backend_pb

    def remove_backend(self, namespace_id, backend_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type backend_id:
        """
        zk_path = objects.BackendDescriptor.uid_to_zk_path(namespace_id, backend_id)
        self.backend_storage.remove(zk_path)

    def remove_namespace_backends(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.backend_storage.remove(namespace_id, recursive=True)

    def does_backend_exist(self, namespace_id, backend_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type backend_id:
        """
        zk_path = objects.BackendDescriptor.uid_to_zk_path(namespace_id, backend_id)
        return self.backend_storage.exists(zk_path)

    def get_endpoint_set(self, namespace_id, endpoint_set_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type endpoint_set_id:
        :param bool sync:
        :rtype: model_pb2.EndpointSet | None
        """
        zk_path = objects.EndpointSetDescriptor.uid_to_zk_path(namespace_id, endpoint_set_id)
        if sync:
            self.endpoint_set_storage.sync(zk_path)
        return self.endpoint_set_storage.get(zk_path)

    def must_get_endpoint_set(self, namespace_id, endpoint_set_id, sync=False):
        """
        :param six.text_type namespace_id:
        :param six.text_type endpoint_set_id:
        :param bool sync:
        :rtype: model_pb2.EndpointSet
        :raises: errors.NotFoundError
        """
        endpoint_set_pb = self.get_endpoint_set(namespace_id, endpoint_set_id, sync=sync)
        if endpoint_set_pb is None:
            raise errors.NotFoundError('Endpoint set "{}:{}" does not exist'.format(namespace_id, endpoint_set_id))
        return endpoint_set_pb

    def create_endpoint_set(self, namespace_id, endpoint_set_id, endpoint_set_pb):
        """
        :param six.text_type namespace_id:
        :param six.text_type endpoint_set_id:
        :type endpoint_set_pb: model_pb2.EndpointSet
        :rtype: model_pb2.EndpointSet
        :raises: errors.ConflictError
        """
        assert isinstance(endpoint_set_pb, model_pb2.EndpointSet)
        assert endpoint_set_pb.meta.id == endpoint_set_id
        assert endpoint_set_pb.meta.namespace_id == namespace_id
        zk_path = objects.EndpointSetDescriptor.uid_to_zk_path(namespace_id, endpoint_set_id)
        try:
            return self.endpoint_set_storage.create(zk_path, endpoint_set_pb)
        except NodeAlreadyExistsError:
            raise errors.ConflictError('Endpoint set "{}:{}" already exists'.format(namespace_id, endpoint_set_id))

    def update_endpoint_set(self, namespace_id, endpoint_set_id, endpoint_set_pb=None):
        """
        :param six.text_type namespace_id:
        :param six.text_type endpoint_set_id:
        :type endpoint_set_pb: model_pb2.EndpointSet | None
        :rtype: list[model_pb2.EndpointSet]
        :raises: errors.NotFoundError
        """
        assert endpoint_set_pb is None or isinstance(endpoint_set_pb, model_pb2.EndpointSet)
        assert endpoint_set_pb is None or endpoint_set_pb.meta.id == endpoint_set_id
        assert endpoint_set_pb is None or endpoint_set_pb.meta.namespace_id == namespace_id
        zk_path = objects.EndpointSetDescriptor.uid_to_zk_path(namespace_id, endpoint_set_id)
        for endpoint_set_pb in self.endpoint_set_storage.guaranteed_update(zk_path, obj=endpoint_set_pb):
            if endpoint_set_pb is None:
                raise errors.NotFoundError('Endpoint set "{}:{}" does not exist'.format(namespace_id, endpoint_set_id))
            yield endpoint_set_pb

    def remove_endpoint_set(self, namespace_id, endpoint_set_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type endpoint_set_id:
        """
        zk_path = objects.EndpointSetDescriptor.uid_to_zk_path(namespace_id, endpoint_set_id)
        self.endpoint_set_storage.remove(zk_path)

    def remove_namespace_endpoint_sets(self, namespace_id):
        """
        :param six.text_type namespace_id:
        """
        self.endpoint_set_storage.remove(namespace_id, recursive=True)

    def does_endpoint_set_exist(self, namespace_id, endpoint_set_id):
        """
        :param six.text_type namespace_id:
        :param six.text_type endpoint_set_id:
        :rtype: bool
        """
        zk_path = objects.EndpointSetDescriptor.uid_to_zk_path(namespace_id, endpoint_set_id)
        return self.endpoint_set_storage.exists(zk_path)
