# coding: utf-8
from datetime import datetime

import bson
import cachetools

from awacs.lib.mongo import get_db
from infra.awacs.proto import internals_pb2


class IGencfgGroupInstancesCache(object):
    pass


class GencfgGroupInstancesCache(IGencfgGroupInstancesCache):
    def __init__(self, mem_maxsize=100):
        self.mem_cache = cachetools.LRUCache(maxsize=mem_maxsize)

    @property
    def _col(self):
        return get_db().gencfg_group_instances_cache

    def ensure_indexes(self):
        self._col.ensure_index('name', name='name_index')
        self._col.ensure_index('version', name='version_index')

    @staticmethod
    def _format_id(name, version, use_mtn):
        rv = name + ':' + version
        if use_mtn:
            rv += ':1'
        return rv

    def cache_gencfg_group_instances(self, name, version, use_mtn, instances):
        """
        :type name: str
        :type version: str
        :type use_mtn: bool
        :type instances: list[awacs.proto.internals_pb2.Instance]
        """
        id_ = self._format_id(name, version, use_mtn)
        content = internals_pb2.InstancesList(instances=instances)
        self.mem_cache[id_] = content
        now = datetime.utcnow()
        self._col.find_and_modify({
            '_id': id_,
        }, {
            '_id': id_,
            'name': name,
            'version': version,
            'use_mtn': use_mtn,
            'content': bson.Binary(content.SerializeToString()),
            'mtime': now,
            'atime': now,
        }, upsert=True)

    def get_gencfg_group_instances(self, name, version, use_mtn):
        """
        :type name: str
        :type version: str
        :type use_mtn: bool
        :rtype: list[awacs.proto.internals_pb2.Instance]
        """
        id_ = self._format_id(name, version, use_mtn)
        if id_ in self.mem_cache:
            return list(self.mem_cache[id_].instances)
        data = self._col.find_and_modify({
            '_id': id_,
        }, {
            '$set': {
                'atime': datetime.utcnow(),
            },
        }, new=True, full_response=True, projection=('mtime', 'content'))
        if not data['value']:
            return None
        content = internals_pb2.InstancesList()
        content.ParseFromString(data['value']['content'])
        self.mem_cache[id_] = content
        return list(content.instances)

    def get_gencfg_group_instances_cache_entry(self, name, version, use_mtn):
        return self._col.find_one(self._format_id(name, version, use_mtn))
