# coding: utf-8
import json
import logging

import gevent
import requests

from infra.swatlib.auth import gencfg
from infra.awacs.proto import internals_pb2


DEFAULT_INSTANCE_WEIGHT = 1.0


class GencfgClientError(Exception):
    pass


class IGencfgClient(object):
    pass


class GencfgClient(object):
    def __init__(self, gencfg_client, cache=None):
        """
        :type gencfg_client: awacs.lib.gencfg_client.GencfgClient
        """
        self._gencfg_client = gencfg_client
        self._cache = cache
        self._log = logging.getLogger('gencfg_client')

    @classmethod
    def from_config(cls, d, cache=None):
        return cls(gencfg_client=gencfg.GencfgClient.from_config(d), cache=cache)

    @staticmethod
    def _create_instance(data, use_mtn):
        """
        :type data: dict
        :type use_mtn: bool
        """
        if use_mtn:
            if 'hbf' not in data:
                raise GencfgClientError('Invalid instance: "hbf" is missing: {!r}'.format(data))
            if 'interfaces' not in data['hbf']:
                raise GencfgClientError('Invalid instance: "hbf.interfaces" is missing: {!r}'.format(data))
            if 'backbone' not in data['hbf']['interfaces']:
                raise GencfgClientError('Invalid instance: "hbf.interfaces.backbone" is missing: {!r}'.format(data))
            host_info = data['hbf']['interfaces']['backbone']
        else:
            host_info = data
        hostname = host_info['hostname']
        ipv4_addr = host_info.get('ipv4addr')
        ipv6_addr = host_info.get('ipv6addr')

        return internals_pb2.Instance(
            host=hostname,
            port=data['port'],
            ipv4_addr=ipv4_addr,
            ipv6_addr=ipv6_addr,
            weight=float(data.get('power', DEFAULT_INSTANCE_WEIGHT)),
        )

    def do_list_group_instances_data(self, name, version):
        try:
            json_str = self._gencfg_client.list_group_instances_json(name, version)
        except requests.HTTPError as e:
            raise GencfgClientError('Request "{}" failed with {} {}: {}'.format(e.request.url,
                                                                                e.response.status_code,
                                                                                e.response.reason,
                                                                                e.response.content))
        except (requests.RequestException, gevent.Timeout) as e:
            raise GencfgClientError('Request to gencfg for instance '
                                    'group "{}" an tag "{}" failed: {}'.format(name, version, e))
        try:
            data = json.loads(json_str)
        except ValueError as e:
            raise GencfgClientError('Gencfg returned invalid JSON: {}'.format(e))
        return data['instances']

    def list_group_instances(self, name, version, use_mtn, use_cache=True):
        use_cache = use_cache and self._cache is not None

        cacheable = (version != 'trunk')
        if use_cache and cacheable:
            cached_group_instances = self._cache.get_gencfg_group_instances(name, version, use_mtn)
            if cached_group_instances is not None:
                return cached_group_instances

        instances_data = self.do_list_group_instances_data(name, version)
        instances = []
        for instance_data in instances_data:
            instances.append(self._create_instance(instance_data, use_mtn))

        if self._cache and cacheable:
            self._cache.cache_gencfg_group_instances(name, version, use_mtn, instances)

        return instances

    def get_group_card(self, name, version):
        try:
            card = self._gencfg_client.get_group_card(name, version)
        except requests.HTTPError as e:
            raise GencfgClientError('Request "{}" failed with {} {}: {}'.format(e.request.url,
                                                                                e.response.status_code,
                                                                                e.response.reason,
                                                                                e.response.content))
        except (requests.RequestException, gevent.Timeout) as e:
            raise GencfgClientError('Request to gencfg for instance '
                                    'group "{}" an tag "{}" failed: {}'.format(name, version, e))
        return card
