# coding: utf-8
from datetime import datetime

import bson
import cachetools

from awacs.lib.mongo import get_db
from infra.awacs.proto import internals_pb2


class INannyInstancesCache(object):
    pass


class NannyInstancesCache(INannyInstancesCache):
    def __init__(self, mem_maxsize=100):
        self.mem_cache = cachetools.LRUCache(maxsize=mem_maxsize)

    @property
    def _col(self):
        return get_db().nanny_instances_cache

    def ensure_indexes(self):
        self._col.ensure_index('service_id', name='service_id_index')
        self._col.ensure_index('snapshot_id', name='snapshot_id_index')

    @staticmethod
    def _format_id(service_id, snapshot_id, use_mtn):
        rv = service_id + ':' + snapshot_id
        if use_mtn:
            rv += ':1'
        return rv

    def cache_nanny_instances(self, service_id, snapshot_id, use_mtn, instances):
        """
        :type service_id: str
        :type snapshot_id: str
        :type use_mtn: bool
        :type instances: list[awacs.proto.internals_pb2.Instance]
        """
        id_ = self._format_id(service_id, snapshot_id, use_mtn)
        content = internals_pb2.InstancesList(instances=instances)
        self.mem_cache[id_] = content
        now = datetime.utcnow()
        self._col.find_and_modify({
            '_id': id_,
        }, {
            '_id': id_,
            'service_id': service_id,
            'snapshot_id': snapshot_id,
            'use_mtn': use_mtn,
            'content': bson.Binary(content.SerializeToString()),
            'mtime': now,
            'atime': now,
        }, upsert=True)

    def get_nanny_instances(self, service_id, snapshot_id, use_mtn):
        """
        :type service_id: str
        :type snapshot_id: str
        :type use_mtn: bool
        :rtype: list[awacs.proto.internals_pb2.Instance]
        """
        id_ = self._format_id(service_id, snapshot_id, use_mtn)
        if id_ in self.mem_cache:
            return list(self.mem_cache[id_].instances)
        data = self._col.find_and_modify({
            '_id': id_,
        }, {
            '$set': {
                'atime': datetime.utcnow(),
            },
        }, new=True, full_response=True, projection=('mtime', 'content'))
        if not data['value']:
            return None
        content = internals_pb2.InstancesList()
        content.ParseFromString(data['value']['content'])
        self.mem_cache[id_] = content
        return list(content.instances)

    def get_nanny_instances_cache_entry(self, name, version, use_mtn):
        return self._col.find_one(self._format_id(name, version, use_mtn))
