# coding: utf-8
import logging
import collections

import gevent
import requests

from awacs.lib import nannyclient
from awacs.resolver.gencfg.client import GencfgClientError
from awacs.resolver.yp.util import (list_allocation_instances,
                                    make_pod_filter_from_service_id,
                                    PodIsNotAssigned,
                                    PodDoesNotHaveIpAddr)
from infra.awacs.proto import internals_pb2
from .. import gencfg
from ..errors import ResolvingError
import six


class NannyClientError(Exception):
    pass


class INannyClient(object):
    pass


class NannyClient(INannyClient):
    def __init__(self, nanny_client, gencfg_client, yp_client_factory, cache=None):
        """
        :type nanny_client: awacs.lib.nannyclient.NannyClient
        :type gencfg_client: awacs.resolver.gencfg.GencfgClient
        :type yp_client_factory: awacs.lib.ypclient.YpObjectServiceClientFactory
        :type cache: awacs.resolver.nanny.cache.NannyInstancesCache
        """
        self._nanny_client = nanny_client
        self._gencfg_client = gencfg_client
        self._yp_client_factory = yp_client_factory
        self._cache = cache
        self._log = logging.getLogger('nannyclient')

    @classmethod
    def from_config(cls, d, gencfg_client, yp_client_factory, cache=None):
        return cls(
            nanny_client=nannyclient.NannyClient.from_config(d),
            gencfg_client=gencfg_client,
            yp_client_factory=yp_client_factory,
            cache=cache
        )

    @staticmethod
    def _create_instance_from_instance_list_instance(i):
        """
        :param i: awacs.lib.nannyclient.Instance
        :rtype: internals_pb2.Instance
        """
        return internals_pb2.Instance(
            host=i.host,
            port=i.port,
            weight=i.power,
            ipv4_addr=i.ipv4_address,
            ipv6_addr=i.ipv6_address,
        )

    def _do_get_snapshot_instances_section(self, service_id, snapshot_id):
        """
        This method *does not* require authorized NannyClient (with correct OAuth token).
        """
        try:
            return self._nanny_client.get_snapshot_instances_section(snapshot_id)
        except (nannyclient.NannyApiRequestException, requests.RequestException, gevent.Timeout) as e:
            raise NannyClientError('Request to Nanny API for instances section of {}:{} '
                                   'has failed: {}'.format(service_id, snapshot_id, e))

    def process_nanny_instances(self, service_id, service_instances, use_mtn):
        """
        :type service_id: str
        :type service_instances: awacs.lib.nannyclient.ServiceInstances
        :type use_mtn: bool
        :rtype: list[awacs.proto.internals_pb2.Instance]
        """
        types = nannyclient.ServiceInstances.InstanceType
        type = service_instances.chosen_type
        if type == types.INSTANCE_LIST:
            instances = []
            for i in service_instances.instance_list:
                instances.append(self._create_instance_from_instance_list_instance(i))
            return instances
        elif type in (types.GENCFG_GROUPS, types.EXTENDED_GENCFG_GROUPS):
            if type == types.GENCFG_GROUPS:
                groups = service_instances.gencfg_groups
            elif type == types.EXTENDED_GENCFG_GROUPS:
                groups = service_instances.extended_gencfg_groups.groups
            else:
                raise AssertionError()
            instances_by_groups = []
            for g in groups:
                name, version = g.name, g.release
                try:
                    instance_pbs = self._gencfg_client.list_group_instances(name, version, use_mtn)
                    instances_by_groups.append(((name, version), instance_pbs))
                except GencfgClientError as e:
                    raise NannyClientError(
                        'Failed to resolve gencfg group {}:{} (use_mtn: {}): {}'.format(name, version, use_mtn, e))
            merged_instances = gencfg.merge_gencfg_group_instances(instances_by_groups)
            return merged_instances
        elif type == types.YP_PODS:
            instances = []
            allocation_pod_filters_by_cluster = collections.defaultdict(set)
            for allocation in service_instances.yp_pods.allocations:
                allocation_pod_filters_by_cluster[allocation.cluster.lower()].add(allocation.pod_filter)
            for cluster, pod_filters in six.iteritems(allocation_pod_filters_by_cluster):
                stub = self._yp_client_factory.get(cluster)
                for pod_filter in pod_filters:
                    try:
                        instances.extend(list_allocation_instances(stub, pod_filter))
                    except (PodIsNotAssigned, PodDoesNotHaveIpAddr) as e:
                        raise NannyClientError(
                            'Failed to resolve pod filter {} (cluster {}): {}'.format(pod_filter, cluster, e))
            return instances
        elif type == types.YP_POD_IDS:
            instances = []
            pod_filter = make_pod_filter_from_service_id(service_id)
            for cluster, pod_ids in service_instances.yp_pod_ids.group_pod_ids_by_cluster():
                stub = self._yp_client_factory.get(cluster.lower())
                try:
                    instances.extend(list_allocation_instances(stub, pod_filter, pod_ids))
                except (PodIsNotAssigned, PodDoesNotHaveIpAddr) as e:
                    raise NannyClientError(
                        'Failed to resolve pod filter {} (cluster {}): {}'.format(pod_filter, cluster, e))
            return instances
        else:
            raise ValueError('Unknown chosen_type {!r}'.format(service_instances.chosen_type))

    def list_nanny_snapshot_instances(self, service_id, snapshot_id, use_mtn, use_cache=True):
        """
        :rtype: list[awacs.proto.internals_pb2.Instance]
        """
        use_cache = use_cache and self._cache is not None

        if use_cache:
            cached_instances = self._cache.get_nanny_instances(service_id, snapshot_id, use_mtn)
            if cached_instances is not None:
                return cached_instances

        instances_section = self._do_get_snapshot_instances_section(service_id, snapshot_id)
        try:
            instances = self.process_nanny_instances(service_id, instances_section, use_mtn)
        except NannyClientError as e:
            raise ResolvingError('Failed to resolve {}:{} (use_mtn: {}) '
                                 'snapshot instances: {}'.format(service_id, snapshot_id, use_mtn, e))

        if self._cache:
            self._cache.cache_nanny_instances(service_id, snapshot_id, use_mtn, instances)

        return instances
