# coding: utf-8
import socket

from six.moves import range

from . import errors


DEFAULT_INSTANCE_WEIGHT = 1.0
RESOLVE_HOSTNAME_ATTEMPTS = 2


def _resolve_host(host, ip_version=socket.AF_INET, attempts=RESOLVE_HOSTNAME_ATTEMPTS):
    """
    :type host: six.text_type
    :param ip_version: socket.AF_INET or socket.AF_INET6
    :type attempts: int
    :rtype: six.text_type
    """
    if ip_version not in (socket.AF_INET, socket.AF_INET6):
        raise ValueError('Unknown IP version {}'.format(ip_version))

    error = None
    for _ in range(attempts):
        try:
            resolved = socket.getaddrinfo(host, None, ip_version)
            return sorted(ip[4][0] for ip in resolved)[0]
        except socket.gaierror as e:
            error = e
            if e.errno in (socket.EAI_NODATA, socket.EAI_NONAME):
                # next attempt would be useless
                break
    assert error
    raise error


def resolve_host(host, ipv6_only=False):
    """
    :type host: six.text_type
    :type ipv6_only: bool
    :rtype: six.text_type
    :raises: InstanceIpAddressResolvingError
    """
    ip_version = (socket.AF_INET, socket.AF_INET6)
    if ipv6_only:
        ip_version = (socket.AF_INET6,)

    err = None
    for ip_version in ip_version:
        try:
            return _resolve_host(host, ip_version=ip_version)
        except socket.gaierror as e:
            err = e
    raise errors.InstanceIpAddressResolvingError('Cannot resolve {} address: {}'.format(host, err))


def get_resolved_ip(instance, ipv6_only=False):
    """
    :type instance: awacs.proto.internals_pb2.Instance
    :type ipv6_only: bool
    :raises: InstanceIpAddressResolvingError
    :rtype: str
    """
    if ipv6_only:
        resolved_ip = instance.ipv6_addr
    else:
        resolved_ip = instance.ipv4_addr or instance.ipv6_addr
    if not resolved_ip:
        resolved_ip = resolve_host(instance.host, ipv6_only=ipv6_only)
    return resolved_ip


def strip_host_domain(host):
    """
    :type host: six.text_type
    :rtype: six.text_type
    """
    return host.replace('.search.yandex.net', '').replace('.yandex.ru', '')
