# coding: utf-8
import logging

import nanny_rpc_client
import yt.yson as yson

from infra.awacs.proto import internals_pb2


import yp.data_model as data_model
from yp_proto.yp.client.api.proto import object_service_pb2


OT_POD = data_model.OT_POD
OT_ENDPOINT = data_model.OT_ENDPOINT
OT_ENDPOINT_SET = data_model.OT_ENDPOINT_SET


DEFAULT_PORT = 80

log = logging.getLogger('yp.util')


class PodIsNotAssigned(Exception):
    pass


class PodDoesNotHaveIpAddr(Exception):
    pass


def make_pod_filter_from_service_id(service_id):
    """
    :param str service_id:
    :rtype: str
    """
    pod_set_id = service_id.replace('_', '-')
    return '[/meta/pod_set_id] = "{}"'.format(pod_set_id)


def list_allocation_instances(stub, pod_filter, pod_ids=None):
    """
    :param stub:
    :param str pod_filter:
    :param set[str] | None pod_ids:
    :rtype: list[internals_pb2.Instance]
    """
    instance_pbs = []
    log.debug('Resolving pod filter {}'.format(pod_filter))

    req_pb = object_service_pb2.TReqSelectObjects(object_type=OT_POD)
    req_pb.selector.paths.append('/meta/id')
    req_pb.selector.paths.append('/status')
    req_pb.filter.query = pod_filter
    resp_pb = stub.SelectObjects(req_pb)
    for pod_pb in resp_pb.results:
        pod_id = yson.loads(pod_pb.values[0])
        if pod_ids is not None and pod_id not in pod_ids:
            continue

        status_yson = yson.loads(pod_pb.values[1])
        scheduling_state = status_yson['scheduling']['state']
        if scheduling_state != 'assigned':
            raise PodIsNotAssigned('Pod "{}" is not assigned: {}'.format(pod_id, scheduling_state))
        ip6_address_allocations = status_yson['ip6_address_allocations']
        if not ip6_address_allocations:
            raise PodDoesNotHaveIpAddr('Pod "{}" is not assigned: {}'.format(pod_id, scheduling_state))
        for addr_allocation in status_yson['ip6_address_allocations']:
            if addr_allocation['vlan_id'] == 'backbone':
                instance_pbs.append(internals_pb2.Instance(
                    host=addr_allocation['persistent_fqdn'],
                    port=DEFAULT_PORT,
                    ipv6_addr=addr_allocation['address'],
                    weight=1
                ))
    return instance_pbs


def does_endpoint_set_exist(stub, endpoint_set_id):
    """
    :param stub:
    :param six.text_type endpoint_set_id:
    :rtype: bool
    """
    req_pb = object_service_pb2.TReqGetObject(object_type=OT_ENDPOINT_SET)
    req_pb.selector.paths.append('/meta/id')
    req_pb.object_id = endpoint_set_id
    try:
        resp_pb = stub.GetObject(req_pb)
    except nanny_rpc_client.exceptions.NotFoundError:
        return False
    return resp_pb.HasField('result')


def list_endpoint_set_instances(stub, endpoint_set_id):
    """
    :param stub:
    :param six.text_type endpoint_set_id:
    :rtype: list[internals_pb2.Instance]
    """
    instance_pbs = []
    req_pb = object_service_pb2.TReqSelectObjects(object_type=OT_ENDPOINT)
    req_pb.selector.paths.append('/meta/id')
    req_pb.selector.paths.append('/spec')
    req_pb.filter.query = '[/meta/endpoint_set_id] = "{}"'.format(endpoint_set_id)
    resp_pb = stub.SelectObjects(req_pb)
    for endpoint_set_pb in resp_pb.results:
        # endpoint_id = endpoint_set_pb.values[0]
        spec_yson = yson.loads(endpoint_set_pb.values[1])
        if not spec_yson['protocol']:
            spec_yson['protocol'] = 'TCP'
        assert spec_yson['protocol'] == 'TCP'
        instance_pbs.append(internals_pb2.Instance(
            host=spec_yson['fqdn'],
            port=spec_yson['port'],
            ipv6_addr=spec_yson['ip6_address'],
            weight=1
        ))
    # TODO handle duplicates?
    return instance_pbs


def list_endpoint_set_instances_sd(resolver, req_pb, req_id=None):
    """
    :type resolver: awacs.lib.yp_service_discovery.sd_resolver.Resolver
    :type req_pb: internals_pb2.TReqResolveEndpoints
    :type req_id: six.text_type
    :rtype: tuple[internals_pb2.TRspResolveEndpoints, list[internals_pb2.Instance]]
    """
    resp_pb = resolver.resolve_endpoints(req_pb=req_pb, req_id=req_id)
    instance_pbs = []
    for e_pb in resp_pb.endpoint_set.endpoints:
        instance_pbs.append(internals_pb2.Instance(
            host=e_pb.fqdn,
            port=e_pb.port,
            ipv6_addr=e_pb.ip6_address,
            weight=1
        ))
    return resp_pb, instance_pbs
