# coding: utf-8
"""
Flask application initialization
"""
import flask
import raven.contrib.flask
import six

from sepelib.core import config
from infra.swatlib.stacksampler import blueprint as stack_sampler_bp

from .oauth_service import oauth_service_bp
from .knob_service import knob_service_bp
from .domain_service import domain_service_bp
from .certificate_service import certificate_service_bp
from .namespace_service import namespace_service_bp
from .component_service import component_service_bp
from .balancer_service import balancer_service_bp
from .l3_balancer_service import l3_balancer_service_bp
from .upstream_service import upstream_service_bp
from .backend_service import backend_service_bp
from .endpoint_set_service import endpoint_set_service_bp
from .status_service import status_service_bp
from .helpers_service import helpers_service_bp
from .dns_record_service import dns_record_service_bp
from .weight_section_service import weight_section_service_bp
from .l7heavy_config_service import l7heavy_config_service_bp
from .statistics_service import statistics_service_bp
from .ui_service import index_bp
from .info_service import info_service_bp
from .debug import debug_blueprint


def configure_sentry(app, sentry_client=None):
    """
    :type app: flask.Flask
    :type sentry_client: raven.Client
    """
    raven.contrib.flask.Sentry(app, client=sentry_client)


def setup_cors_and_headers(app, allowed_origins, version, version_timestamp, hostname):
    """
    :type app: flask.Flask
    :type allowed_origins: list[six.text_type]
    :type version: six.binary_type
    :type version_timestamp: six.binary_type
    :type hostname: six.binary_type
    """

    @app.before_request
    def handle_options_request():
        if flask.request.method == 'OPTIONS':
            return ''  # skip authentication process

    @app.after_request
    def add_headers(response):
        response.headers['X-Backend-Version'] = version
        response.headers['X-Backend-Version-Timestamp'] = version_timestamp
        response.headers['X-Backend-Host'] = hostname
        response.headers['Access-Control-Allow-Methods'] = 'PUT, DELETE'
        response.headers['Access-Control-Expose-Headers'] = ('etag,x-total-items,x-backend-version,'
                                                             'x-backend-version-timestamp')
        response.headers['Access-Control-Allow-Headers'] = 'content-type,if-match'
        response.headers['Access-Control-Allow-Credentials'] = 'true'
        origin = flask.request.environ.get('HTTP_ORIGIN')
        if origin in allowed_origins:
            if origin == 'null':
                response.headers['Access-Control-Allow-Origin'] = '*'
            else:
                response.headers['Access-Control-Allow-Origin'] = origin
        return response


def create_app(name, hostname, version, version_timestamp, sentry_client=None, debug=False, production=False):
    """
    :type name: str
    :type hostname: str
    :type version: str
    :type version_timestamp: int
    :type sentry_client: raven.Client
    :type debug: bool
    :type production: bool
    :rtype: flask.Flask
    """
    # Create application
    app = flask.Flask(name)
    # By default Flask sorts keys in json to improve cache ability.
    # Our users don't cache responses based on content,
    # so let's try to improve performance.
    app.config['JSON_SORT_KEYS'] = False

    allowed_origins = config.get_value('web.http.cors.allowed_origins', [])
    setup_cors_and_headers(app,
                           allowed_origins=allowed_origins,
                           version=version,
                           version_timestamp=six.text_type(version_timestamp).encode('ascii'),
                           hostname=hostname)
    app.hostname = hostname
    app.version = version

    # Additional customization
    app.debug = debug
    # Configure assets
    if sentry_client is not None:
        configure_sentry(app, sentry_client)
    # Finally, register url handlers
    app.register_blueprint(oauth_service_bp)
    app.register_blueprint(knob_service_bp)
    app.register_blueprint(domain_service_bp)
    app.register_blueprint(certificate_service_bp)
    app.register_blueprint(dns_record_service_bp)
    app.register_blueprint(namespace_service_bp)
    app.register_blueprint(component_service_bp)
    app.register_blueprint(balancer_service_bp)
    app.register_blueprint(l3_balancer_service_bp)
    app.register_blueprint(upstream_service_bp)
    app.register_blueprint(backend_service_bp)
    app.register_blueprint(endpoint_set_service_bp)
    app.register_blueprint(weight_section_service_bp)
    app.register_blueprint(l7heavy_config_service_bp)
    app.register_blueprint(statistics_service_bp)
    app.register_blueprint(helpers_service_bp)
    app.register_blueprint(index_bp)
    app.register_blueprint(info_service_bp)
    app.register_blueprint(debug_blueprint)
    return app


def create_status_app(name, hostname, version, version_timestamp, sentry_client=None, debug=False, production=False):
    """
    :type name: str
    :type hostname: str
    :type version: str
    :type version_timestamp: int
    :type sentry_client: raven.Client
    :type debug: bool
    :type production: bool
    :rtype: flask.Flask
    """
    # Create application
    app = flask.Flask(name)
    # By default Flask sorts keys in json to improve cache ability.
    # Our users don't cache responses based on content,
    # so let's try to improve performance.
    app.config['JSON_SORT_KEYS'] = False

    allowed_origins = config.get_value('web.http.cors.allowed_origins', [])
    setup_cors_and_headers(app,
                           allowed_origins=allowed_origins,
                           version=version,
                           version_timestamp=six.text_type(version_timestamp).encode('ascii'),
                           hostname=hostname)
    app.hostname = hostname
    app.version = version

    # Additional customization
    app.debug = debug
    # Configure assets
    if sentry_client is not None:
        configure_sentry(app, sentry_client)
    # Finally, register url handlers
    app.register_blueprint(status_service_bp)
    app.register_blueprint(stack_sampler_bp.sampler_bp)
    return app
