# coding: utf-8
import logging

import inject
from sepelib.core import config

from awacs.lib import staffclient, nannystaffcache
from awacs.lib.rpc.exceptions import ForbiddenError, BadRequestError
from infra.awacs.proto import model_pb2
from .actions import ALL_PERMISSIONS, EDIT_SPEC, EDIT_AUTH, REMOVE, CREATE


log = logging.getLogger(__name__)


class StaffAuth(object):
    staff_client = inject.attr(staffclient.IStaffClient)
    nanny_staff_cache = inject.attr(nannystaffcache.INannyStaffCache)

    @classmethod
    def _get_group_ids(cls, login):
        try:
            return cls.staff_client.resolve_login_to_group_ids(login)
        except Exception:
            log.exception('Failed to retrieve staff group ids for %s, using an empty list...', login)
            return []

    @classmethod
    def _get_cached_group_ids(cls, login):
        try:
            return cls.nanny_staff_cache.get_group_ids(login)
        except nannystaffcache.NannyStaffCacheEntryNotFound:
            log.exception('Failed to retrieve cached staff group ids for %s, using an empty list...', login)
            return []

    @classmethod
    def get_allowed_actions(cls, acl_pb, login):
        if login in acl_pb.owners.logins:
            return ALL_PERMISSIONS

        owner_group_ids = set(acl_pb.owners.group_ids)
        if owner_group_ids:
            group_ids = cls._get_group_ids(login)
            for group_id in group_ids:
                if group_id in owner_group_ids:
                    return ALL_PERMISSIONS

            cached_group_ids = cls._get_cached_group_ids(login)
            for group_id in cached_group_ids:
                if group_id in owner_group_ids:
                    return ALL_PERMISSIONS

        return set()

    @classmethod
    def authorize_create(cls, namespace_id, acl_pb, login):
        if CREATE not in cls.get_allowed_actions(acl_pb, login):
            raise ForbiddenError('User "{}" is not authorized '
                                 'to create objects in namespace "{}"'.format(login, namespace_id))

    @classmethod
    def authorize_update(cls, acl_pb, pb, req_pb, login, implicit_spec_update=False):
        """
        :param acl_pb: model_pb2.StaffAuth
        :param pb: current pb
        :param req_pb: request pb
        :param str login:
        :param bool implicit_spec_update: If your request_pb does not have a `spec` field,
            set this param to True
        :return:
        """
        actions = set()

        pb_fields = pb.DESCRIPTOR.fields_by_name
        req_fields = req_pb.DESCRIPTOR.fields_by_name

        if 'meta' in req_fields:
            if 'meta' not in pb_fields:
                raise RuntimeError('"req_pb" has "meta" field that is missing from "pb"')

            if req_pb.HasField('meta'):
                pb_meta_fields = pb.meta.DESCRIPTOR.fields_by_name
                req_meta_fields = req_pb.meta.DESCRIPTOR.fields_by_name
                if 'auth' in req_meta_fields:
                    if 'auth' not in pb_meta_fields:
                        raise RuntimeError('"req_pb" has "meta.auth" field that is missing from "pb"')
                    if req_pb.meta.HasField('auth') and pb.meta.auth != req_pb.meta.auth:
                        actions.add(EDIT_AUTH)

        if 'spec' in req_fields:
            if 'spec' not in pb_fields:
                raise RuntimeError('"req_pb" has "spec" field that is missing from "pb"')

            if req_pb.HasField('spec') and pb.spec != req_pb.spec:
                actions.add(EDIT_SPEC)

        if implicit_spec_update:
            actions.add(EDIT_SPEC)

        allowed_actions = cls.get_allowed_actions(acl_pb, login)

        if not actions.issubset(allowed_actions):
            forbidden_action_names = []
            for forbidden_action in sorted(actions - allowed_actions):
                if forbidden_action == EDIT_AUTH:
                    forbidden_action_names.append('EDIT_AUTH')
                elif forbidden_action == EDIT_SPEC:
                    forbidden_action_names.append('EDIT_SPEC')
                else:
                    raise RuntimeError('Unknown action: {}'.format(forbidden_action))
            raise ForbiddenError('User "{}" is not authorized to perform such actions: "{}"'.format(
                login, '", "'.join(forbidden_action_names)))

    @classmethod
    def authorize_remove(cls, acl_pb, login):
        """
        :type acl_pb: model_pb2.StaffAuth
        :type login: str
        :raises: ForbiddenError
        """
        allowed_actions = cls.get_allowed_actions(acl_pb, login)
        if REMOVE not in allowed_actions:
            raise ForbiddenError('User "{}" is not authorized to remove this object'.format(login))


def get_acl(pb):
    auth_type = pb.meta.auth.type
    if auth_type == pb.meta.auth.STAFF:
        return pb.meta.auth.staff
    elif auth_type == pb.meta.auth.NONE:
        raise BadRequestError('ACL is not specified')
    else:
        auth_type_name = pb.meta.auth.AuthType.Name(auth_type)
        raise BadRequestError('Auth type "{}" is not supported'.format(auth_type_name))


def authorize_create(namespace_pb, auth_subject):
    """
    :param namespace_pb:
    :type auth_subject: awacs.lib.rpc.authentication.AuthSubject
    """
    login = auth_subject.login
    if (not config.get_value('run.auth', default=True) or
            login in config.get_value('run.root_users', default=())):
        return

    acl = get_acl(namespace_pb)
    if isinstance(acl, model_pb2.StaffAuth):
        StaffAuth.authorize_create(namespace_pb.meta.id, acl, login)
    else:
        raise AssertionError('acl is not model_pb2.StaffAuth')


def authorize_update(pb, update_req_pb, auth_subject, acl=None, implicit_spec_update=False):
    """
    :param pb:
    :param update_req_pb:
    :param auth_subject: awacs.lib.rpc.authentication.AuthSubject
    :param bool implicit_spec_update: If your request_pb does not have a `spec` field,
        set this param to True
    """
    login = auth_subject.login
    if (not config.get_value('run.auth', default=True) or
            login in config.get_value('run.root_users', default=())):
        return

    acl = acl or get_acl(pb)
    if isinstance(acl, model_pb2.StaffAuth):
        StaffAuth.authorize_update(
            acl, pb, update_req_pb, login, implicit_spec_update=implicit_spec_update)
    else:
        raise AssertionError('acl is not model_pb2.StaffAuth')


def authorize_cert_update(pb, update_req_pb, auth_subject, acl=None):
    authorize_update(pb, update_req_pb, auth_subject, acl)
    if not config.get_value('run.auth', True) or auth_subject.login in config.get_value('run.root_users', ()):
        return
    spec_has_changed = update_req_pb.HasField('spec') and pb.spec != update_req_pb.spec
    if spec_has_changed:
        raise ForbiddenError('Only admins can update certificate spec')


def authorize_remove(pb, auth_subject, acl=None):
    """
    :param pb:
    :param auth_subject: awacs.lib.rpc.authentication.AuthSubject
    """
    acl = acl or get_acl(pb)
    login = auth_subject.login
    if (not config.get_value('run.auth', default=True) or
            login in config.get_value('run.root_users', default=())):
        return

    acl = acl or get_acl(pb)
    if isinstance(acl, model_pb2.StaffAuth):
        StaffAuth.authorize_remove(acl, login)
    else:
        raise AssertionError('acl is not model_pb2.StaffAuth')


def is_root(auth_subject):
    """
    :param auth_subject: awacs.lib.rpc.authentication.AuthSubject
    :rtype: bool
    """
    login = auth_subject.login
    return (
        not config.get_value('run.auth', default=True)
        or login in config.get_value('run.root_users', default=())
    )


def authorize_root_operation(auth_subject, operation):
    """
    :param auth_subject: awacs.lib.rpc.authentication.AuthSubject
    :param str operation: operation name
    """
    if is_root(auth_subject):
        return

    raise ForbiddenError('Only admins are allowed to {}'.format(operation))
