# coding: utf-8
import re
from sepelib.core import config

from awacs.lib import rpc
from awacs.lib.rpc.exceptions import ForbiddenError
from awacs.lib.gutils import gevent_idle_iter
from awacs.model import db, apicache, cache
from awacs.model.dao import IDao
from awacs.model.db import IMongoStorage
from awacs.model.util import clone_pb, omit_duplicate_items_from_auth
from awacs.model.zk import IZkStorage
from infra.awacs.proto import api_pb2, model_pb2
from awacs.web.auth.core import authorize_update, authorize_remove, get_acl, authorize_create
from awacs.web.validation import backend, util
from awacs.web.validation.util import is_root
from .util import AwacsBlueprint, forbid_action_during_namespace_order, validate_namespace_total_objects_count


backend_service_bp = AwacsBlueprint('rpc_backend_service', __name__, '/api')

COMMON_NAMESPACE_BACKENDS_TYPES = (model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD, model_pb2.BackendSelector.MANUAL)


@backend_service_bp.method('CreateBackend',
                           request_type=api_pb2.CreateBackendRequest,
                           response_type=api_pb2.CreateBackendResponse)
def create_backend(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateBackendRequest
    :param auth_subject:
    :return:
    """
    backend.validate_request(req_pb)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    namespace_id = req_pb.meta.namespace_id
    backend_id = req_pb.meta.id

    c = cache.IAwacsCache.instance()
    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id=namespace_id)
    validate_namespace_total_objects_count('backend', len(c.list_all_backends(namespace_id)), namespace_pb)
    authorize_create(namespace_pb, auth_subject)

    if zk.does_backend_exist(namespace_id, backend_id):
        raise rpc.exceptions.ConflictError(
            'Backend "{}" already exists in namespace "{}".'.format(backend_id, namespace_id))

    if (namespace_pb.spec.easy_mode_settings.non_sd_backends_creation_disabled.value and
            req_pb.spec.selector.type not in COMMON_NAMESPACE_BACKENDS_TYPES):
        raise ForbiddenError('Only YP_ENDPOINT_SETS_SD and MANUAL backends can be created in this namespace')

    if (namespace_pb.spec.layout_type == namespace_pb.spec.NS_LAYOUT_L3_ONLY and
            req_pb.spec.selector.type == model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD):
        raise ForbiddenError('YP_ENDPOINT_SETS_SD backends can not be created in this namespace')

    if (namespace_pb.spec.layout_type == namespace_pb.spec.NS_LAYOUT_GLOBAL and
            req_pb.spec.selector.type != model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD):
        raise ForbiddenError('Only YP_ENDPOINT_SETS_SD backends can be created in global namespace')

    if not is_root(auth_subject.login):
        if req_pb.meta.is_system.value:
            raise ForbiddenError('"meta.is_system": system backend can only be created by roots')
        if req_pb.spec.is_global.value:
            raise ForbiddenError('"spec.is_global": global backend can only be created by roots')
    if req_pb.meta.is_system.value and req_pb.spec.is_global.value:
        raise ForbiddenError('"meta.is_system,spec.is_global": global backend cannot be system at the same time')

    selector_pb = req_pb.spec.selector
    backend_type = selector_pb.type
    if backend_type == model_pb2.BackendSelector.NANNY_SNAPSHOTS:
        service_ids = [snapshot_pb.service_id for snapshot_pb in selector_pb.nanny_snapshots]
        util.validate_services_policy(service_ids)
    elif (req_pb.validate_yp_endpoint_sets
          and backend_type in (model_pb2.BackendSelector.YP_ENDPOINT_SETS,
                               model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD)):
        util.validate_yp_endpoint_sets(backend_id=backend_id,
                                       yp_endpoint_set_pbs=selector_pb.yp_endpoint_sets)
    backend_pb = IDao.instance().create_backend(
        meta_pb=req_pb.meta,
        spec_pb=req_pb.spec,
        login=auth_subject.login)
    # no need to fill the statuses because the backend has just been created and
    # there is very little chance it has already been processed by ctls
    return api_pb2.CreateBackendResponse(backend=backend_pb)


def annotate_with_endpoint_set(ac, mongo_storage, backend_pb, balancer_state_pbs, balancer_state_pb=None):
    endpoint_set_pb = ac.get_endpoint_set(backend_pb.meta.namespace_id, backend_pb.meta.id)
    if endpoint_set_pb:
        if balancer_state_pb:
            endpoint_set_pb = apicache.copy_statuses(endpoint_set_pb, balancer_state_pb)
        endpoint_set_pb = apicache.copy_per_balancer_statuses(entity_pb=endpoint_set_pb,
                                                              balancer_state_pbs=balancer_state_pbs)
        endpoint_set_pb = apicache.copy_per_balancer_l3_statuses(endpoint_set_pb)
        endpoint_set_pb = apicache.copy_per_balancer_dns_record_statuses(endpoint_set_pb)
    endpoint_set_rev_meta_pbs = []
    annotated_backend_pb = api_pb2.AnnotatedBackend(
        backend=backend_pb,
        endpoint_set=endpoint_set_pb,
        endpoint_set_rev_metas=endpoint_set_rev_meta_pbs
    )
    if endpoint_set_pb:
        for endpoint_set_rev_status_pb in endpoint_set_pb.statuses:
            meta_pb = mongo_storage.must_get_endpoint_set_rev(endpoint_set_rev_status_pb.id).meta
            endpoint_set_rev_meta_pbs.append(meta_pb)
            for backend_version in meta_pb.backend_versions:
                annotated_backend_pb.backend_rev_to_endpoint_set_revs[backend_version].revisions.append(
                    endpoint_set_rev_status_pb.id)

        for endpoint_set_rev_l3_status_pb in endpoint_set_pb.l3_statuses:
            meta_pb = mongo_storage.must_get_endpoint_set_rev(endpoint_set_rev_l3_status_pb.id).meta
            endpoint_set_rev_meta_pbs.append(meta_pb)
            for backend_version in meta_pb.backend_versions:
                annotated_backend_pb.backend_rev_to_endpoint_set_revs[backend_version].revisions.append(
                    endpoint_set_rev_l3_status_pb.id)

        for endpoint_set_rev_dns_record_status_pb in endpoint_set_pb.dns_record_statuses:
            meta_pb = mongo_storage.must_get_endpoint_set_rev(endpoint_set_rev_dns_record_status_pb.id).meta
            endpoint_set_rev_meta_pbs.append(meta_pb)
            for backend_version in meta_pb.backend_versions:
                annotated_backend_pb.backend_rev_to_endpoint_set_revs[backend_version].revisions.append(
                    endpoint_set_rev_dns_record_status_pb.id)
    return annotated_backend_pb


@backend_service_bp.method('ListBackends',
                           request_type=api_pb2.ListBackendsRequest,
                           response_type=api_pb2.ListBackendsResponse,
                           max_in_flight=5)
def list_backends(req_pb, _):
    backend.validate_request(req_pb)
    if not req_pb.HasField('field_mask'):
        req_pb.field_mask.AllFieldsFromDescriptor(model_pb2.Backend.DESCRIPTOR)

    c = cache.IAwacsCache.instance()
    ac = apicache.IAwacsApiCache.instance()

    query = {}
    if req_pb.query.id_regexp:
        query[ac.BackendsQueryTarget.ID_REGEXP] = re.compile(req_pb.query.id_regexp)
    if req_pb.query.validated_status_in:
        query[ac.BackendsQueryTarget.VALIDATED_STATUS_IN] = req_pb.query.validated_status_in
    if req_pb.query.in_progress_status_in:
        query[ac.BackendsQueryTarget.IN_PROGRESS_STATUS_IN] = req_pb.query.in_progress_status_in
    if req_pb.query.active_status_in:
        query[ac.BackendsQueryTarget.ACTIVE_STATUS_IN] = req_pb.query.active_status_in
    if req_pb.query.nanny_service_id_in:
        query[ac.BackendsQueryTarget.NANNY_SERVICE_ID_IN] = req_pb.query.nanny_service_id_in
    if req_pb.query.gencfg_group_name_in:
        query[ac.BackendsQueryTarget.GENCFG_GROUP_NAME_IN] = req_pb.query.gencfg_group_name_in
    if req_pb.query.yp_endpoint_set_full_id_in:
        query[ac.BackendsQueryTarget.YP_ENDPOINT_SET_FULL_ID_IN] = req_pb.query.yp_endpoint_set_full_id_in
        for es_pb in query[ac.BackendsQueryTarget.YP_ENDPOINT_SET_FULL_ID_IN]:
            es_pb.cluster = es_pb.cluster.lower()
    query[ac.BackendsQueryTarget.ONLY_SYSTEM] = req_pb.query.only_system
    query[ac.BackendsQueryTarget.EXCLUDE_SYSTEM] = req_pb.query.exclude_system

    sort = []
    if req_pb.sort_target == req_pb.ID:
        sort.append(ac.BackendsSortTarget.ID)
    elif req_pb.sort_target == req_pb.MTIME:
        sort.append(ac.BackendsSortTarget.MTIME)
    sort.append(-1 if req_pb.sort_order == api_pb2.DESCEND else 1)

    kwargs = dict(sort=sort,
                  query=query,
                  skip=req_pb.skip or None,
                  limit=req_pb.limit or None)

    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.balancer_id

    if balancer_id:
        backend_pbs, total = ac.list_balancer_backends(namespace_id=namespace_id, balancer_id=balancer_id, **kwargs)
    elif namespace_id:
        backend_pbs, total = ac.list_namespace_backends(namespace_id=namespace_id, **kwargs)
    else:
        backend_pbs, total = ac.list_backends(**kwargs)

    if req_pb.annotate_with_endpoint_sets:
        annotated_backend_pbs = []
        mongo_storage = IMongoStorage.instance()
        balancer_state_pb = None
        if balancer_id:
            balancer_state_pb = c.get_balancer_state_or_empty(namespace_id=namespace_id,
                                                              balancer_id=balancer_id)
        all_balancer_state_pbs = c.list_all_balancer_states()
        namespace_balancer_state_pbs = c.list_all_balancer_states(namespace_id=namespace_id)
        for backend_pb in gevent_idle_iter(backend_pbs, idle_period=10):
            if backend_pb.spec.is_global.value:
                balancer_state_pbs = all_balancer_state_pbs
            else:
                balancer_state_pbs = namespace_balancer_state_pbs
            annotated_backend_pbs.append(annotate_with_endpoint_set(ac=ac,
                                                                    mongo_storage=mongo_storage,
                                                                    backend_pb=backend_pb,
                                                                    balancer_state_pb=balancer_state_pb,
                                                                    balancer_state_pbs=balancer_state_pbs))
        return api_pb2.ListBackendsResponse(annotated_backends=annotated_backend_pbs, total=total)
    else:
        resp_pb = api_pb2.ListBackendsResponse(total=total)
        for backend_pb in gevent_idle_iter(backend_pbs, idle_period=30):
            req_pb.field_mask.MergeMessage(backend_pb, resp_pb.backends.add())
        return resp_pb


@backend_service_bp.method('GetBackend',
                           request_type=api_pb2.GetBackendRequest,
                           response_type=api_pb2.GetBackendResponse)
def get_backend(req_pb, _):
    backend.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    backend_id = req_pb.id

    c = cache.IAwacsCache.instance()
    ac = apicache.IAwacsApiCache.instance()

    if req_pb.consistency == api_pb2.STRONG:
        zk = IZkStorage.instance()
        zk.sync_balancer_states(namespace_id)
        backend_pb = zk.must_get_backend(namespace_id, backend_id, sync=True)
        if backend_pb.spec.is_global.value:
            balancer_state_pbs = c.list_all_balancer_states()
        else:
            balancer_state_pbs = c.list_all_balancer_states(namespace_id=namespace_id)
        backend_pb = apicache.copy_per_balancer_statuses(entity_pb=backend_pb, balancer_state_pbs=balancer_state_pbs)
        backend_pb = apicache.copy_per_balancer_l3_statuses(backend_pb)
        backend_pb = apicache.copy_per_balancer_dns_record_statuses(backend_pb)
    else:
        backend_pb = ac.must_get_backend(namespace_id, backend_id)

    if req_pb.annotate_with_endpoint_set:
        mongo_storage = IMongoStorage.instance()
        if backend_pb.spec.is_global.value:
            balancer_state_pbs = c.list_all_balancer_states()
        else:
            balancer_state_pbs = c.list_all_balancer_states(namespace_id=namespace_id)
        backend_pb = annotate_with_endpoint_set(ac=ac,
                                                mongo_storage=mongo_storage,
                                                backend_pb=backend_pb,
                                                balancer_state_pb=None,
                                                balancer_state_pbs=balancer_state_pbs)
        return api_pb2.GetBackendResponse(annotated_backend=backend_pb)
    else:
        return api_pb2.GetBackendResponse(backend=backend_pb)


@backend_service_bp.method('UpdateBackend',
                           request_type=api_pb2.UpdateBackendRequest,
                           response_type=api_pb2.UpdateBackendResponse)
def update_backend(req_pb, auth_subject):
    backend.validate_request(req_pb)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    namespace_id = req_pb.meta.namespace_id
    backend_id = req_pb.meta.id

    zk = IZkStorage.instance()
    c = cache.IAwacsCache.instance()
    dao = IDao.instance()
    backend_pb = zk.must_get_backend(namespace_id, backend_id)
    if backend_pb.meta.is_system.value:
        raise ForbiddenError('System backend cannot be modified')
    namespace_pb = c.must_get_namespace(namespace_id)
    try:
        authorize_update(backend_pb, req_pb, auth_subject)
    except ForbiddenError:
        authorize_update(backend_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    if (namespace_pb.spec.easy_mode_settings.non_sd_backends_creation_disabled.value and
            backend_pb.spec.selector.type in COMMON_NAMESPACE_BACKENDS_TYPES and
            req_pb.spec.selector.type not in COMMON_NAMESPACE_BACKENDS_TYPES):
        raise ForbiddenError('Only YP_ENDPOINT_SETS_SD and MANUAL backends can be created in this namespace')

    if (namespace_pb.spec.layout_type == namespace_pb.spec.NS_LAYOUT_L3_ONLY and
            req_pb.spec.selector.type == model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD):
        raise ForbiddenError('YP_ENDPOINT_SETS_SD backends can not be created in this namespace')

    if (namespace_pb.spec.layout_type == namespace_pb.spec.NS_LAYOUT_GLOBAL and
            req_pb.spec.selector.type != model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD):
        raise ForbiddenError('Only YP_ENDPOINT_SETS_SD backends can be created in global namespace')

    spec_has_changed = req_pb.HasField('spec') and backend_pb.spec != req_pb.spec
    auth_has_changed = req_pb.meta.HasField('auth') and backend_pb.meta.auth != req_pb.meta.auth
    is_global_has_changed = req_pb.spec.HasField('is_global') and backend_pb.spec.is_global != req_pb.spec.is_global
    is_system_has_changed = req_pb.meta.HasField('is_system') and backend_pb.meta.is_system != req_pb.meta.is_system

    if not spec_has_changed and not auth_has_changed and not is_system_has_changed:
        return api_pb2.UpdateBackendResponse(backend=backend_pb)

    is_author_root = auth_subject.login in config.get_value('run.root_users', default=())
    if is_system_has_changed:
        if not is_author_root:
            raise ForbiddenError('"meta.is_system" can only be changed by roots')
        if backend_pb.spec.is_global.value or req_pb.spec.is_global.value:
            raise ForbiddenError('"meta.is_system" cannot be changed for global backend')

    if is_global_has_changed:
        if not is_author_root:
            raise ForbiddenError('"spec.is_global" can only be changed by roots')
        if not req_pb.spec.is_global.value:
            raise ForbiddenError('"spec.is_global" cannot be changed from True to False')
        if backend_pb.meta.is_system.value or req_pb.meta.is_system.value:
            raise ForbiddenError('"spec.is_global" cannot be changed for system backend')

    if spec_has_changed:
        backend.validate_selector_type(namespace_id,
                                       backend_id,
                                       curr_selector_type=backend_pb.spec.selector.type,
                                       updated_selector_type=req_pb.spec.selector.type,
                                       comment=req_pb.meta.comment)

    if (backend_pb.meta.is_system.value or req_pb.meta.is_system.value) and not is_author_root:
        raise ForbiddenError('"meta.is_system": system backend can only be updated by roots')

    changed_to_manual = False
    changed_to_yp_endpoint_sets_sd = False
    if req_pb.HasField('spec'):
        req_selector_pb = req_pb.spec.selector
        req_backend_type = req_selector_pb.type
        changed_to_manual = (
            backend_pb.spec.selector.type != model_pb2.BackendSelector.MANUAL
            and req_backend_type == model_pb2.BackendSelector.MANUAL
        )
        changed_to_yp_endpoint_sets_sd = (
            backend_pb.spec.selector.type != model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD
            and req_backend_type == model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD
        )
        if req_backend_type == model_pb2.BackendSelector.NANNY_SNAPSHOTS:
            new_service_ids = set(snapshot_pb.service_id for snapshot_pb in req_pb.spec.selector.nanny_snapshots)
            old_service_ids = set(snapshot_pb.service_id for snapshot_pb in backend_pb.spec.selector.nanny_snapshots)
            util.validate_services_policy(new_service_ids - old_service_ids)
        elif (req_pb.validate_yp_endpoint_sets
              and req_backend_type in (model_pb2.BackendSelector.YP_ENDPOINT_SETS,
                                       model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD)):
            util.validate_yp_endpoint_sets(backend_id=backend_id,
                                           yp_endpoint_set_pbs=req_selector_pb.yp_endpoint_sets)
    backend_pb = dao.update_backend(
        namespace_id=namespace_id,
        backend_id=backend_id,
        comment=req_pb.meta.comment,
        login=auth_subject.login,
        version=req_pb.meta.version,
        updated_spec_pb=req_pb.spec if spec_has_changed else None,
        updated_auth_pb=req_pb.meta.auth if auth_has_changed else None,
        updated_is_system_pb=req_pb.meta.is_system if is_system_has_changed else None,
        # clear resolver status if backend switched to MANUAL or YP_ENDPOINT_SETS_SD:
        updated_resolver_status_pb=(model_pb2.BackendResolverStatus()
                                    if (changed_to_manual or changed_to_yp_endpoint_sets_sd)
                                    else None)
    )

    backend_pb = apicache.copy_per_balancer_statuses(entity_pb=backend_pb,
                                                     balancer_state_pbs=c.list_all_balancer_states())
    backend_pb = apicache.copy_per_balancer_l3_statuses(backend_pb)
    backend_pb = apicache.copy_per_balancer_dns_record_statuses(backend_pb)
    return api_pb2.UpdateBackendResponse(backend=backend_pb)


@backend_service_bp.method('RemoveBackend',
                           request_type=api_pb2.RemoveBackendRequest,
                           response_type=api_pb2.RemoveBackendResponse,
                           is_destructive=True)
def remove_backend(req_pb, auth_subject):
    backend.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    backend_id = req_pb.id

    zk = IZkStorage.instance()

    backend_pb = zk.must_get_backend(namespace_id, backend_id)
    if backend_pb.meta.is_system.value:
        raise ForbiddenError('system backend cannot be removed')
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_remove(backend_pb, auth_subject)
    except ForbiddenError:
        authorize_remove(backend_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    updated_spec_pb = clone_pb(backend_pb.spec)
    updated_spec_pb.deleted = True

    IDao.instance().update_backend(
        namespace_id=namespace_id,
        backend_id=backend_id,
        version=req_pb.version,
        comment='Marked as deleted by {}'.format(auth_subject.login),
        login=auth_subject.login,
        updated_spec_pb=updated_spec_pb
    )

    return api_pb2.RemoveBackendResponse()


@backend_service_bp.method('GetBackendRevision',
                           request_type=api_pb2.GetBackendRevisionRequest,
                           response_type=api_pb2.GetBackendRevisionResponse)
def get_backend_revision(req_pb, _):
    backend.validate_request(req_pb)
    rev_id = req_pb.id

    mongo_storage = db.IMongoStorage.instance()
    rev_pb = mongo_storage.must_get_backend_rev(rev_id)
    return api_pb2.GetBackendRevisionResponse(revision=rev_pb)


@backend_service_bp.method('ListBackendRevisions',
                           request_type=api_pb2.ListBackendRevisionsRequest,
                           response_type=api_pb2.ListBackendRevisionsResponse,
                           max_in_flight=5)
def list_backend_revisions(req_pb, _):
    backend.validate_request(req_pb)
    backend_id = req_pb.id
    namespace_id = req_pb.namespace_id

    zk = IZkStorage.instance()
    zk.must_get_backend(namespace_id, backend_id)

    mongo_storage = db.IMongoStorage.instance()
    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = mongo_storage.list_backend_revs(namespace_id, backend_id, skip=skip, limit=limit)
    return api_pb2.ListBackendRevisionsResponse(revisions=revs.items, total=revs.total)


@backend_service_bp.method('GetBackendRemovalChecks',
                           request_type=api_pb2.GetBackendRemovalChecksRequest,
                           response_type=api_pb2.GetBackendRemovalChecksResponse)
def get_backend_removal_checks(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetBackendRemovalChecksRequest
    """
    backend.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    backend_id = req_pb.id

    zk = IZkStorage.instance()
    backend_pb = zk.must_get_backend(namespace_id, backend_id)

    resp_pb = api_pb2.GetBackendRemovalChecksResponse()
    resp_pb.checks.extend(backend.get_backend_removal_checks(backend_pb))
    return resp_pb
