# coding: utf-8
import logging

import semantic_version
import six
from datetime import datetime, timedelta
from sepelib.core import config as appconfig

from awacs.lib import rpc
from awacs.lib.gutils import gevent_idle_iter
from awacs.lib.order_processor.model import cancel_order
from awacs.lib.yamlparser.wrappers_util import dump_tlem_pb
from awacs.model import db, cache, apicache
from awacs.model.apicache import copy_statuses
from awacs.model.balancer.state_handler import L7BalancerStateHandler
from awacs.model.balancer.container_spec import update_l3_decap_tunnel_if_needed, ensure_shawshank_layer
from awacs.model.balancer.order.util import make_nanny_service_id
from awacs.model.components import (
    is_pushclient_enabled,
    get_pushclient_supervisor_component,
    find_pushclient_supervisor_component_version,
    find_last_component_version,
    get_component_config,
    is_set,
)
from awacs.model.dao import IDao
from awacs.model.errors import ConflictError
from awacs.model.uiaspectsutil import fill_ui_balancer_aspects
from awacs.model.util import clone_pb, omit_duplicate_items_from_auth, check_condition
from awacs.model.validation import validate_and_parse_yaml_balancer_config, get_yaml
from awacs.model.zk import IZkStorage
from awacs.model.balancer.generator import get_rev_index_pb
from awacs.web.auth.core import authorize_update, authorize_remove, get_acl, authorize_create
from awacs.web.trust_xffy import get_current_xffy_behavior, update_xffy_header_actions, get_xffy_header_index
from awacs.web.util import (
    AwacsBlueprint,
    validate_nanny_service_nonexistence,
    forbid_action_during_namespace_order,
    validate_namespace_total_objects_count,
)
from awacs.web.validation import balancer
from awacs.wrappers.l7macro import (
    VERSION_0_2_8 as L7_MACRO_VERSION_0_2_8,
    VERSION_0_3_0 as L7_MACRO_VERSION_0_3_0,
    VALID_VERSIONS as L7_MACRO_VALID_VERSIONS,
    L7Macro
)
from awacs.wrappers.base import Holder
from infra.awacs.proto import api_pb2, model_pb2

WEBAUTH_CAN_BE_DISABLED_TIME = timedelta(days=1)

log = logging.getLogger(__name__)
balancer_service_bp = AwacsBlueprint('rpc_balancer_service', __name__, '/api')


@balancer_service_bp.method('CreateBalancer',
                            request_type=api_pb2.CreateBalancerRequest,
                            response_type=api_pb2.CreateBalancerResponse)
def create_balancer(req_pb, auth_subject):
    is_author_root = auth_subject.login in appconfig.get_value('run.root_users', default=())
    balancer.validate_request(req_pb)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    balancer_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id

    zk = IZkStorage.instance()

    if zk.does_balancer_exist(namespace_id, balancer_id):
        raise rpc.exceptions.ConflictError(
            'Balancer "{}" already exists in namespace "{}"'.format(balancer_id, namespace_id))

    c = cache.IAwacsCache.instance()
    namespace_pb = zk.must_get_namespace(namespace_id)
    validate_namespace_total_objects_count('balancer', len(c.list_all_balancers(namespace_id)), namespace_pb)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    spec_pb = req_pb.spec if req_pb.HasField('spec') and req_pb.spec != model_pb2.BalancerSpec() else None
    order_pb = req_pb.order if req_pb.HasField('order') and req_pb.order != model_pb2.BalancerOrder.Content() else None

    if namespace_pb.spec.easy_mode_settings.l7_macro_only.value:
        is_l7_macro = True
        if spec_pb is not None:
            is_l7_macro = req_pb.spec.yandex_balancer.mode == req_pb.spec.yandex_balancer.EASY_MODE
        elif order_pb is not None:
            is_l7_macro = True
        if not is_l7_macro:
            raise rpc.exceptions.ForbiddenError('Only EASY_MODE balancers can be created in this namespace')

    if namespace_pb.spec.cloud_type == model_pb2.CT_AZURE and req_pb.meta.location.type != req_pb.meta.location.AZURE_CLUSTER:
        raise rpc.exceptions.BadRequestError('"meta.location.type": must be AZURE_CLUSTER if namespace cloud type '
                                             '== CT_AZURE')
    if namespace_pb.spec.cloud_type != model_pb2.CT_AZURE and req_pb.meta.location.type == req_pb.meta.location.AZURE_CLUSTER:
        raise rpc.exceptions.BadRequestError('"meta.location.type": must not be AZURE_CLUSTER if namespace cloud != '
                                             'CT_AZURE')

    rev_index_pb = None
    authorize_create(namespace_pb, auth_subject)
    if spec_pb:
        if not is_author_root:
            raise rpc.exceptions.BadRequestError('"spec" can not be used, use "order" instead')
        balancer.validate_balancer_service_and_set_location_if_needed(req_pb.meta, spec_pb)
        spec_has_yaml = bool(spec_pb.yandex_balancer.yaml)
        if spec_has_yaml:
            # check yaml size before validating it
            balancer.validate_balancer_yaml_size(spec_pb.yandex_balancer.yaml, field_name=u'spec.yandex_balancer.yaml')
        holder = validate_and_parse_yaml_balancer_config(spec_pb,
                                                         full_balancer_id=(namespace_id, balancer_id),
                                                         namespace_pb=namespace_pb)
        rev_index_pb = get_rev_index_pb(namespace_id, spec_pb, holder)

        if not spec_has_yaml:
            # validate yaml size after calling validate_and_parse_yaml_upstream_config so now it has 'yaml'
            balancer.validate_balancer_yaml_size(spec_pb.yandex_balancer.yaml, field_name=u'spec.yandex_balancer.yaml')
        ensure_shawshank_layer(c, balancer_spec_pb=spec_pb)
        update_l3_decap_tunnel_if_needed(balancer_spec_pb=spec_pb)

    if order_pb:
        prj = order_pb.instance_tags.prj or namespace_pb.spec.balancer_constraints.instance_tags.prj or namespace_id
        if namespace_pb.spec.balancer_constraints.instance_tags.prj:
            if prj != namespace_pb.spec.balancer_constraints.instance_tags.prj:
                raise rpc.exceptions.BadRequestError('"order.instance_tags.prj": must be equal to "{}"'
                                                     .format(namespace_pb.spec.balancer_constraints.instance_tags.prj))
        else:
            balancer.validate_balancer_normalized_prj(namespace_id, prj)
        order_pb.instance_tags.prj = prj

        validate_nanny_service_nonexistence(make_nanny_service_id(req_pb.meta.id, req_pb.meta.location.type))
        order_pb.activate_balancer = True
        if order_pb.copy_nanny_service.balancer_id:
            balancer.validate_nanny_service_for_copy(namespace_id, order_pb.copy_nanny_service.balancer_id)
            balancer.validate_balancer_location_for_copy(namespace_id, order_pb.allocation_request.location)

        if (
                order_pb.env_type == model_pb2.BalancerSpec.L7_ENV_TESTING
                and namespace_pb.spec.env_type == model_pb2.NamespaceSpec.NS_ENV_PRODUCTION
        ):
            raise rpc.exceptions.BadRequestError(
                'Balancer cannot be created in TESTING mode in PRODUCTION namespace'
            )
        elif (
                order_pb.env_type in (
                model_pb2.BalancerSpec.L7_ENV_PRESTABLE,
                model_pb2.BalancerSpec.L7_ENV_PRODUCTION,
        )
                and namespace_pb.spec.env_type == model_pb2.NamespaceSpec.NS_ENV_TESTING
        ):
            raise rpc.exceptions.BadRequestError(
                'PRESTABLE and PRODUCTION mode balancers cannot be created in TESTING namespace'
            )

    balancer_pb, balancer_state_pb = IDao.instance().create_balancer(
        meta_pb=req_pb.meta,
        spec_pb=spec_pb,
        order_content_pb=order_pb,
        login=auth_subject.login,
        rev_index_pb=rev_index_pb)

    apicache.copy_statuses(balancer_pb, balancer_state_pb)
    return api_pb2.CreateBalancerResponse(balancer=balancer_pb)


@balancer_service_bp.method('ListBalancers',
                            request_type=api_pb2.ListBalancersRequest,
                            response_type=api_pb2.ListBalancersResponse,
                            max_in_flight=5)
def list_balancers(req_pb, _):
    balancer.validate_request(req_pb)
    if not req_pb.HasField('field_mask'):
        req_pb.field_mask.AllFieldsFromDescriptor(model_pb2.Balancer.DESCRIPTOR)

    if req_pb.query.yp_endpoint_set_full_id_in:
        return api_pb2.ListBalancersResponse()

    c = apicache.IAwacsApiCache.instance()

    query = {}
    if req_pb.query.l7_macro_version_in:
        query[c.BalancersQueryTarget.L7_MACRO_VERSION_IN] = req_pb.query.l7_macro_version_in
    if req_pb.query.nanny_service_id_in:
        query[c.BalancersQueryTarget.NANNY_SERVICE_ID_IN] = req_pb.query.nanny_service_id_in
    if req_pb.query.component_in:
        query[c.BalancersQueryTarget.COMPONENT_IN] = req_pb.query.component_in

    balancer_pbs, total = c.list_balancers(namespace_id=req_pb.namespace_id or None,
                                           query=query,
                                           skip=req_pb.skip or None,
                                           limit=req_pb.limit or None)
    resp_pb = api_pb2.ListBalancersResponse(total=total)
    for balancer_pb in gevent_idle_iter(balancer_pbs, idle_period=30):
        req_pb.field_mask.MergeMessage(balancer_pb, resp_pb.balancers.add())
    return resp_pb


@balancer_service_bp.method('GetBalancer',
                            request_type=api_pb2.GetBalancerRequest,
                            response_type=api_pb2.GetBalancerResponse)
def get_balancer(req_pb, _):
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    if req_pb.consistency == api_pb2.STRONG:
        zk = IZkStorage.instance()
        balancer_pb = zk.must_get_balancer(namespace_id, balancer_id, sync=True)
        balancer_state_pb = zk.must_get_balancer_state(namespace_id, balancer_id, sync=True)
        balancer_pb = copy_statuses(balancer_pb, balancer_state_pb)
    else:
        c = apicache.IAwacsApiCache.instance()
        balancer_pb = c.must_get_balancer(namespace_id=namespace_id, balancer_id=balancer_id)
    return api_pb2.GetBalancerResponse(balancer=balancer_pb)


@balancer_service_bp.method('GetBalancerAspectsSet',
                            request_type=api_pb2.GetBalancerAspectsSetRequest,
                            response_type=api_pb2.GetBalancerAspectsSetResponse)
def get_balancer_aspects_set(req_pb, _):
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    if req_pb.consistency == api_pb2.STRONG:
        zk = IZkStorage.instance()
        balancer_aspects_set_pb = zk.must_get_balancer_aspects_set(namespace_id, balancer_id, sync=True)
    else:
        c = cache.IAwacsCache.instance()
        balancer_aspects_set_pb = c.must_get_balancer_aspects_set(namespace_id=namespace_id, balancer_id=balancer_id)

    filled_balancer_aspects_set_pb = clone_pb(balancer_aspects_set_pb)
    fill_ui_balancer_aspects(balancer_id, filled_balancer_aspects_set_pb.content)

    return api_pb2.GetBalancerAspectsSetResponse(aspects_set=filled_balancer_aspects_set_pb)


@balancer_service_bp.method('ListBalancerAspectsSets',
                            request_type=api_pb2.ListBalancerAspectsSetsRequest,
                            response_type=api_pb2.ListBalancerAspectsSetsResponse,
                            max_in_flight=5)
def list_balancer_aspects_sets(req_pb, _):
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id

    c = cache.IAwacsCache.instance()
    balancer_aspects_set_pbs = c.list_all_balancer_aspects_sets(namespace_id=namespace_id)

    filled_balancer_aspects_set_pbs = []
    for balancer_aspects_set_pb in balancer_aspects_set_pbs:
        balancer_id = balancer_aspects_set_pb.meta.balancer_id
        filled_balancer_aspects_set_pb = clone_pb(balancer_aspects_set_pb)
        fill_ui_balancer_aspects(balancer_id, filled_balancer_aspects_set_pb.content)
        filled_balancer_aspects_set_pbs.append(filled_balancer_aspects_set_pb)

    return api_pb2.ListBalancerAspectsSetsResponse(aspects_sets=filled_balancer_aspects_set_pbs)


@balancer_service_bp.method('GetBalancerState',
                            request_type=api_pb2.GetBalancerStateRequest,
                            response_type=api_pb2.GetBalancerStateResponse)
def get_balancer_state(req_pb, _):
    balancer.validate_request(req_pb)
    balancer_id = req_pb.id
    namespace_id = req_pb.namespace_id

    zk = IZkStorage.instance()
    balancer_state_pb = zk.must_get_balancer_state(namespace_id, balancer_id)
    return api_pb2.GetBalancerStateResponse(state=balancer_state_pb)


@balancer_service_bp.method('UpdateBalancer',
                            request_type=api_pb2.UpdateBalancerRequest,
                            response_type=api_pb2.UpdateBalancerResponse)
def update_balancer(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateBalancerRequest
    """
    is_author_root = auth_subject.login in appconfig.get_value('run.root_users', default=())
    balancer.validate_request(req_pb)
    zk = IZkStorage.instance()
    c = cache.IAwacsCache.instance()

    op_pb = zk.get_balancer_operation(req_pb.meta.namespace_id, req_pb.meta.id)
    if op_pb:
        raise rpc.exceptions.BadRequestError(u'Cannot change balancer config while balancer operation is in progress')

    omit_duplicate_items_from_auth(req_pb.meta.auth)
    balancer_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id

    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_update(balancer_pb, req_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_update(balancer_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    flags_has_changed = req_pb.meta.HasField('flags') and req_pb.meta.flags != balancer_pb.meta.flags
    if flags_has_changed and not is_author_root:
        raise rpc.exceptions.BadRequestError('"meta.flags" can only be changed by someone with root privileges.')

    rev_index_pbs = None
    updated_spec_pb = req_pb.spec
    if not req_pb.HasField('spec'):
        spec_has_changed = False
    else:
        updated_spec_has_yaml = bool(updated_spec_pb.yandex_balancer.yaml)
        if updated_spec_has_yaml:
            # check yaml size before validating it
            balancer.validate_balancer_yaml_size(updated_spec_pb.yandex_balancer.yaml,
                                                 field_name=u'spec.yandex_balancer.yaml')
        holder = validate_and_parse_yaml_balancer_config(updated_spec_pb,
                                                         full_balancer_id=(namespace_id, balancer_id),
                                                         namespace_pb=namespace_pb)

        updated_spec_pb = clone_pb(updated_spec_pb)  # don't modify the original req_pb. ideally we'd have CoW here
        balancer.validate_changed_components(namespace_pb, balancer_pb, updated_spec_pb, is_author_root)
        ensure_shawshank_layer(c, updated_spec_pb)
        update_l3_decap_tunnel_if_needed(updated_spec_pb)

        spec_has_changed = balancer_pb.spec != updated_spec_pb
        if spec_has_changed:
            if (balancer_pb.spec.yandex_balancer.mode != updated_spec_pb.yandex_balancer.mode and
                    namespace_pb.spec.easy_mode_settings.l7_macro_only.value and
                    updated_spec_pb.yandex_balancer.mode != updated_spec_pb.yandex_balancer.EASY_MODE):
                raise rpc.exceptions.ForbiddenError('Only EASY_MODE balancers are allowed in this namespace')
            balancer.validate_spec_invariants(req_pb, balancer_pb, updated_spec_pb, is_author_root)
            if not updated_spec_has_yaml:
                # validate yaml size after calling validate_and_parse_yaml_upstream_config so now it has 'yaml'
                balancer.validate_balancer_yaml_size(updated_spec_pb.yandex_balancer.yaml,
                                                     field_name=u'spec.yandex_balancer.yaml')

            webauth_was_enabled = balancer_pb.spec.yandex_balancer.config.l7_macro.HasField('webauth')
            webauth_enabled = updated_spec_pb.yandex_balancer.config.l7_macro.HasField('webauth')

            now = datetime.utcnow()
            if not webauth_was_enabled and webauth_enabled and not balancer_pb.meta.flags.forbid_disabling_webauth.value:
                req_pb.meta.flags.forbid_disabling_webauth.value = True
                req_pb.meta.flags.forbid_disabling_webauth.not_before.FromDatetime(now + WEBAUTH_CAN_BE_DISABLED_TIME)

            if (webauth_was_enabled and not webauth_enabled and not is_author_root and
                    check_condition(balancer_pb.meta.flags.forbid_disabling_webauth, now)):
                raise rpc.exceptions.BadRequestError('"webauth": can only be disabled by roots. Please contact '
                                                     'support if you absolutely must do it')

            if (webauth_enabled and
                    (updated_spec_pb.yandex_balancer.config.l7_macro.webauth.action ==
                     updated_spec_pb.yandex_balancer.config.l7_macro.webauth.AUTHENTICATE_USING_IDM) and
                    not namespace_pb.meta.webauth.responsible.logins
            ):
                raise rpc.exceptions.BadRequestError('"webauth.action": can not be AUTHENTICATE_USING_IDM if '
                                                     'meta.webauth.responsible is empty in namespace')
            if (updated_spec_pb.yandex_balancer.mode == updated_spec_pb.yandex_balancer.EASY_MODE and
                    updated_spec_pb.yandex_balancer.config.l7_macro.HasField('webauth') and
                    updated_spec_pb.yandex_balancer.config.l7_macro.version and
                    semantic_version.Version(
                        updated_spec_pb.yandex_balancer.config.l7_macro.version) >= L7_MACRO_VERSION_0_2_8
            ):
                MIN_PGINX_VERSION = '220-1'
                pginx_config = get_component_config(model_pb2.ComponentMeta.PGINX_BINARY)
                parsed_version = pginx_config.parse_version(updated_spec_pb.components.pginx_binary.version)
                if parsed_version < pginx_config.parse_version(MIN_PGINX_VERSION):
                    raise rpc.exceptions.BadRequestError('"spec.components.pginx_binary.version": must be >= {} '
                                                         'if l7_macro.version >= {} and l7_macro.webauth is set'.
                                                         format(MIN_PGINX_VERSION, L7_MACRO_VERSION_0_2_8))

            if (not is_author_root and updated_spec_pb.custom_service_settings.service !=
                    balancer_pb.spec.custom_service_settings.service):
                raise rpc.exceptions.BadRequestError('"spec.custom_service_settings.service": can not be changed.'
                                                     'Please contact support if you really need')

            if updated_spec_pb.env_type != balancer_pb.spec.env_type:
                if (
                        balancer_pb.spec.env_type != model_pb2.BalancerSpec.L7_ENV_UNKNOWN
                        or updated_spec_pb.env_type == model_pb2.BalancerSpec.L7_ENV_UNKNOWN
                ):
                    if not is_author_root:
                        raise rpc.exceptions.BadRequestError('"spec.env_type" can only be changed by root')
                elif updated_spec_pb.env_type == model_pb2.BalancerSpec.L7_ENV_TESTING:
                    if namespace_pb.spec.env_type == model_pb2.NamespaceSpec.NS_ENV_PRODUCTION:
                        # TODO also ensure RPS limit for testing
                        raise rpc.exceptions.BadRequestError(
                            '"spec.env_type" cannot be set to TESTING when namespace is not TESTING')
                elif namespace_pb.spec.env_type == model_pb2.NamespaceSpec.NS_ENV_TESTING:
                    # TODO ensure RPS for prestable
                    raise rpc.exceptions.BadRequestError(
                        '"spec.env_type" can be set only to TESTING if namespace is in TESTING')

            rev_index_pbs = list(balancer_pb.meta.indices)
            rev_index_pbs.append(get_rev_index_pb(namespace_id, balancer_pb.spec, holder))

    location_has_changed = req_pb.meta.HasField('location') and req_pb.meta.location != balancer_pb.meta.location
    if balancer_pb.meta.location.yp_cluster and location_has_changed:
        raise rpc.exceptions.BadRequestError('"meta.location.yp_cluster" can not be changed')
    if balancer_pb.meta.location.gencfg_dc and location_has_changed and not is_author_root:
        raise rpc.exceptions.BadRequestError('"meta.location.gencfg_dc" can only be changed by '
                                             'someone with root privileges. Please contact support if you really need')

    flags_has_changed = req_pb.meta.HasField('flags') and req_pb.meta.flags != balancer_pb.meta.flags

    transport_paused_has_changed = (req_pb.meta.HasField('transport_paused') and
                                    balancer_pb.meta.transport_paused.value != req_pb.meta.transport_paused.value)
    if transport_paused_has_changed:
        forbid_action_during_namespace_order(namespace_pb, auth_subject)

    auth_has_changed = req_pb.meta.HasField('auth') and balancer_pb.meta.auth != req_pb.meta.auth

    if not any((spec_has_changed,
                auth_has_changed,
                transport_paused_has_changed,
                flags_has_changed,
                location_has_changed)):
        return api_pb2.UpdateBalancerResponse(balancer=balancer_pb)

    balancer_pb = IDao.instance().update_balancer(
        namespace_id=namespace_id,
        balancer_id=balancer_id,
        version=req_pb.meta.version,
        comment=req_pb.meta.comment,
        login=auth_subject.login,
        updated_spec_pb=updated_spec_pb if spec_has_changed else None,
        updated_auth_pb=req_pb.meta.auth if auth_has_changed else None,
        updated_transport_paused_pb=req_pb.meta.transport_paused if transport_paused_has_changed else None,
        updated_location_pb=req_pb.meta.location if location_has_changed else None,
        updated_flags_pb=req_pb.meta.flags if flags_has_changed else None,
        rev_index_pbs=rev_index_pbs,
    )
    state_pb = cache.IAwacsCache.instance().get_balancer_state_or_empty(
        namespace_id=namespace_id, balancer_id=balancer_id)
    apicache.copy_statuses(balancer_pb, state_pb)

    return api_pb2.UpdateBalancerResponse(balancer=balancer_pb)


@balancer_service_bp.method('CancelBalancerOrder',
                            request_type=api_pb2.CancelBalancerOrderRequest,
                            response_type=api_pb2.CancelBalancerOrderResponse)
def cancel_balancer_order(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelBalancerOrderRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()

    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_remove(balancer_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_remove(balancer_pb, auth_subject, acl=get_acl(namespace_pb))
    if req_pb.force:
        if auth_subject.login not in appconfig.get_value('run.root_users', ()):
            raise rpc.exceptions.ForbiddenError('Only root users can force cancel balancer order')
    else:
        forbid_action_during_namespace_order(namespace_pb, auth_subject)
        balancer.validate_cancel_balancer_order(balancer_pb)

    for balancer_pb in zk.update_balancer(namespace_id=namespace_id, balancer_id=balancer_id, balancer_pb=balancer_pb):
        cancel_order(balancer_pb, author=auth_subject.login, comment='Cancelled in UI')
    return api_pb2.CancelBalancerOrderResponse()


@balancer_service_bp.method('ChangeBalancerOrder',
                            request_type=api_pb2.ChangeBalancerOrderRequest,
                            response_type=api_pb2.ChangeBalancerOrderResponse)
def change_balancer_order(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ChangeBalancerOrderRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()

    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)
    try:
        authorize_remove(balancer_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        namespace_pb = zk.must_get_namespace(namespace_id)
        authorize_remove(balancer_pb, auth_subject, acl=get_acl(namespace_pb))

    if req_pb.HasField('change_abc_service_id'):
        balancer.validate_change_balancer_order_abc_service_id(req_pb, balancer_pb)
        for balancer_pb in zk.update_balancer(namespace_id, balancer_id, balancer_pb):
            balancer_pb.order.content.abc_service_id = req_pb.change_abc_service_id.abc_service_id
            balancer_pb.order.progress.state.attempts = 0
        return api_pb2.ChangeBalancerOrderResponse()
    else:
        raise rpc.exceptions.BadRequestError('Supported order change operations: "change_abc_service_id"')


@balancer_service_bp.method('RemoveBalancer',
                            request_type=api_pb2.RemoveBalancerRequest,
                            response_type=api_pb2.RemoveBalancerResponse,
                            is_destructive=True)
def remove_balancer(req_pb, auth_subject):
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()
    namespace_pb = zk.must_get_namespace(namespace_id)
    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)

    try:
        authorize_remove(balancer_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_remove(balancer_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    for check in balancer.get_balancer_removal_checks(balancer_pb):
        if check.state != check.PASSED:
            raise rpc.exceptions.ForbiddenError(check.message)

    for balancer_pb in zk.update_balancer(namespace_id, balancer_id):
        balancer_pb.spec.deleted = True
        balancer_pb.ClearField('removal')
        balancer_pb.removal.content.mode = req_pb.mode

    return api_pb2.RemoveBalancerResponse()


@balancer_service_bp.method('CancelBalancerRemoval',
                            request_type=api_pb2.CancelBalancerRemovalRequest,
                            response_type=api_pb2.CancelBalancerRemovalResponse)
def cancel_balancer_removal(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelBalancerRemovalRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()

    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_remove(balancer_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_remove(balancer_pb, auth_subject, acl=get_acl(namespace_pb))
    if req_pb.force:
        if auth_subject.login not in appconfig.get_value('run.root_users', ()):
            raise rpc.exceptions.ForbiddenError('Only root users can force cancel balancer order')
    else:
        forbid_action_during_namespace_order(namespace_pb, auth_subject)
        balancer.validate_cancel_balancer_removal(balancer_pb)

    for balancer_pb in zk.update_balancer(namespace_id=namespace_id, balancer_id=balancer_id, balancer_pb=balancer_pb):
        cancel_order(balancer_pb, author=auth_subject.login, comment='Cancelled in UI', field='removal')
    return api_pb2.CancelBalancerRemovalResponse()


@balancer_service_bp.method('GetBalancerRevision',
                            request_type=api_pb2.GetBalancerRevisionRequest,
                            response_type=api_pb2.GetBalancerRevisionResponse)
def get_balancer_revision(req_pb, _):
    balancer.validate_request(req_pb)
    rev_id = req_pb.id

    rev_pb = db.IMongoStorage.instance().must_get_balancer_rev(rev_id)
    return api_pb2.GetBalancerRevisionResponse(revision=rev_pb)


@balancer_service_bp.method('ListBalancerRevisions',
                            request_type=api_pb2.ListBalancerRevisionsRequest,
                            response_type=api_pb2.ListBalancerRevisionsResponse,
                            max_in_flight=5)
def list_balancer_revisions(req_pb, _):
    balancer.validate_request(req_pb)
    balancer_id = req_pb.id
    namespace_id = req_pb.namespace_id

    zk = IZkStorage.instance()
    zk.must_get_balancer(namespace_id, balancer_id)

    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = db.IMongoStorage.instance().list_balancer_revs(
        namespace_id=namespace_id, balancer_id=balancer_id, skip=skip, limit=limit)
    return api_pb2.ListBalancerRevisionsResponse(revisions=revs.items, total=revs.total)


@balancer_service_bp.method('CreateBalancerOperation',
                            request_type=api_pb2.CreateBalancerOperationRequest,
                            response_type=api_pb2.CreateBalancerOperationResponse)
def create_balancer_operation(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateBalancerOperationRequest
    """
    balancer.validate_request(req_pb)

    balancer_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id

    c = cache.IAwacsCache.instance()

    namespace_pb = c.must_get_namespace(namespace_id=namespace_id)
    authorize_create(namespace_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    if c.does_balancer_operation_exist(namespace_id, balancer_id):
        raise rpc.exceptions.ConflictError(
            'Balancer operation for "{}" is already in progress in namespace "{}"'.format(balancer_id, namespace_id))

    try:
        pb = IDao.instance().create_balancer_operation(
            meta_pb=req_pb.meta,
            order_pb=req_pb.order,
            login=auth_subject.login)
    except ConflictError:
        raise rpc.exceptions.ConflictError('Balancer operation for "{}" is already in progress in namespace "{}"'
                                           .format(balancer_id, namespace_id))

    return api_pb2.CreateBalancerOperationResponse(operation=pb)


@balancer_service_bp.method('ListBalancerOperations',
                            request_type=api_pb2.ListBalancerOperationsRequest,
                            response_type=api_pb2.ListBalancerOperationsResponse,
                            max_in_flight=5)
def list_balancer_operations(req_pb, _):
    """
    :type req_pb: api_pb2.ListBalancerOperationsRequest
    """
    balancer.validate_request(req_pb)
    balancer_op_pbs, total = apicache.IAwacsApiCache.instance().list_namespace_balancer_operations(
        namespace_id=req_pb.namespace_id,
        skip=req_pb.skip or None,
        limit=req_pb.limit or None)
    return api_pb2.ListBalancerOperationsResponse(total=total, operations=balancer_op_pbs)


@balancer_service_bp.method('GetBalancerOperation',
                            request_type=api_pb2.GetBalancerOperationRequest,
                            response_type=api_pb2.GetBalancerOperationResponse)
def get_balancer_operation(req_pb, _):
    """
    :type req_pb: api_pb2.GetBalancerOperationRequest
    """
    balancer.validate_request(req_pb)
    pb = apicache.IAwacsApiCache.instance().must_get_balancer_operation(req_pb.namespace_id, req_pb.id)
    return api_pb2.GetBalancerOperationResponse(operation=pb)


@balancer_service_bp.method('CancelBalancerOperation',
                            request_type=api_pb2.CancelBalancerOperationRequest,
                            response_type=api_pb2.CancelBalancerOperationResponse)
def cancel_balancer_operation(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelBalancerOperationRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    c = cache.IAwacsCache.instance()

    namespace_pb = c.must_get_namespace(namespace_id)
    authorize_remove(namespace_pb, auth_subject)
    balancer_op_pb = c.must_get_balancer_operation(namespace_id, balancer_id)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    if req_pb.force:
        if auth_subject.login not in appconfig.get_value('run.root_users', ()):
            raise rpc.exceptions.ForbiddenError('Only root users can force cancel balancer operation')
    else:
        balancer.validate_cancel_balancer_operation(balancer_op_pb)

    for balancer_op_pb in IZkStorage.instance().update_balancer_operation(namespace_id=namespace_id,
                                                                          balancer_id=balancer_id,
                                                                          balancer_operation_pb=balancer_op_pb):
        cancel_order(balancer_op_pb, author=auth_subject.login, comment='Cancelled via API')
    return api_pb2.CancelBalancerOperationResponse()


@balancer_service_bp.method('UpdateBalancerOperationOrderContext',
                            request_type=api_pb2.UpdateBalancerOperationOrderContextRequest,
                            response_type=api_pb2.UpdateBalancerOperationOrderContextResponse)
def update_balancer_operation_order_context(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateBalancerOperationOrderContextRequest
    """
    is_author_root = auth_subject.login in appconfig.get_value('run.root_users', default=())
    if not is_author_root:
        raise rpc.exceptions.BadRequestError('Updating balancer operation context is available only for root_users')

    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()
    c = cache.IAwacsCache.instance()

    namespace_pb = c.must_get_namespace(namespace_id)
    authorize_remove(namespace_pb, auth_subject)
    balancer_op_pb = c.must_get_balancer_operation(namespace_id, balancer_id)

    for balancer_op_pb in zk.update_balancer_operation(namespace_id=namespace_id,
                                                       balancer_id=balancer_id,
                                                       balancer_operation_pb=balancer_op_pb):
        order_pb = balancer_op_pb.order
        for field_pb in req_pb.fields:
            if field_pb.value:
                order_pb.progress.context[field_pb.key] = field_pb.value
            elif field_pb.key in order_pb.progress.context:
                del order_pb.progress.context[field_pb.key]
    return api_pb2.UpdateBalancerOperationOrderContextResponse()


@balancer_service_bp.method('ChangeBalancerOperationOrder',
                            request_type=api_pb2.ChangeBalancerOperationOrderRequest,
                            response_type=api_pb2.ChangeBalancerOperationOrderResponse)
def change_balancer_operation_order(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ChangeBalancerOperationOrderRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_remove(namespace_pb, auth_subject)
    balancer_op_pb = zk.must_get_balancer_operation(namespace_id, balancer_id)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    if req_pb.HasField('change_instance_tags'):
        if not balancer_op_pb.order.content.HasField('migrate_from_gencfg_to_yp_lite'):
            raise rpc.exceptions.BadRequestError('Unsupported change for this balancer operation type')
        balancer.validate_change_balancer_operation_order_instance_tags(balancer_op_pb)
        for balancer_op_pb in zk.update_balancer_operation(namespace_id=namespace_id,
                                                           balancer_id=balancer_id,
                                                           balancer_operation_pb=balancer_op_pb):
            balancer_op_pb.order.content.migrate_from_gencfg_to_yp_lite.fallback.ctype = req_pb.change_instance_tags.ctype
            balancer_op_pb.order.content.migrate_from_gencfg_to_yp_lite.fallback.prj = req_pb.change_instance_tags.prj
            balancer_op_pb.order.progress.state.attempts = 0
        return api_pb2.ChangeBalancerOperationOrderResponse()
    else:
        raise rpc.exceptions.BadRequestError('Supported order change operations: "change_instance_tags"')


@balancer_service_bp.method('ChangeBalancerRemoval',
                            request_type=api_pb2.ChangeBalancerRemovalRequest,
                            response_type=api_pb2.ChangeBalancerRemovalResponse,
                            is_destructive=True)
def change_balancer_removal(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ChangeBalancerRemovalRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_remove(namespace_pb, auth_subject)
    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)
    if not balancer_pb.spec.deleted:
        raise rpc.exceptions.BadRequestError('Can be used only for deleted balancers')

    if req_pb.HasField('approve_after_service_shutdown'):
        for balancer_pb in zk.update_balancer(namespace_id=namespace_id,
                                              balancer_id=balancer_id):
            balancer_pb.removal.approval.after_service_shutdown.value = True
            balancer_pb.removal.approval.after_service_shutdown.author = auth_subject.login
            balancer_pb.removal.approval.after_service_shutdown.mtime.GetCurrentTime()
        return api_pb2.ChangeBalancerRemovalResponse()
    else:
        raise rpc.exceptions.BadRequestError('Supported removal change operations: "approve_after_service_shutdown"')


@balancer_service_bp.method('UpdateBalancerTransportPaused',
                            request_type=api_pb2.UpdateBalancerTransportPausedRequest,
                            response_type=api_pb2.UpdateBalancerTransportPausedResponse)
def update_balancer_transport_paused(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateBalancerTransportPausedRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_update(namespace_pb, req_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    b_pb = IDao.instance().update_balancer(namespace_id, balancer_id,
                                           version=None,
                                           comment=req_pb.transport_paused.comment,
                                           login=auth_subject.login,
                                           updated_transport_paused_pb=req_pb.transport_paused)
    return api_pb2.UpdateBalancerTransportPausedResponse(transport_paused=b_pb.meta.transport_paused)


@balancer_service_bp.method('EnableBalancerPushclient',
                            request_type=api_pb2.EnableBalancerPushclientRequest,
                            response_type=api_pb2.EnableBalancerPushclientResponse)
def enable_balancer_pushclient(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.EnableBalancerPushclientRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()
    c = cache.IAwacsCache.instance()

    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_update(namespace_pb, req_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)
    supervisor_component_type, supervisor_component_pb = get_pushclient_supervisor_component(
        balancer_pb.spec.components)
    if not supervisor_component_type and not supervisor_component_pb:
        raise rpc.exceptions.BadRequestError('One of INSTANCECTL_CONF and AWACSLET must be set to enable pushclient')

    pushclient_enabled_for_supervisor = is_pushclient_enabled(supervisor_component_type,
                                                              supervisor_component_pb.version)
    pushclient_pb = balancer_pb.spec.components.pushclient
    pushclient_binary_set = is_set(pushclient_pb)
    if pushclient_enabled_for_supervisor and pushclient_binary_set:
        raise rpc.exceptions.BadRequestError('Pushclient is already enabled')

    if not pushclient_enabled_for_supervisor:
        new_version = find_pushclient_supervisor_component_version(c, supervisor_component_type,
                                                                   supervisor_component_pb.version,
                                                                   with_pushclient=True)
        if new_version is None:
            log.exception('No components with enabled pushclient found for balancer {}:{}'
                          .format(req_pb.namespace_id, req_pb.id))
            raise rpc.exceptions.InternalError('No components with enabled pushclient found')
        supervisor_component_pb.version = new_version

    if not pushclient_binary_set:
        binary_version = find_last_component_version(c, model_pb2.ComponentMeta.PUSHCLIENT)
        if binary_version is None:
            log.exception('No pushclient components found for balancer {}:{}'.format(req_pb.namespace_id, req_pb.id))
            raise rpc.exceptions.InternalError('No pushclient components found')
        pushclient_pb.state = pushclient_pb.SET
        pushclient_pb.version = binary_version

    rev_index_pbs = list(balancer_pb.meta.indices)
    rev_index_pbs.append(get_rev_index_pb(namespace_id, balancer_pb.spec))

    IDao.instance().update_balancer(namespace_id, balancer_id,
                                    version=req_pb.version,
                                    comment=req_pb.comment,
                                    login=auth_subject.login,
                                    updated_spec_pb=balancer_pb.spec,
                                    rev_index_pbs=rev_index_pbs)
    return api_pb2.EnableBalancerPushclientResponse()


@balancer_service_bp.method('DisableBalancerPushclient',
                            request_type=api_pb2.DisableBalancerPushclientRequest,
                            response_type=api_pb2.DisableBalancerPushclientResponse)
def disable_balancer_pushclient(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.DisableBalancerPushclientRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()
    c = cache.IAwacsCache.instance()

    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_update(namespace_pb, req_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)
    supervisor_component_type, supervisor_component_pb = get_pushclient_supervisor_component(
        balancer_pb.spec.components)
    if not supervisor_component_type and not supervisor_component_pb:
        raise rpc.exceptions.BadRequestError('One of INSTANCECTL_CONF and AWACSLET must be set to disable pushclient')

    pushclient_enabled_for_supervisor = is_pushclient_enabled(supervisor_component_type,
                                                              supervisor_component_pb.version)
    pushclient_pb = balancer_pb.spec.components.pushclient
    pushclient_binary_set = is_set(pushclient_pb)
    if not pushclient_enabled_for_supervisor and not pushclient_binary_set:
        raise rpc.exceptions.BadRequestError('Pushclient is already disabled')

    if pushclient_enabled_for_supervisor:
        new_version = find_pushclient_supervisor_component_version(c, supervisor_component_type,
                                                                   supervisor_component_pb.version,
                                                                   with_pushclient=False)
        if new_version is None:
            log.exception('No components with disabled pushclient found')
            raise rpc.exceptions.InternalError('No components with disabled pushclient found')
        supervisor_component_pb.version = new_version
    if pushclient_binary_set:
        pushclient_pb.state = pushclient_pb.REMOVED
        pushclient_pb.ClearField('version')

    rev_index_pbs = list(balancer_pb.meta.indices)
    rev_index_pbs.append(get_rev_index_pb(namespace_id, balancer_pb.spec))

    IDao.instance().update_balancer(namespace_id, balancer_id,
                                    version=req_pb.version,
                                    comment=req_pb.comment,
                                    login=auth_subject.login,
                                    updated_spec_pb=balancer_pb.spec,
                                    rev_index_pbs=rev_index_pbs)
    return api_pb2.DisableBalancerPushclientResponse()


@balancer_service_bp.method(u'GetBalancerXFFYBehavior',
                            request_type=api_pb2.GetBalancerXFFYBehaviorRequest,
                            response_type=api_pb2.GetBalancerXFFYBehaviorResponse)
def get_balancer_xffy_behavior(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetBalancerXFFYBehaviorRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()
    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)
    if balancer_pb.spec.yandex_balancer.mode != balancer_pb.spec.yandex_balancer.EASY_MODE:
        raise rpc.exceptions.BadRequestError(u'Balancer is not in EASY_MODE')

    l7_macro_pb = balancer_pb.spec.yandex_balancer.config.l7_macro
    l7_macro = L7Macro(l7_macro_pb)
    xffy_header_index = get_xffy_header_index(l7_macro)
    return api_pb2.GetBalancerXFFYBehaviorResponse(behavior=get_current_xffy_behavior(l7_macro, xffy_header_index),
                                                   xffy_header_index=xffy_header_index,
                                                   upgraded_to_03x=l7_macro.get_version() >= L7_MACRO_VERSION_0_3_0)


@balancer_service_bp.method(u'UpdateBalancerL7MacroTo03x',
                            request_type=api_pb2.UpdateBalancerL7MacroTo03xRequest,
                            response_type=api_pb2.UpdateBalancerL7MacroTo03xResponse)
def update_l7macro_to_03x(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateBalancerL7MacroTo03xRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()
    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_update(namespace_pb, req_pb, auth_subject, implicit_spec_update=True)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)
    if balancer_pb.spec.yandex_balancer.mode != balancer_pb.spec.yandex_balancer.EASY_MODE:
        raise rpc.exceptions.BadRequestError(u'Balancer config must use "l7_macro" to upgrade it')

    l7_macro_pb = balancer_pb.spec.yandex_balancer.config.l7_macro
    if semantic_version.Version(l7_macro_pb.version) >= L7_MACRO_VERSION_0_3_0:
        raise rpc.exceptions.BadRequestError(u'l7_macro is already upgraded (version "{}")'.format(l7_macro_pb.version))

    update_xffy_header_actions(l7_macro_pb, req_pb.trust_x_forwarded_for_y)

    latest_3x_version = max(x for x in L7_MACRO_VALID_VERSIONS if x < semantic_version.Version(u'0.4.0'))
    l7_macro_pb.version = six.text_type(latest_3x_version)
    l7_macro_pb.core.trust_x_forwarded_for_y = req_pb.trust_x_forwarded_for_y
    if l7_macro_pb.announce_check_reply.use_upstream_handler:
        # active_check_reply module that is used for graceful shutdown
        # can not proxy requests to upstream:
        l7_macro_pb.announce_check_reply.compat.disable_graceful_shutdown = True

    yaml = get_yaml(balancer_pb.spec.yandex_balancer.config, dump_fn=dump_tlem_pb)
    balancer_pb.spec.yandex_balancer.yaml = yaml

    rev_index_pbs = list(balancer_pb.meta.indices)
    rev_index_pbs.append(get_rev_index_pb(namespace_id, balancer_pb.spec))

    updated_balancer_pb = IDao.instance().update_balancer(
        namespace_id, balancer_id,
        version=req_pb.version,
        comment=req_pb.comment,
        login=auth_subject.login,
        updated_spec_pb=balancer_pb.spec,
        rev_index_pbs=rev_index_pbs)
    return api_pb2.UpdateBalancerL7MacroTo03xResponse(balancer=updated_balancer_pb)


@balancer_service_bp.method('GetBalancerRemovalChecks',
                            request_type=api_pb2.GetBalancerRemovalChecksRequest,
                            response_type=api_pb2.GetBalancerRemovalChecksResponse)
def get_balancer_removal_checks(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetBalancerRemovalChecksRequest
    """
    balancer.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()
    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)

    resp_pb = api_pb2.GetBalancerRemovalChecksResponse()
    resp_pb.checks.extend(balancer.get_balancer_removal_checks(balancer_pb))
    return resp_pb


# AWACS-1326: for first initialization only
@balancer_service_bp.method('UpdateBalancerIndices',
                            request_type=api_pb2.UpdateBalancerIndicesRequest,
                            response_type=api_pb2.UpdateBalancerIndicesResponse)
def update_balancer_indices(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateBalancerIndicesRequest
    :rtype api_pb2.UpdateBalancerIndicesResponse
    """
    if auth_subject.login not in appconfig.get_value('run.root_users', default=()):
        raise rpc.exceptions.ForbiddenError('Method is only allowed for roots.')

    if not req_pb.id:
        raise rpc.exceptions.BadRequestError('No "id" specified.')
    if not req_pb.namespace_id:
        raise rpc.exceptions.BadRequestError('No "namespace_id" specified.')

    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.id

    zk = IZkStorage.instance()
    mongo = db.IMongoStorage.instance()

    balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)
    state_pb = zk.get_balancer_state(namespace_id, balancer_id)

    rev_pbs = []
    active_rev_ctime = None
    entity = L7BalancerStateHandler(state_pb).select_balancer()

    if entity:
        rev_status = entity.select_active_rev()
        if rev_status is not None and (
            active_rev_ctime is None or rev_status.pb.ctime.ToMicroseconds() > active_rev_ctime
        ):
            active_rev_ctime = rev_status.pb.ctime.ToMicroseconds()

        for rev_status in entity.list_revs():
            if active_rev_ctime is None or rev_status.pb.ctime.ToMicroseconds() >= active_rev_ctime:
                rev_pb = mongo.get_balancer_rev(rev_status.pb.revision_id)
                rev_pbs.append(rev_pb)

    if not rev_pbs:
        raise rpc.exceptions.InternalError('No balancer revisions present in state')

    rev_index_pbs = []
    for rev_pb in rev_pbs:
        if active_rev_ctime is None or rev_pb.meta.ctime.ToMicroseconds() >= active_rev_ctime:
            ind_pb = get_rev_index_pb(namespace_id, rev_pb.spec)
            ind_pb.id = rev_pb.meta.id
            ind_pb.ctime.CopyFrom(rev_pb.meta.ctime)
            rev_index_pbs.append(ind_pb)

    IDao.instance().update_indices(
        object_name='balancer',
        id=balancer_id,
        namespace_id=namespace_id,
        version=balancer_pb.meta.version,
        rev_index_pbs=rev_index_pbs)

    return api_pb2.UpdateBalancerIndicesResponse()
