# coding: utf-8
import logging
import re
import six
from datetime import datetime

from sepelib.core import config

from awacs.lib import rpc
from awacs.lib.rpc import exceptions
from awacs.lib.gutils import gevent_idle_iter
from awacs.lib.certs import decimal_to_hexadecimal
from awacs.lib.order_processor.model import cancel_order
from awacs.lib.rpc.exceptions import ForbiddenError, BadRequestError
from awacs.model import db, apicache
from awacs.model.apicache import IAwacsApiCache
from awacs.model.cache import IAwacsCache
from awacs.model.dao import IDao
from awacs.model.db import IMongoStorage
from awacs.model.errors import ConflictError
from awacs.model.util import clone_pb, omit_duplicate_items_from_auth
from awacs.model.zk import IZkStorage
from infra.awacs.proto import api_pb2, model_pb2
from awacs.web.auth.core import authorize_remove, get_acl, authorize_cert_update, authorize_create
from awacs.web.validation import certificate
from .util import AwacsBlueprint, forbid_action_during_namespace_order, validate_namespace_total_objects_count


certificate_service_bp = AwacsBlueprint('rpc_certificate_service', __name__, '/api')


def _augment_req_discoverability(req_discoverability_pb, curr_discoverability_pb, login, now):
    """
    :type req_discoverability_pb: model_pb2.DiscoverabilityCondition
    :type curr_discoverability_pb: model_pb2.DiscoverabilityCondition | None
    :type login: six.text_type
    :type now: datetime
    """
    if curr_discoverability_pb is not None:
        if req_discoverability_pb.default.value != curr_discoverability_pb.default.value:
            req_discoverability_pb.default.mtime.FromDatetime(now)
            req_discoverability_pb.default.author = login
        for location, cond_pb in six.iteritems(req_discoverability_pb.per_location.values):
            if (location in curr_discoverability_pb.per_location.values and
                    curr_discoverability_pb.per_location.values[location].value != cond_pb.value):
                cond_pb.mtime.FromDatetime(now)
                cond_pb.author = login
    else:
        req_discoverability_pb.default.mtime.FromDatetime(now)
        req_discoverability_pb.default.author = login
        for _, cond_pb in six.iteritems(req_discoverability_pb.per_location.values):
            cond_pb.mtime.FromDatetime(now)
            cond_pb.author = login


@certificate_service_bp.method('OrderCertificate',
                               request_type=api_pb2.OrderCertificateRequest,
                               response_type=api_pb2.OrderCertificateResponse)
def order_certificate(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.OrderCertificateRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    omit_duplicate_items_from_auth(req_pb.meta.auth)

    cert_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id

    cache = IAwacsCache.instance()
    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id=namespace_id)
    validate_namespace_total_objects_count('certificate', len(cache.list_all_certs(namespace_id)), namespace_pb)
    authorize_create(namespace_pb, auth_subject)

    if zk.does_cert_exist(namespace_id, cert_id):
        raise rpc.exceptions.ConflictError('Certificate "{}" already exists in namespace "{}"'.format(
            cert_id, namespace_id))

    try:
        cert_pb = IDao.instance().create_cert(
            meta_pb=req_pb.meta,
            order_pb=req_pb.order,
            login=auth_subject.login)
    except ConflictError:
        raise rpc.exceptions.ConflictError('Certificate "{}" already exists in namespace "{}"'.format(
            cert_id, namespace_id))

    return api_pb2.OrderCertificateResponse(certificate=cert_pb)


@certificate_service_bp.method('CreateCertificate',
                               request_type=api_pb2.CreateCertificateRequest,
                               response_type=api_pb2.CreateCertificateResponse)
def create_certificate(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateCertificateRequest
    """
    is_author_root = auth_subject.login in config.get_value('run.root_users', default=())
    if not is_author_root:
        raise ForbiddenError('Can only be created directly by roots, use OrderCertificate instead')

    certificate.validate_request(req_pb, auth_subject)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    namespace_id = req_pb.meta.namespace_id
    certificate_id = req_pb.meta.id

    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id=namespace_id)
    authorize_create(namespace_pb, auth_subject)

    if zk.does_cert_exist(namespace_id, certificate_id):
        raise rpc.exceptions.ConflictError(
            'Certificate "{}" already exists in namespace "{}".'.format(certificate_id, namespace_id))

    for cert_pb in IAwacsCache.instance().list_all_certs():
        if cert_pb.spec.storage == req_pb.spec.storage:
            raise rpc.exceptions.ConflictError(
                'Existing certificate {}:{} has the same storage as the one being imported ({}:{}), {})'.format(
                    req_pb.meta.namespace_id, req_pb.meta.id,
                    cert_pb.meta.namespace_id, cert_pb.meta.id, cert_pb.spec.storage))

    cert_pb = IDao.instance().create_cert(
        meta_pb=req_pb.meta,
        spec_pb=req_pb.spec,
        login=auth_subject.login)

    return api_pb2.CreateCertificateResponse(certificate=cert_pb)


@certificate_service_bp.method('UpdateCertificate',
                               request_type=api_pb2.UpdateCertificateRequest,
                               response_type=api_pb2.UpdateCertificateResponse)
def update_certificate(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateCertificateRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    namespace_id = req_pb.meta.namespace_id
    cert_id = req_pb.meta.id

    zk = IZkStorage.instance()

    cert_pb = zk.must_get_cert(namespace_id, cert_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_cert_update(cert_pb, req_pb, auth_subject)
    except ForbiddenError:
        authorize_cert_update(cert_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    auth_has_changed = req_pb.meta.HasField('auth') and cert_pb.meta.auth != req_pb.meta.auth
    unrevokable_has_changed = (req_pb.meta.HasField('unrevokable') and
                               cert_pb.meta.unrevokable != req_pb.meta.unrevokable)
    discoverability_has_changed = (req_pb.meta.HasField('discoverability') and
                                   cert_pb.meta.discoverability != req_pb.meta.discoverability)
    spec_has_changed = req_pb.spec and req_pb.spec != model_pb2.CertificateSpec() and cert_pb.spec != req_pb.spec

    now = datetime.utcnow()
    is_author_root = auth_subject.login in config.get_value('run.root_users', default=())
    if unrevokable_has_changed:
        if not is_author_root:
            raise ForbiddenError('"meta.unrevokable" can only be changed by roots')
        req_pb.meta.unrevokable.mtime.FromDatetime(now)
        req_pb.meta.unrevokable.author = auth_subject.login
    if discoverability_has_changed:
        if not is_author_root:
            raise ForbiddenError('"meta.discoverability" can only be changed by roots')
        _augment_req_discoverability(
            req_discoverability_pb=req_pb.meta.discoverability,
            curr_discoverability_pb=cert_pb.meta.discoverability if cert_pb.meta.HasField('discoverability') else None,
            login=auth_subject.login,
            now=now)

    if not auth_has_changed and not spec_has_changed and not unrevokable_has_changed and not discoverability_has_changed:
        return api_pb2.UpdateCertificateResponse(certificate=cert_pb)

    cert_pb = IDao.instance().update_cert(
        namespace_id=namespace_id,
        cert_id=cert_id,
        comment=req_pb.meta.comment,
        login=auth_subject.login,
        version=req_pb.meta.version,
        updated_meta_pb=req_pb.meta if (
                auth_has_changed or unrevokable_has_changed or discoverability_has_changed) else None,
        updated_spec_pb=req_pb.spec if spec_has_changed else None,
    )
    balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states(namespace_id=namespace_id)
    cert_pb = apicache.copy_per_balancer_statuses(entity_pb=cert_pb, balancer_state_pbs=balancer_state_pbs)
    return api_pb2.UpdateCertificateResponse(certificate=cert_pb)


@certificate_service_bp.method('UpdateCertificateRenewal',
                               request_type=api_pb2.UpdateCertificateRenewalRequest,
                               response_type=api_pb2.UpdateCertificateRenewalResponse)
def update_certificate_renewal(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateCertificateRenewalRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.meta.namespace_id
    cert_id = req_pb.meta.id

    zk = IZkStorage.instance()

    cert_renewal_pb = zk.must_get_cert_renewal(namespace_id, cert_id)
    try:
        authorize_cert_update(cert_renewal_pb, req_pb, auth_subject)
    except ForbiddenError:
        namespace_pb = zk.must_get_namespace(namespace_id)
        authorize_cert_update(cert_renewal_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    auth_has_changed = req_pb.meta.HasField('auth') and cert_renewal_pb.meta.auth != req_pb.meta.auth
    if auth_has_changed:
        raise BadRequestError('Changing certificate renewal auth is prohibited')

    target_discoverability_has_changed = (req_pb.meta.HasField('target_discoverability') and
                                          cert_renewal_pb.meta.target_discoverability != req_pb.meta.target_discoverability)
    if target_discoverability_has_changed:
        raise BadRequestError('Changing certificate renewal target_discoverability is prohibited')

    spec_has_changed = (req_pb.spec and
                        req_pb.spec != model_pb2.CertificateSpec() and
                        cert_renewal_pb.spec != req_pb.spec)
    target_rev_has_changed = cert_renewal_pb.meta.target_rev != req_pb.meta.target_rev

    if not spec_has_changed and not target_rev_has_changed:
        return api_pb2.UpdateCertificateRenewalResponse(certificate_renewal=cert_renewal_pb)

    cert_renewal_pb = IDao.instance().update_cert_renewal(
        namespace_id=namespace_id,
        cert_renewal_id=cert_id,
        comment=req_pb.meta.comment,
        login=auth_subject.login,
        version=req_pb.meta.version,
        updated_spec_pb=req_pb.spec if spec_has_changed else None,
        updated_target_rev=req_pb.meta.target_rev if target_rev_has_changed else None,
    )
    return api_pb2.UpdateCertificateRenewalResponse(certificate_renewal=cert_renewal_pb)


@certificate_service_bp.method('ListCertificates',
                               request_type=api_pb2.ListCertificatesRequest,
                               response_type=api_pb2.ListCertificatesResponse,
                               max_in_flight=5)
def list_certificates(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListCertificatesRequest
    """
    certificate.validate_request(req_pb, auth_subject)

    c = IAwacsApiCache.instance()
    query = {}
    if req_pb.query.id_regexp:
        query[c.CertsQueryTarget.ID_REGEXP] = re.compile(req_pb.query.id_regexp)

    cert_pbs, total = c.list_namespace_certs(namespace_id=req_pb.namespace_id,
                                             skip=req_pb.skip or None,
                                             limit=req_pb.limit or None,
                                             query=query)
    resp_pb = api_pb2.ListCertificatesResponse(total=total)
    resp_pb.certificates.extend(cert_pbs)
    return resp_pb


@certificate_service_bp.method('ListCertificateRenewals',
                               request_type=api_pb2.ListCertificateRenewalsRequest,
                               response_type=api_pb2.ListCertificateRenewalsResponse,
                               max_in_flight=5)
def list_certificate_renewals(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListCertificateRenewalsRequest
    """
    certificate.validate_request(req_pb, auth_subject)

    c = IAwacsApiCache.instance()
    query = {}
    if req_pb.query.id_regexp:
        query[c.CertRenewalsQueryTarget.ID_REGEXP] = re.compile(req_pb.query.id_regexp)
    if req_pb.query.HasField('incomplete'):
        query[c.CertRenewalsQueryTarget.INCOMPLETE] = req_pb.query.incomplete.value
    if req_pb.query.HasField('paused'):
        query[c.CertRenewalsQueryTarget.PAUSED] = req_pb.query.paused.value
    if req_pb.query.validity_not_before.HasField('gte'):
        query[c.CertRenewalsQueryTarget.VALIDITY_NOT_BEFORE_GTE] = req_pb.query.validity_not_before.gte.ToDatetime()
    if req_pb.query.validity_not_before.HasField('lte'):
        query[c.CertRenewalsQueryTarget.VALIDITY_NOT_BEFORE_LTE] = req_pb.query.validity_not_before.lte.ToDatetime()

    sort = []
    if req_pb.sort_target == req_pb.TARGET_CERT_VALIDITY_NOT_AFTER:
        sort.append(c.CertRenewalsSortTarget.TARGET_CERT_VALIDITY_NOT_AFTER)
    else:
        sort.append(c.CertRenewalsSortTarget.ID)
    sort.append(-1 if req_pb.sort_order == api_pb2.DESCEND else 1)

    cert_renewal_pbs, total = c.list_cert_renewals(
        namespace_id=req_pb.namespace_id or None,
        skip=req_pb.skip or None,
        limit=req_pb.limit or None,
        query=query,
        sort=sort
    )
    resp_pb = api_pb2.ListCertificateRenewalsResponse(total=total)
    for cert_renewal_pb in gevent_idle_iter(cert_renewal_pbs, idle_period=50):
        resp_pb.certificate_renewals.add().CopyFrom(cert_renewal_pb)
    return resp_pb


@certificate_service_bp.method('GetCertificate',
                               request_type=api_pb2.GetCertificateRequest,
                               response_type=api_pb2.GetCertificateResponse)
def get_certificate(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetCertificateRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    cert_id = req_pb.id
    zk = IZkStorage.instance()
    cert_pb = zk.must_get_cert(namespace_id, cert_id)
    balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states(namespace_id=namespace_id)
    cert_pb = apicache.copy_per_balancer_statuses(entity_pb=cert_pb, balancer_state_pbs=balancer_state_pbs)
    if cert_pb.spec.fields.serial_number:  # see SWAT-6724
        cert_pb.spec.fields.serial_number = decimal_to_hexadecimal(cert_pb.spec.fields.serial_number)
    return api_pb2.GetCertificateResponse(certificate=cert_pb)


@certificate_service_bp.method('GetCertificateRenewal',
                               request_type=api_pb2.GetCertificateRenewalRequest,
                               response_type=api_pb2.GetCertificateRenewalResponse)
def get_certificate_renewal(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetCertificateRenewalRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    cert_id = req_pb.id
    zk = IZkStorage.instance()
    cert_renewal_pb = zk.must_get_cert_renewal(namespace_id, cert_id)
    if cert_renewal_pb.spec.fields.serial_number:  # see SWAT-6724
        cert_renewal_pb.spec.fields.serial_number = decimal_to_hexadecimal(cert_renewal_pb.spec.fields.serial_number)
    return api_pb2.GetCertificateRenewalResponse(certificate_renewal=cert_renewal_pb)


@certificate_service_bp.method('RemoveCertificate',
                               request_type=api_pb2.RemoveCertificateRequest,
                               response_type=api_pb2.RemoveCertificateResponse,
                               is_destructive=True)
def remove_certificate(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.RemoveCertificateRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    cert_id = req_pb.id

    zk = IZkStorage.instance()

    cert_pb = zk.must_get_cert(namespace_id, cert_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_remove(cert_pb, auth_subject)
    except ForbiddenError:
        authorize_remove(cert_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    certificate.validate_remove_cert(req_pb, cert_pb)

    updated_spec_pb = clone_pb(cert_pb.spec)
    updated_spec_pb.state = req_pb.state
    cert_pb = IDao.instance().update_cert(
        namespace_id=namespace_id,
        cert_id=cert_id,
        version=req_pb.version,
        comment='Marked as {} by {}'.format(model_pb2.CertificateSpec.State.Name(req_pb.state),
                                            auth_subject.login),
        login=auth_subject.login,
        updated_spec_pb=updated_spec_pb
    )
    return api_pb2.RemoveCertificateResponse(certificate=cert_pb)


@certificate_service_bp.method('CancelCertificateOrder',
                               request_type=api_pb2.CancelCertificateOrderRequest,
                               response_type=api_pb2.CancelCertificateOrderResponse)
def cancel_certificate_order(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelCertificateOrderRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    cert_id = req_pb.id

    zk = IZkStorage.instance()

    cert_pb = zk.must_get_cert(namespace_id, cert_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_remove(cert_pb, auth_subject)
    except ForbiddenError:
        authorize_remove(cert_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    certificate.validate_cancel_cert_order(cert_pb)

    for cert_pb in zk.update_cert(namespace_id=namespace_id, cert_id=cert_id, cert_pb=cert_pb):
        cancel_order(cert_pb, author=auth_subject.login, comment='Cancelled in UI')
    return api_pb2.CancelCertificateOrderResponse()


@certificate_service_bp.method('GetCertificateRevision',
                               request_type=api_pb2.GetCertificateRevisionRequest,
                               response_type=api_pb2.GetCertificateRevisionResponse)
def get_certificate_revision(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetCertificateRevisionRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    rev_id = req_pb.id

    mongo_storage = db.IMongoStorage.instance()
    rev_pb = mongo_storage.must_get_cert_rev(rev_id)
    return api_pb2.GetCertificateRevisionResponse(revision=rev_pb)


@certificate_service_bp.method('ListCertificateRevisions',
                               request_type=api_pb2.ListCertificateRevisionsRequest,
                               response_type=api_pb2.ListCertificateRevisionsResponse,
                               max_in_flight=5)
def list_certificate_revisions(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListCertificateRevisionsRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    cert_id = req_pb.id
    namespace_id = req_pb.namespace_id

    zk = IZkStorage.instance()
    zk.must_get_cert(namespace_id, cert_id)

    mongo_storage = db.IMongoStorage.instance()
    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = mongo_storage.list_cert_revs(namespace_id, cert_id, skip=skip, limit=limit)
    return api_pb2.ListCertificateRevisionsResponse(revisions=revs.items, total=revs.total)


@certificate_service_bp.method('RestoreCertificateFromBackup',
                               request_type=api_pb2.RestoreCertificateFromBackupRequest,
                               response_type=api_pb2.RestoreCertificateFromBackupResponse)
def restore_certificate_from_backup(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.RestoreCertificateFromBackupRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    cert_id = req_pb.id

    zk = IZkStorage.instance()

    cert_renewal_pb = zk.must_get_cert_renewal(namespace_id, cert_id)
    backup_spec_pb = IMongoStorage.instance().must_get_cert_rev(cert_renewal_pb.meta.target_rev).spec
    cert_pb = zk.must_get_cert(namespace_id, cert_id)
    try:
        authorize_cert_update(cert_pb, req_pb, auth_subject)
    except ForbiddenError:
        namespace_pb = zk.must_get_namespace(namespace_id)
        authorize_cert_update(cert_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))
    certificate.validate_restore_cert_from_backup(backup_spec_pb)

    comment = 'Paused to restore from backup'
    paused_pb = clone_pb(cert_renewal_pb.meta.paused)
    paused_pb.value = True
    paused_pb.author = auth_subject.login
    paused_pb.comment = comment
    paused_pb.mtime.GetCurrentTime()

    IDao.instance().update_cert_renewal(
        namespace_id=namespace_id,
        cert_renewal_id=cert_id,
        comment=comment,
        login=auth_subject.login,
        version=cert_renewal_pb.meta.version,
        updated_paused_pb=paused_pb,
    )

    IDao.instance().update_cert(
        namespace_id=namespace_id,
        cert_id=cert_id,
        version=req_pb.version,
        comment='Restored cert spec from backup',
        login=auth_subject.login,
        updated_spec_pb=backup_spec_pb,
    )
    logging.getLogger('cert-renewal-api').exception('Certificate %s was restored from backup', cert_id, exc_info=False)
    return api_pb2.RestoreCertificateFromBackupResponse()


@certificate_service_bp.method('UnpauseCertificateRenewal',
                               request_type=api_pb2.UnpauseCertificateRenewalRequest,
                               response_type=api_pb2.UnpauseCertificateRenewalResponse)
def unpause_certificate_renewal(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UnpauseCertificateRenewalRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    cert_id = req_pb.id

    zk = IZkStorage.instance()

    cert_renewal_pb = zk.must_get_cert_renewal(namespace_id, cert_id)
    try:
        authorize_cert_update(cert_renewal_pb, req_pb, auth_subject)
    except ForbiddenError:
        namespace_pb = zk.must_get_namespace(namespace_id)
        authorize_cert_update(cert_renewal_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    if not cert_renewal_pb.meta.paused.value:
        return api_pb2.UnpauseCertificateRenewalResponse()

    updated_target_discoverability_pb = None
    if req_pb.HasField('target_discoverability'):
        updated_target_discoverability_pb = req_pb.target_discoverability
        is_author_root = auth_subject.login in config.get_value('run.root_users', default=())
        if not is_author_root:
            raise ForbiddenError('"target_discoverability" can only be set by roots')
        _augment_req_discoverability(
            req_discoverability_pb=updated_target_discoverability_pb,
            curr_discoverability_pb=None,
            login=auth_subject.login,
            now=datetime.utcnow())

    comment = 'Unpaused renewal automation'
    paused_pb = clone_pb(cert_renewal_pb.meta.paused)
    paused_pb.value = False
    paused_pb.author = auth_subject.login
    paused_pb.comment = comment
    paused_pb.mtime.GetCurrentTime()
    IDao.instance().update_cert_renewal(
        namespace_id=namespace_id,
        cert_renewal_id=cert_id,
        comment=comment,
        login=auth_subject.login,
        version=cert_renewal_pb.meta.version,
        updated_paused_pb=paused_pb,
        updated_target_discoverability=updated_target_discoverability_pb
    )
    return api_pb2.UnpauseCertificateRenewalResponse()


@certificate_service_bp.method('ForceCertificateRenewal',
                               request_type=api_pb2.ForceCertificateRenewalRequest,
                               response_type=api_pb2.ForceCertificateRenewalResponse)
def force_certificate_renewal(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ForceCertificateRenewalRequest
    """
    certificate.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    cert_id = req_pb.id

    zk = IZkStorage.instance()

    cert_pb = zk.must_get_cert(namespace_id, cert_id)
    try:
        authorize_cert_update(cert_pb, req_pb, auth_subject)
    except ForbiddenError:
        namespace_pb = zk.must_get_namespace(namespace_id)
        authorize_cert_update(cert_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    for pb in zk.update_cert(namespace_id=namespace_id, cert_id=cert_id):
        if pb.meta.force_renewal.value:
            raise exceptions.BadRequestError('Renewal is already forced')
        pb.meta.force_renewal.value = True
        pb.meta.force_renewal.mtime.GetCurrentTime()
        pb.meta.force_renewal.author = auth_subject.login
        pb.meta.force_renewal.comment = 'Forced certificate renewal'
        pb.meta.force_renewal.cert_ttl = req_pb.cert_ttl

    return api_pb2.ForceCertificateRenewalResponse()
