# coding: utf-8
import logging

import inject
from sepelib.core import config
from six.moves.urllib import parse as urlparse

from awacs.lib import rpc
from awacs.lib.gutils import gevent_idle_iter
from awacs.model import cache
from awacs.model.dao import IDao
from awacs.model.zk import IZkStorage
from infra.awacs.proto import api_pb2, model_pb2
from awacs.web.util import AwacsBlueprint
from awacs.web.validation import component
from infra.swatlib import sandbox


logger = logging.getLogger(__name__)
component_service_bp = AwacsBlueprint('rpc_component_service', __name__, '/api')


def component_type_to_str(component_type):
    return model_pb2.ComponentMeta.Type.Name(component_type)


def component_type_from_str(component_type_str):
    return model_pb2.ComponentMeta.Type.Value(component_type_str)


def component_status_from_str(component_type_str):
    return model_pb2.ComponentStatus.StatusValue.Value(component_type_str)


def validate_is_root(login):
    if login not in config.get_value('run.root_users', ()):
        raise rpc.exceptions.ForbiddenError('Only root users can change components')


def check_and_complete_sandbox_resource(resource_pb, field_name='spec.source.sandbox_resource'):
    sb_client = inject.instance(sandbox.ISandboxClient)
    task = sb_client.get_task(resource_pb.task_id)
    if resource_pb.task_type != task['type']:
        raise rpc.exceptions.BadRequestError('"{}.task_type" is different with type of task '
                                             '{}, real: {}'.format(field_name, resource_pb.task_id, task['type']))
    resource = sb_client.get_resource(resource_pb.resource_id)
    if resource_pb.resource_type != resource['type']:
        raise rpc.exceptions.BadRequestError('"{}.resource_type" is different with type of resource '
                                             '{}, real: {}'.format(field_name, resource_pb.resource_id,
                                                                   resource['type']))
    skynet_id = resource['skynet_id']
    if resource_pb.rbtorrent and resource_pb.rbtorrent != skynet_id:
        raise rpc.exceptions.BadRequestError('"{}.rbtorrent" is different with skynet_id of resource '
                                             '{}, real: {}'.format(field_name, resource_pb.resource_id, skynet_id))
    else:
        resource_pb.rbtorrent = skynet_id


def check_url_file(resource_pb, field_name='spec.source.url_file'):
    parsed = urlparse.urlparse(resource_pb.url)
    if not all([parsed.scheme, parsed.netloc, parsed.path]):
        raise rpc.exceptions.BadRequestError('"{}.url": is not valid URL'.format(field_name))


@component_service_bp.method('GetComponent',
                             request_type=api_pb2.GetComponentRequest,
                             response_type=api_pb2.GetComponentRequest)
def get_component(req_pb, _):
    """
    :type req_pb: api_pb2.GetComponentRequest
    :rtype: api_pb2.GetComponentRequest
    """
    component.validate_request(req_pb)
    version = req_pb.version
    if req_pb.consistency == api_pb2.STRONG:
        zk = IZkStorage.instance()
        component_pb = zk.must_get_component(req_pb.type, version, sync=True)
    else:
        c = cache.IAwacsCache.instance()
        component_pb = c.must_get_component(component_type=req_pb.type, version=version)
    return api_pb2.GetComponentResponse(component=component_pb)


@component_service_bp.method('ListComponents',
                             request_type=api_pb2.ListComponentsRequest,
                             response_type=api_pb2.ListComponentsResponse,
                             max_in_flight=5)
def list_components(req_pb, _):
    """
    :type req_pb: api_pb2.ListComponentsRequest
    :rtype: api_pb2.ListComponentsResponse
    """
    component.validate_request(req_pb)
    if not req_pb.HasField('field_mask'):
        req_pb.field_mask.AllFieldsFromDescriptor(model_pb2.Component.DESCRIPTOR)
    c = cache.IAwacsCache.instance()

    query = {}
    if req_pb.query.type_in:
        query[c.ComponentQueryTarget.TYPE_IN] = [component_type_from_str(t) for t in req_pb.query.type_in]
    if req_pb.query.status_in:
        query[c.ComponentQueryTarget.STATUS_IN] = [component_status_from_str(t) for t in req_pb.query.status_in]

    sort = []
    if req_pb.sort_target == req_pb.CTIME:
        sort.append(c.ComponentsSortTarget.CTIME)
    elif req_pb.sort_target == req_pb.TYPE:
        sort.append(c.ComponentsSortTarget.TYPE)
    elif req_pb.sort_target == req_pb.VERSION:
        sort.append(c.ComponentsSortTarget.VERSION)
    elif req_pb.sort_target == req_pb.STATUS:
        sort.append(c.ComponentsSortTarget.STATUS)
    if sort:
        sort.append(-1 if req_pb.sort_order == api_pb2.DESCEND else 1)

    component_pbs, total = c.list_components(skip=req_pb.skip or None,
                                             limit=req_pb.limit or None,
                                             query=query,
                                             sort=sort or None)

    resp_pb = api_pb2.ListComponentsResponse(total=total)
    for component_pb in gevent_idle_iter(component_pbs, idle_period=30):
        req_pb.field_mask.MergeMessage(component_pb, resp_pb.components.add())
    return resp_pb


@component_service_bp.method('GetComponentUsage',
                             request_type=api_pb2.GetComponentUsageRequest,
                             response_type=api_pb2.GetComponentUsageResponse,
                             max_in_flight=5)
def get_component_usage(req_pb, _):
    """
    :type req_pb: api_pb2.GetComponentUsageRequest
    :rtype: api_pb2.GetComponentUsageResponse
    """
    component.validate_request(req_pb)
    c = cache.IAwacsCache.instance()
    by_version = c.count_balancer_usage_by_component_version(req_pb.type)
    return api_pb2.GetComponentUsageResponse(balancer_counts_by_version=by_version)


@component_service_bp.method('DraftComponent',
                             request_type=api_pb2.DraftComponentRequest,
                             response_type=api_pb2.DraftComponentResponse)
def draft_component(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.DraftComponentRequest
    :rtype: api_pb2.DraftComponentRequest
    """
    component.validate_request(req_pb)
    validate_is_root(auth_subject.login)
    if req_pb.spec.source.HasField('sandbox_resource'):
        check_and_complete_sandbox_resource(req_pb.spec.source.sandbox_resource)
    if req_pb.spec.source.HasField('url_file'):
        check_url_file(req_pb.spec.source.url_file)

    component_type = component_type_to_str(req_pb.type)
    version = req_pb.version

    zk = IZkStorage.instance()
    if zk.does_component_exist(component_type, version):
        raise rpc.exceptions.ConflictError(
            'Component "{}" of version "{}" already exists'.format(component_type, version))

    component_pb = IDao.instance().draft_component(
        component_type=req_pb.type,
        version=version,
        spec_pb=req_pb.spec,
        login=auth_subject.login,
        message=req_pb.message,
        startrek_issue_key=req_pb.startrek_issue_key,
    )

    return api_pb2.DraftComponentResponse(component=component_pb)


@component_service_bp.method('PublishComponent',
                             request_type=api_pb2.PublishComponentRequest,
                             response_type=api_pb2.PublishComponentResponse)
def publish_component(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.PublishComponentRequest
    :rtype: api_pb2.PublishComponentResponse
    """
    component.validate_request(req_pb)
    validate_is_root(auth_subject.login)
    version = req_pb.version

    zk = IZkStorage.instance()
    component_pb = zk.must_get_component(req_pb.type, version)
    if component_pb.status.status != component_pb.status.DRAFTED:
        raise rpc.exceptions.BadRequestError('Only drafted components can be published')
    if component_pb.status.drafted.author == auth_subject.login:
        raise rpc.exceptions.BadRequestError('Different people should draft and publish one component')
    component_pb = IDao.instance().publish_component(
        component_type=req_pb.type,
        version=version,
        startrek_issue_key=req_pb.startrek_issue_key,
        login=auth_subject.login,
        message=req_pb.message,
    )
    return api_pb2.PublishComponentResponse(component=component_pb)


@component_service_bp.method('RetireComponent',
                             request_type=api_pb2.RetireComponentRequest,
                             response_type=api_pb2.RetireComponentResponse,
                             is_destructive=True)
def retire_component(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.RetireComponentRequest
    :rtype: api_pb2.RetireComponentResponse
    """
    component.validate_request(req_pb)
    validate_is_root(auth_subject.login)
    component_type_str = component_type_to_str(req_pb.type)
    version = req_pb.version

    zk = IZkStorage.instance()
    component_pb = zk.must_get_component(req_pb.type, version)
    if component_pb.status.status != component_pb.status.PUBLISHED:
        raise rpc.exceptions.BadRequestError('Only published components can be retired')
    if component_pb.status.published.marked_as_default:
        raise rpc.exceptions.BadRequestError('Default components can not be retired')
    if req_pb.superseded_by == version:
        raise rpc.exceptions.BadRequestError('"superseded_by": can not be superseded by itself')
    superseded_by_component = zk.get_component(req_pb.type, req_pb.superseded_by)
    if superseded_by_component is None:
        raise rpc.exceptions.BadRequestError('"superseded_by": component {} of version "{}" does not exist'
                                             ''.format(component_type_str, req_pb.superseded_by))
    if superseded_by_component.status.status != superseded_by_component.status.PUBLISHED:
        raise rpc.exceptions.BadRequestError('"superseded_by": must be PUBLISHED version')

    c = cache.IAwacsCache.instance()
    for balancer_pb in c.list_balancers().items:
        balancer_component_pb = getattr(balancer_pb.spec.components, component_type_str.lower(), None)
        if balancer_component_pb is None:
            raise rpc.exceptions.ServiceUnavailable('No "{}" field in "spec.components" of balancer "{}:{}"'.format(
                component_type_str.lower(), balancer_pb.meta.namespace_id, balancer_pb.meta.id
            ))
        if balancer_component_pb.version == version:
            raise rpc.exceptions.BadRequestError('Component {} of version {} is used in balancer "{}:{}"'
                                                 .format(component_type_str, version, balancer_pb.meta.namespace_id,
                                                         balancer_pb.meta.id))
    component_pb = IDao.instance().retire_component(req_pb.type, version, req_pb.superseded_by, auth_subject.login,
                                                    message=req_pb.message)
    return api_pb2.RetireComponentResponse(component=component_pb)


@component_service_bp.method('SetComponentAsDefault',
                             request_type=api_pb2.SetComponentAsDefaultRequest,
                             response_type=api_pb2.SetComponentAsDefaultResponse,
                             is_destructive=True)
def set_component_as_default(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.SetComponentAsDefaultRequest
    :rtype: api_pb2.SetComponentAsDefaultResponse
    """
    component.validate_request(req_pb)
    validate_is_root(auth_subject.login)
    version = req_pb.version
    cluster = req_pb.cluster

    zk = IZkStorage.instance()
    component_pb = zk.must_get_component(req_pb.type, version)
    if component_pb.status.status != component_pb.status.PUBLISHED:
        raise rpc.exceptions.BadRequestError('Only published components can be marked as default')
    if cluster in component_pb.status.published.marked_as_default:
        raise rpc.exceptions.BadRequestError('Component is already marked as default in {}'.format(cluster))
    IDao.instance().set_component_version_as_default(req_pb.type, version, cluster, login=auth_subject.login,
                                                     message=req_pb.message)
    return api_pb2.SetComponentAsDefaultResponse()


@component_service_bp.method('RemoveComponent',
                             request_type=api_pb2.RemoveComponentRequest,
                             response_type=api_pb2.RemoveComponentResponse,
                             is_destructive=True)
def remove_component(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.RemoveComponentRequest
    :rtype: api_pb2.RemoveComponentResponse
    """
    component.validate_request(req_pb)
    validate_is_root(auth_subject.login)
    version = req_pb.version

    zk = IZkStorage.instance()
    component_pb = zk.must_get_component(req_pb.type, version)
    if component_pb.status.status != component_pb.status.DRAFTED:
        raise rpc.exceptions.BadRequestError('Only drafted components can be removed')
    IDao.instance().delete_component(req_pb.type, version)
    return api_pb2.RemoveComponentResponse()
