"""
Debugging utilities.
"""
from __future__ import unicode_literals

import copy
import sys
import textwrap
import traceback

import flask
import objgraph
import six
import ujson
from datetime import datetime

from awacs.model.cache import IAwacsCache
from infra.swatlib.auth.util import login_exempt


debug_blueprint = flask.Blueprint('debug', __name__, url_prefix='/_debug')

AWACS_UI_BALANCER_HREF_TEMPLATE = 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/balancers/list/{}/show/'
AWACS_UI_L3_BALANCER_HREF_TEMPLATE = 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/l3-balancers/list/{}/show/'
AWACS_UI_CERT_HREF_TEMPLATE = 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/certs/list/{}/show/'
AWACS_UI_NAMESPACE_HREF_TEMPLATE = 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/show/'
L3MGR_UI_CONFIG_HREF_TEMPLATE = 'https://l3.tt.yandex-team.ru/service/{service_id}/config/{config_id}'
NANNY_UI_SERVICE_HREF_TEMPLATE = 'https://nanny.yandex-team.ru/ui/#/services/catalog/{service_id}/runtime_attrs_history/{snapshot_id}/'


def wrap_in_bold_red_if_greater(num, limit):
    if num > limit:
        return u'<span style="font-weight: bold; color: red;">{}</span>'.format(num)
    else:
        return num


@debug_blueprint.route('/')
@login_exempt
def debug_index_view():
    """
    Page with links to available debug pages.
    """
    html = textwrap.dedent("""
    <strong>Debug links</strong>
    <ul>
        <li><a href="{}">View in-progress configs</a></li>
        <li><a href="{}">View paused</a></li>
        <li><a href="{}">View objects being created</a></li>
        <li><a href="{}">View thread stacks</a></li>
        <li><a href="{}">View most common types</a></li>
        <li><a href="{}">View FQDN cache</a></li>
        <li><a href="{}">View large namespaces</a></li>
        <li><a href="{}">View stuck alerting</a></li>
        <li><a href="{}">View indiscoverable certs</a></li>
    </ul>
    """.format(flask.url_for('debug.in_progress_view'),
               flask.url_for('debug.paused_view'),
               flask.url_for('debug.being_created_view'),
               flask.url_for('debug.thread_stacks_view'),
               flask.url_for('debug.most_common_types_view'),
               flask.url_for('debug.fqdn_cache_view'),
               flask.url_for('debug.large_namespaces_view'),
               flask.url_for('debug.stuck_alerting_view'),
               flask.url_for('debug.indiscoverable_certs_view'),
               ))
    return flask.Response(html)


@debug_blueprint.route('/in_progress/')
@login_exempt
def in_progress_view():
    lines = []
    stuck_only = flask.request.args.get('stuck_only', False)
    c = IAwacsCache.instance()
    now = datetime.utcnow()
    for l7_or_l3_or_downtimed, balancers_configs_progress in [
        ('L7', c.get_balancer_in_progress_configs(
            query={c.InProgressL7ConfigsQueryTarget.STUCK_ONLY: stuck_only, c.InProgressL7ConfigsQueryTarget.DOWNTIMED: False})
         ),
        ('Downtimed L7', c.get_balancer_in_progress_configs(
            query={c.InProgressL7ConfigsQueryTarget.STUCK_ONLY: stuck_only, c.InProgressL7ConfigsQueryTarget.DOWNTIMED: True})
         ),
        ('L3', c.get_l3_balancer_in_progress_configs(
            query={c.InProgressL3ConfigsQueryTarget.STUCK_ONLY: stuck_only})
         ),
    ]:
        lines.append(u'<h1>{} balancer in-progress configs</h1>'.format(l7_or_l3_or_downtimed))
        if not balancers_configs_progress:
            lines.append('<p>&mdash;</p>')
            continue
        lines.append('<ul>')
        for (namespace_id, id_), in_progress_configs in sorted(six.iteritems(balancers_configs_progress)):
            config = min(in_progress_configs)
            if l7_or_l3_or_downtimed in ('L7', 'Downtimed L7'):
                config_href = NANNY_UI_SERVICE_HREF_TEMPLATE.format(service_id=config.service_id,
                                                                    snapshot_id=config.snapshot_id)
                awacs_href = AWACS_UI_BALANCER_HREF_TEMPLATE.format(namespace_id, id_)
            else:
                config_href = L3MGR_UI_CONFIG_HREF_TEMPLATE.format(service_id=config.service_id,
                                                                   config_id=config.config_id)
                awacs_href = AWACS_UI_L3_BALANCER_HREF_TEMPLATE.format(namespace_id, id_)

            how_long = now - config.ctime
            conf_description = (u'<li><a href="{href}">{namespace_id}:{id_}</a> is '
                                u'<a href="{config_href}">in progress since {ctime} UTC</a> ({how_long})'.format(
                                    href=awacs_href,
                                    namespace_id=namespace_id,
                                    id_=id_,
                                    config_href=config_href,
                                    ctime=config.ctime,
                                    how_long=u'{} days {:.2f} hours'.format(how_long.days, how_long.seconds / 3600.)))
            if l7_or_l3_or_downtimed == 'Downtimed L7':
                downtime_flag_pb = c.get_downtime_flag(namespace_id, id_)
                conf_description += " -- downtimed until {} by {}".format(
                    downtime_flag_pb.not_after.ToDatetime(),
                    downtime_flag_pb.author
                )
            conf_description += "</li>"
            lines.append(conf_description)
        lines.append(u'</ul>')
    return flask.Response(u'\n'.join(lines), content_type='text/html')


@debug_blueprint.route('/large_namespaces/')
@login_exempt
def large_namespaces_view():
    c = IAwacsCache.instance()
    marked_lines = []
    unmarked_lines = []
    should_not_be_marked = copy.deepcopy(c._marked_large_namespaces)
    for ns_id, info in c.list_large_namespaces():
        line = (
            u'<li><a href="{href}">{namespace_id}</a> has {total} significant objects: '
            u'{domains} domains, {backends} backends, {upstreams} upstreams</li>'.format(
                href=AWACS_UI_NAMESPACE_HREF_TEMPLATE.format(ns_id),
                namespace_id=ns_id,
                domains=wrap_in_bold_red_if_greater(info['domains'], c._large_ns_individual_limit),
                backends=wrap_in_bold_red_if_greater(info['backends'], c._large_ns_individual_limit),
                upstreams=wrap_in_bold_red_if_greater(info['upstreams'], c._large_ns_individual_limit),
                total=wrap_in_bold_red_if_greater(info['domains'] + info['backends'] + info['upstreams'],
                                                  c._large_ns_total_limit)
            )
        )
        if ns_id in c._marked_large_namespaces:
            marked_lines.append(line)
            should_not_be_marked.discard(ns_id)
        else:
            unmarked_lines.append(line)

    lines = [u'<h1>Large namespaces</h1>']
    if marked_lines or unmarked_lines or should_not_be_marked:
        lines.append(u'<h2>Not marked as large in config</h2>')
        if unmarked_lines:
            lines.append(u'<ul>')
            lines.extend(unmarked_lines)
            lines.append(u'</ul>')
        else:
            lines.append('&mdash;')

        lines.append(u'<h2>Marked as large in config</h2>')
        if marked_lines:
            lines.append(u'<ul>')
            lines.extend(marked_lines)
            lines.append(u'</ul>')
        else:
            lines.append('&mdash;')

        lines.append(u'<h2>Marked as large in config, but are not large at this moment</h2>')
        if should_not_be_marked:
            lines.append(u'<ul>')
            for ns_id in should_not_be_marked:
                if ns_id not in c._namespace_objects_count:
                    lines.append(u'<li>Namespace "{}" does not exist</li>'.format(ns_id))
                else:
                    info = c._namespace_objects_count[ns_id]
                    lines.append(
                        u'<li><a href="{href}">{namespace_id}</a> has {total} significant objects: '
                        u'{domains} domains, {backends} backends, {upstreams} upstreams</li>'.format(
                            href=AWACS_UI_NAMESPACE_HREF_TEMPLATE.format(ns_id),
                            namespace_id=ns_id,
                            domains=info['domains'],
                            backends=info['backends'],
                            upstreams=info['upstreams'],
                            total=info['domains'] + info['backends'] + info['upstreams']
                        ))
            lines.append(u'</ul>')
        else:
            lines.append('&mdash;')

    else:
        lines.append('&mdash;')
    return flask.Response(u'\n'.join(lines), content_type='text/html')


@debug_blueprint.route('/stuck-alerting/')
@login_exempt
def stuck_alerting_view():
    c = IAwacsCache.instance()
    lines = [
        u'<li><a href="{href}">{namespace_id}</a></li>'.format(
            href=AWACS_UI_NAMESPACE_HREF_TEMPLATE.format(ns_id),
            namespace_id=ns_id,
        )
        for ns_id in c.list_namespaces_with_stuck_alerting()
    ]
    return flask.Response(
        u'<ul>' + u'\n'.join(lines) + u'</ul',
        content_type='text/html',
    )


@debug_blueprint.route('/paused/')
@login_exempt
def paused_view():
    lines = []
    paused_too_long_only = flask.request.args.get('paused_too_long_only', False)
    c = IAwacsCache.instance()
    now = datetime.utcnow()
    for l7_or_l3, paused_balancers in [
        ('L7', c.list_paused_balancers(paused_too_long_only=paused_too_long_only)),
        ('L3', c.list_paused_l3_balancers(paused_too_long_only=paused_too_long_only)),
    ]:
        lines.append(u'<h1>Paused {} balancers</h1>'.format(l7_or_l3))
        if not paused_balancers:
            lines.append('<p>&mdash;</p>')
            continue
        lines.append('<ul>')
        for paused in sorted(paused_balancers):
            if l7_or_l3 == 'L7':
                awacs_href = AWACS_UI_BALANCER_HREF_TEMPLATE.format(paused.namespace_id, paused.id_)
            else:
                awacs_href = AWACS_UI_L3_BALANCER_HREF_TEMPLATE.format(paused.namespace_id, paused.id_)
            how_long = now - paused.paused_at
            lines.append(
                u'<li><a href="{href}">{namespace_id}:{id_}</a> has been '
                u'paused since {paused_at} UTC ({how_long})</li>'.format(
                    href=awacs_href,
                    namespace_id=paused.namespace_id,
                    id_=paused.id_,
                    paused_at=paused.paused_at,
                    how_long=u'{} days {:.2f} hours'.format(how_long.days, how_long.seconds / 3600.)))
        lines.append(u'</ul>')
    return flask.Response(u'\n'.join(lines), content_type='text/html')


@debug_blueprint.route('/indiscoverable_certs/')
@login_exempt
def indiscoverable_certs_view():
    lines = []
    indiscoverable_too_long_only = flask.request.args.get('indiscoverable_too_long_only', False)
    c = IAwacsCache.instance()
    now = datetime.utcnow()
    indiscoverable_certs = c.list_indiscoverable_certs(indiscoverable_too_long_only=indiscoverable_too_long_only)
    lines.append(u'<h1>Indiscoverable certs</h1>')
    if indiscoverable_certs:
        lines.append('<ul>')
        for indiscoverable_cert in sorted(indiscoverable_certs):
            awacs_href = AWACS_UI_CERT_HREF_TEMPLATE.format(indiscoverable_cert.namespace_id, indiscoverable_cert.id_)
            how_long = now - indiscoverable_cert.since
            lines.append(
                u'<li><a href="{href}">{namespace_id}:{id_}</a> has been indiscoverable by default '
                u'since {since} UTC ({how_long})</li>'.format(
                    href=awacs_href,
                    namespace_id=indiscoverable_cert.namespace_id,
                    id_=indiscoverable_cert.id_,
                    since=indiscoverable_cert.since,
                    how_long=u'{} days {:.2f} hours'.format(how_long.days, how_long.seconds / 3600.)))
        lines.append(u'</ul>')
    else:
        lines.append('<p>&mdash;</p>')
    return flask.Response(u'\n'.join(lines), content_type='text/html')


@debug_blueprint.route('/being_created/')
@login_exempt
def being_created_view():
    lines = []
    stuck_only = flask.request.args.get('stuck_only', False)
    c = IAwacsCache.instance()
    now = datetime.utcnow()
    for ns_or_l3, being_created in (
            ('Namespace', c.list_namespaces_being_created(stuck_only=stuck_only)),
            ('L3 balancer', c.list_l3_balancers_being_created(stuck_only=stuck_only)),
    ):
        lines.append(u'<h1>{}s being created</h1>'.format(ns_or_l3))
        if not being_created:
            lines.append('<p>&mdash;</p>')
        lines.append('<ul>')
        for id_, state in sorted(six.iteritems(being_created)):
            if ns_or_l3 == 'Namespace':
                awacs_href = AWACS_UI_NAMESPACE_HREF_TEMPLATE.format(id_)
            else:
                awacs_href = AWACS_UI_L3_BALANCER_HREF_TEMPLATE.format(id_[0], id_[1])
                id_ = ':'.join(id_)
            how_long = now - state.ctime
            lines.append(
                u'<li><a href="{href}">{id_}</a> is '
                u'being created since {ctime} UTC ({how_long}) Current state: {state}</li>'.format(
                    href=awacs_href,
                    id_=id_,
                    ctime=state.ctime,
                    how_long=u'{} days {:.2f} hours'.format(how_long.days, how_long.seconds / 3600.),
                    state=state.state))
        lines.append(u'</ul>')
    return flask.Response(u'\n'.join(lines), content_type='text/html')


@debug_blueprint.route('/thread_stacks/')
@login_exempt
def thread_stacks_view():
    """
    Returns human readable list of all thread stacks.
    """
    lines = [u'*** STACKTRACE - START ***']
    for thread_id, stack in sys._current_frames().items():
        lines.append(u"\n# ThreadID: {}".format(thread_id))
        for filename, lineno, name, line in traceback.extract_stack(stack):
            lines.append(u'File: "{}", line {}, in {}'.format(filename, lineno, name))
            if line:
                lines.append(line.strip())
    return flask.Response(u'\n'.join(lines).encode('utf-8'), content_type='text/plain')


@debug_blueprint.route('/most_common_types')
@login_exempt
def most_common_types_view():
    """
    Returns top of objects types by object count.
    """
    lines = [u'*** MOST COMMON TYPES - START ***']
    for n, c in objgraph.most_common_types(limit=300):
        lines.append(u'{}\t{}\n'.format(n, c))
    return flask.Response(u'\n'.join(lines).encode('utf-8'), content_type='text/plain')


@debug_blueprint.route('/fqdn_cache')
@login_exempt
def fqdn_cache_view():
    """
    Returns json mapping of current FQDN cache.
    """
    return flask.Response(ujson.dumps(IAwacsCache.instance().list_domain_fqdns()), content_type='application/json')
