# coding: utf-8
import six
import logging

from awacs.lib import rpc, dns_resolver, validators
from awacs.lib.order_processor.model import cancel_order
from awacs.model import db, apicache, cache
from awacs.model.dao import IDao
from awacs.model.db import find_endpoint_set_revision_spec
from awacs.model.dns_records.dns_record import get_included_backend_ids, DnsRecordStateHandler
from awacs.model.util import omit_duplicate_items_from_auth, clone_pb
from awacs.model.zk import IZkStorage
from infra.awacs.proto import api_pb2, model_pb2
from awacs.web.auth.core import authorize_update, authorize_remove, get_acl, authorize_create
from awacs.web.validation import dns_record, util
from .util import AwacsBlueprint, forbid_action_during_namespace_order, validate_namespace_total_objects_count


dns_record_service_bp = AwacsBlueprint('rpc_dns_record_service', __name__, '/api')
log = logging.getLogger(__name__)


@dns_record_service_bp.method('CreateDnsRecord',
                              request_type=api_pb2.CreateDnsRecordRequest,
                              response_type=api_pb2.CreateDnsRecordResponse)
def create_dns_record(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateDnsRecordRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    dns_record_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id

    c = cache.IAwacsCache.instance()
    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id=namespace_id)
    validate_namespace_total_objects_count('dns_record', len(c.list_all_dns_records(namespace_id=namespace_id)),
                                           namespace_pb)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    authorize_create(namespace_pb, auth_subject)

    backends_type = req_pb.order.address.backends.type or req_pb.spec.address.backends.type
    if (namespace_pb.spec.easy_mode_settings.prohibit_explicit_dns_selector.value and
            backends_type not in (req_pb.spec.address.backends.BALANCERS, req_pb.spec.address.backends.L3_BALANCERS)):
        raise rpc.exceptions.ForbiddenError('Only DNS records with BALANCERS or L3_BALANCERS selector '
                                            'can be created in this namespace')

    if zk.does_dns_record_exist(namespace_id, dns_record_id):
        raise rpc.exceptions.ConflictError(
            'DNS record "{}" already exists in namespace "{}"'.format(dns_record_id, namespace_id))

    dns_record_pb, dns_record_state_pb = IDao.instance().create_dns_record(
        meta_pb=req_pb.meta,
        spec_pb=req_pb.spec if req_pb.spec and req_pb.spec != model_pb2.DnsRecordSpec() else None,
        order_content_pb=req_pb.order if req_pb.order and req_pb.order != model_pb2.DnsRecordOrder.Content() else None,
        login=auth_subject.login)

    apicache.copy_dns_record_statuses(dns_record_pb, dns_record_state_pb)
    return api_pb2.CreateDnsRecordResponse(dns_record=dns_record_pb)


@dns_record_service_bp.method('ListDnsRecords',
                              request_type=api_pb2.ListDnsRecordsRequest,
                              response_type=api_pb2.ListDnsRecordsResponse,
                              max_in_flight=5)
def list_dns_records(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListDnsRecordsRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    c = apicache.IAwacsApiCache.instance()
    dns_record_pbs, total = c.list_dns_records(namespace_id=req_pb.namespace_id,
                                               skip=req_pb.skip or None,
                                               limit=req_pb.limit or None)
    resp_pb = api_pb2.ListDnsRecordsResponse(total=total)
    resp_pb.dns_records.extend(dns_record_pbs)
    return resp_pb


@dns_record_service_bp.method('GetDnsRecord',
                              request_type=api_pb2.GetDnsRecordRequest,
                              response_type=api_pb2.GetDnsRecordResponse)
def get_dns_record(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetDnsRecordRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    dns_record_id = req_pb.id

    zk = IZkStorage.instance()
    dns_record_pb = zk.must_get_dns_record(namespace_id, dns_record_id)
    dns_record_state_pb = zk.must_get_dns_record_state(namespace_id, dns_record_id)
    dns_record_pb = apicache.copy_dns_record_statuses(dns_record_pb, dns_record_state_pb)
    return api_pb2.GetDnsRecordResponse(dns_record=dns_record_pb)


@dns_record_service_bp.method('UpdateDnsRecord',
                              request_type=api_pb2.UpdateDnsRecordRequest,
                              response_type=api_pb2.UpdateDnsRecordResponse)
def update_dns_record(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateDnsRecordRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    dns_record_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id

    zk = IZkStorage.instance()

    dns_record_pb = zk.must_get_dns_record(namespace_id, dns_record_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_update(dns_record_pb, req_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_update(dns_record_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    spec_has_changed = req_pb.HasField('spec') and dns_record_pb.spec != req_pb.spec
    auth_has_changed = req_pb.meta.HasField('auth') and dns_record_pb.meta.auth != req_pb.meta.auth

    if not any((spec_has_changed, auth_has_changed)):
        return api_pb2.UpdateDnsRecordResponse(dns_record=dns_record_pb)

    if spec_has_changed:
        if (namespace_pb.spec.easy_mode_settings.prohibit_explicit_dns_selector.value and
                req_pb.spec.address.backends.type != dns_record_pb.spec.address.backends.type and
                req_pb.spec.address.backends.type not in (req_pb.spec.address.backends.BALANCERS,
                                                          req_pb.spec.address.backends.L3_BALANCERS)):
            raise rpc.exceptions.ForbiddenError('Only DNS records with BALANCERS or L3_BALANCERS selector '
                                                'are allowed in this namespace')
        dns_record.validate_name_server_and_zone(dns_record_pb.spec, req_pb.spec)
        if req_pb.spec.address.backends.l3_balancers != dns_record_pb.spec.address.backends.l3_balancers:
            raise rpc.exceptions.BadRequestError('"spec.address.backends.l3_balancers" can not be changed')

    dns_record_pb = IDao.instance().update_dns_record(
        namespace_id=namespace_id,
        dns_record_id=dns_record_id,
        version=req_pb.meta.version,
        comment=req_pb.meta.comment,
        login=auth_subject.login,
        updated_spec_pb=req_pb.spec if spec_has_changed else None,
        updated_auth_pb=req_pb.meta.auth if auth_has_changed else None,
    )

    dns_record_state_pb = cache.IAwacsCache.instance().get_dns_record_state_or_empty(
        namespace_id=namespace_id, dns_record_id=dns_record_id)
    apicache.copy_dns_record_statuses(dns_record_pb, dns_record_state_pb)

    return api_pb2.UpdateDnsRecordResponse(dns_record=dns_record_pb)


@dns_record_service_bp.method('CancelDnsRecordOrder',
                              request_type=api_pb2.CancelDnsRecordOrderRequest,
                              response_type=api_pb2.CancelDnsRecordOrderResponse,
                              is_destructive=True)
def cancel_dns_record_order(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelDnsRecordOrderRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    dns_record_id = req_pb.id

    zk = IZkStorage.instance()

    dns_record_pb = zk.must_get_dns_record(namespace_id, dns_record_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_remove(dns_record_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_remove(dns_record_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    dns_record.validate_cancel_dns_record_order(dns_record_pb)

    for dns_record_pb in zk.update_dns_record(namespace_id=namespace_id,
                                              dns_record_id=dns_record_id,
                                              dns_record_pb=dns_record_pb):
        cancel_order(dns_record_pb, author=auth_subject.login, comment='Cancelled order')
    return api_pb2.CancelDnsRecordOrderResponse()


@dns_record_service_bp.method('RemoveDnsRecord',
                              request_type=api_pb2.RemoveDnsRecordRequest,
                              response_type=api_pb2.RemoveDnsRecordResponse,
                              is_destructive=True)
def remove_dns_record(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.RemoveDnsRecordRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    dns_record_id = req_pb.id

    zk = IZkStorage.instance()

    dns_record_pb = zk.must_get_dns_record(namespace_id, dns_record_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_remove(dns_record_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_remove(dns_record_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    dns_record.validate_remove_dns_record(dns_record_pb)

    updated_spec_pb = clone_pb(dns_record_pb.spec)
    updated_spec_pb.deleted = True
    dns_record_pb = IDao.instance().update_dns_record(
        namespace_id=namespace_id,
        dns_record_id=dns_record_id,
        version=req_pb.version,
        comment='Marked as removed by {}'.format(auth_subject.login),
        login=auth_subject.login,
        updated_spec_pb=updated_spec_pb
    )
    return api_pb2.RemoveDnsRecordResponse(dns_record=dns_record_pb)


@dns_record_service_bp.method('CreateDnsRecordOperation',
                              request_type=api_pb2.CreateDnsRecordOperationRequest,
                              response_type=api_pb2.CreateDnsRecordOperationResponse)
def create_dns_record_operation(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateDnsRecordOperationRequest
    """
    util.restrict_to_admins(auth_subject)
    dns_record.validate_request(req_pb, auth_subject)
    dns_record_op_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id

    c = cache.IAwacsCache.instance()
    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id=namespace_id)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    authorize_create(namespace_pb, auth_subject)

    if zk.does_dns_record_operation_exist(namespace_id, dns_record_op_id):
        raise rpc.exceptions.ConflictError(
            'DNS record operation "{}" already exists in namespace "{}"'.format(dns_record_op_id, namespace_id))

    dns_record_op_pb = IDao.instance().create_dns_record_operation(
        meta_pb=req_pb.meta,
        order_pb=req_pb.order,
        login=auth_subject.login)

    return api_pb2.CreateDnsRecordOperationResponse(operation=dns_record_op_pb)


@dns_record_service_bp.method('ListDnsRecordOperations',
                              request_type=api_pb2.ListDnsRecordOperationsRequest,
                              response_type=api_pb2.ListDnsRecordOperationsResponse,
                              max_in_flight=5)
def list_dns_record_operations(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListDnsRecordOperationsRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    c = apicache.IAwacsApiCache.instance()
    dns_record_pbs, total = c.list_dns_record_operations(namespace_id=req_pb.namespace_id,
                                                         dns_record_id=req_pb.dns_record_id,
                                                         skip=req_pb.skip or None,
                                                         limit=req_pb.limit or None)
    resp_pb = api_pb2.ListDnsRecordOperationsResponse(total=total)
    resp_pb.operations.extend(dns_record_pbs)
    return resp_pb


@dns_record_service_bp.method('GetDnsRecordOperation',
                              request_type=api_pb2.GetDnsRecordOperationRequest,
                              response_type=api_pb2.GetDnsRecordOperationResponse)
def get_dns_record_operation(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetDnsRecordOperationRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    dns_record_op_pb = IZkStorage.instance().must_get_dns_record_operation(req_pb.namespace_id, req_pb.id)
    return api_pb2.GetDnsRecordOperationResponse(operation=dns_record_op_pb)


@dns_record_service_bp.method('CancelDnsRecordOperation',
                              request_type=api_pb2.CancelDnsRecordOperationRequest,
                              response_type=api_pb2.CancelDnsRecordOperationResponse,
                              is_destructive=True)
def cancel_dns_record_operation(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelDnsRecordOperationRequest
    """
    util.restrict_to_admins(auth_subject)
    dns_record.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    dns_record_op_id = req_pb.id

    zk = IZkStorage.instance()

    dns_record_op_pb = zk.must_get_dns_record_operation(namespace_id, dns_record_op_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_remove(namespace_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    dns_record.validate_cancel_dns_record_operation(dns_record_op_pb)

    for dns_record_pb in zk.update_dns_record_operation(namespace_id=namespace_id,
                                                        dns_record_op_id=dns_record_op_id,
                                                        dns_record_op_pb=dns_record_op_pb):
        cancel_order(dns_record_pb, author=auth_subject.login, comment='Cancelled order')
    return api_pb2.CancelDnsRecordOrderResponse()


@dns_record_service_bp.method('GetDnsRecordState',
                              request_type=api_pb2.GetDnsRecordStateRequest,
                              response_type=api_pb2.GetDnsRecordStateResponse)
def get_dns_record_state(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetDnsRecordStateRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    dns_record_id = req_pb.id

    zk = IZkStorage.instance()
    dns_record_state_pb = zk.must_get_dns_record_state(namespace_id, dns_record_id)
    return api_pb2.GetDnsRecordStateResponse(state=dns_record_state_pb)


@dns_record_service_bp.method('GetDnsRecordRevision',
                              request_type=api_pb2.GetDnsRecordRevisionRequest,
                              response_type=api_pb2.GetDnsRecordRevisionResponse)
def get_dns_record_revision(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetDnsRecordRevisionRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    rev_id = req_pb.id

    rev_pb = db.IMongoStorage.instance().must_get_dns_record_rev(rev_id)
    return api_pb2.GetDnsRecordRevisionResponse(revision=rev_pb)


@dns_record_service_bp.method('ListDnsRecordRevisions',
                              request_type=api_pb2.ListDnsRecordRevisionsRequest,
                              response_type=api_pb2.ListDnsRecordRevisionsResponse,
                              max_in_flight=5)
def list_dns_record_revisions(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListDnsRecordRevisionsRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    dns_record_id = req_pb.id
    namespace_id = req_pb.namespace_id

    zk = IZkStorage.instance()
    zk.must_get_dns_record(namespace_id, dns_record_id)

    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = db.IMongoStorage.instance().list_dns_record_revs(
        namespace_id=namespace_id, dns_record_id=dns_record_id, skip=skip, limit=limit)
    return api_pb2.ListDnsRecordRevisionsResponse(revisions=revs.items, total=revs.total)


@dns_record_service_bp.method('ListNameServers',
                              request_type=api_pb2.ListNameServersRequest,
                              response_type=api_pb2.ListNameServersResponse,
                              max_in_flight=5)
def list_name_servers(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListNameServersRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    c = apicache.IAwacsApiCache.instance()
    name_server_pbs, total = c.list_name_servers(namespace_id=req_pb.namespace_id or None,
                                                 skip=req_pb.skip or None,
                                                 limit=req_pb.limit or None)
    resp_pb = api_pb2.ListNameServersResponse(total=total)
    resp_pb.name_servers.extend(name_server_pbs)
    return resp_pb


@dns_record_service_bp.method('GetNameServer',
                              request_type=api_pb2.GetNameServerRequest,
                              response_type=api_pb2.GetNameServerResponse,
                              max_in_flight=5)
def get_name_server(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetNameServerRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    return api_pb2.GetNameServerResponse(name_server=cache.IAwacsCache.instance().must_get_name_server(
        namespace_id=req_pb.namespace_id,
        name_server_id=req_pb.id
    ))


@dns_record_service_bp.method('GetNameServerConfig',
                              request_type=api_pb2.GetNameServerConfigRequest,
                              response_type=api_pb2.GetNameServerConfigResponse,
                              max_in_flight=2)
def get_name_server_config(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetNameServerConfigRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    name_server_id = req_pb.id

    c = cache.IAwacsCache.instance()
    name_server_pb = c.must_get_name_server(namespace_id=namespace_id, name_server_id=name_server_id)
    resp_pb = api_pb2.GetNameServerConfigResponse(zone=name_server_pb.spec.zone)
    for dns_record_pb in c.list_all_dns_records(name_server_full_id=(namespace_id, name_server_id)):
        unique_ipv6_addrs = set()
        if dns_record_pb.spec.address.backends.type == model_pb2.DnsBackendsSelector.L3_BALANCERS:
            for l3_pb in dns_record_pb.spec.address.backends.l3_balancers:
                l3_balancer_pb = c.get_l3_balancer(dns_record_pb.meta.namespace_id, l3_pb.id)
                if l3_balancer_pb is None:
                    log.error("Found non-existed L3 balancer {} in DNS record's '{}:{}' spec".format(
                        l3_pb.id, dns_record_pb.meta.namespace_id, dns_record_pb.meta.id
                    ))
                    continue
                for vs_pb in l3_balancer_pb.spec.virtual_servers:
                    if validators.ipv6(vs_pb.ip):
                        unique_ipv6_addrs.add(vs_pb.ip)
        else:
            # filter endpoint sets that are already removed from spec, but not processed by discoverer yet
            included_backend_ids = set(get_included_backend_ids(dns_record_pb.meta.namespace_id, dns_record_pb.spec))
            dns_record_state_pb = c.must_get_dns_record_state(dns_record_pb.meta.namespace_id, dns_record_pb.meta.id)
            valid_vector = DnsRecordStateHandler(dns_record_state_pb).generate_vectors().valid
            for endpoint_set_id, endpoint_set_version in six.iteritems(valid_vector.endpoint_set_versions):
                if endpoint_set_version.deleted:
                    continue
                if endpoint_set_id not in included_backend_ids:
                    continue
                endpoint_set_spec = find_endpoint_set_revision_spec(endpoint_set_version)
                for instance_pb in endpoint_set_spec.instances:
                    # XXX: we rely on the presence of ipv6 addresses in endpoint set spec,
                    # but AFAIK this is not ensured by backend controller
                    unique_ipv6_addrs.add(instance_pb.ipv6_addr)

        if unique_ipv6_addrs:
            config_record_pb = resp_pb.records.add()
            config_record_pb.address.zone = dns_record_pb.spec.address.zone
            for ipv6_addr in unique_ipv6_addrs:
                config_record_pb.address.ipv6_addrs.add(value=ipv6_addr)

    return resp_pb


@dns_record_service_bp.method('ResolveDnsRecord',
                              request_type=api_pb2.ResolveDnsRecordRequest,
                              response_type=api_pb2.ResolveDnsRecordResponse,
                              max_in_flight=2)
def resolve_dns_record(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ResolveDnsRecordRequest
    """
    dns_record.validate_request(req_pb, auth_subject)
    c = cache.IAwacsCache.instance()
    resp_pb = api_pb2.ResolveDnsRecordResponse()

    dns_record_pb = c.must_get_dns_record(dns_record_id=req_pb.id, namespace_id=req_pb.namespace_id)
    if dns_record_pb.spec.incomplete or dns_record_pb.order.cancelled.value:
        content_pb = dns_record_pb.order.content
    else:
        content_pb = dns_record_pb.spec
    name_server_pb = c.must_get_name_server(namespace_id=content_pb.name_server.namespace_id,
                                            name_server_id=content_pb.name_server.id)
    domain_name = u'{}.{}'.format(content_pb.address.zone, name_server_pb.spec.zone)
    try:
        resolve_response = dns_resolver.IDnsResolver.instance().resolve(domain_name)
        resp_pb.cnames_chain.extend(resolve_response.cnames_chain)
        resp_pb.ipv4_addrs.extend(resolve_response.ipv4_addrs)
        resp_pb.ipv6_addrs.extend(resolve_response.ipv6_addrs)
    except Exception as e:
        resp_pb.error = u'Failed to resolve domain name: {}'.format(six.text_type(e))
    return resp_pb
