import six
import re

from sepelib.core import config as appconfig

from awacs.lib import rpc, dns_resolver
from awacs.lib.order_processor.model import cancel_order
from awacs.model import db, apicache
from awacs.model.apicache import IAwacsApiCache
from awacs.model.cache import IAwacsCache
from awacs.model.dao import IDao
from awacs.model.errors import ConflictError
from awacs.model.util import clone_pb
from awacs.model.zk import IZkStorage
from infra.awacs.proto import api_pb2, model_pb2
from awacs.web.auth.core import authorize_remove, authorize_create, get_acl
from awacs.web.util import AwacsBlueprint, forbid_action_during_namespace_order, validate_namespace_total_objects_count
from awacs.web.validation import domain


domain_service_bp = AwacsBlueprint(u'rpc_domain_service', __name__, u'/api')


@domain_service_bp.method(u'CreateDomain',
                          request_type=api_pb2.CreateDomainRequest,
                          response_type=api_pb2.CreateDomainResponse)
def create_domain(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateDomainRequest
    """
    domain.validate_request(req_pb, auth_subject)

    domain_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id

    cache = IAwacsCache.instance()

    namespace_pb = cache.must_get_namespace(namespace_id=namespace_id)
    validate_namespace_total_objects_count(u'domain', len(cache.list_all_domains(namespace_id)), namespace_pb)
    authorize_create(namespace_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    if (req_pb.order.type == model_pb2.DomainSpec.Config.WILDCARD and
            namespace_id not in appconfig.get_value('run.can_have_wildcard_domain_namespace_ids', ())):
        raise rpc.exceptions.BadRequestError('Wildcard domain is not allowed in this namespace. Please contact support '
                                             'if you need to use it.')

    if cache.does_domain_exist(namespace_id, domain_id):
        raise rpc.exceptions.ConflictError(u'Domain "{}" already exists in namespace "{}"'.format(
            domain_id, namespace_id))

    try:
        pb = IDao.instance().create_domain(
            meta_pb=req_pb.meta,
            order_pb=req_pb.order,
            login=auth_subject.login)
    except ConflictError:
        raise rpc.exceptions.ConflictError(u'Domain "{}" already exists in namespace "{}"'.format(
            domain_id, namespace_id))

    return api_pb2.CreateDomainResponse(domain=pb)


@domain_service_bp.method(u'CreateDomainOperation',
                          request_type=api_pb2.CreateDomainOperationRequest,
                          response_type=api_pb2.CreateDomainOperationResponse)
def create_domain_operation(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateDomainOperationRequest
    """
    domain.validate_request(req_pb, auth_subject)

    domain_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id

    cache = IAwacsCache.instance()

    namespace_pb = cache.must_get_namespace(namespace_id=namespace_id)
    authorize_create(namespace_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    if cache.does_domain_operation_exist(namespace_id, domain_id):
        raise rpc.exceptions.ConflictError(u'Domain operation for "{}" is already in progress in namespace "{}"'.format(
            domain_id, namespace_id))

    try:
        pb = IDao.instance().create_domain_operation(
            meta_pb=req_pb.meta,
            order_pb=req_pb.order,
            login=auth_subject.login)
    except ConflictError:
        raise rpc.exceptions.ConflictError(u'Domain operation for "{}" is already in progress in namespace "{}"'.format(
            domain_id, namespace_id))

    return api_pb2.CreateDomainOperationResponse(operation=pb)


@domain_service_bp.method(u'ListDomains',
                          request_type=api_pb2.ListDomainsRequest,
                          response_type=api_pb2.ListDomainsResponse,
                          max_in_flight=5)
def list_domains(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListDomainsRequest
    """
    domain.validate_request(req_pb, auth_subject)

    c = IAwacsApiCache.instance()
    query = {}
    if req_pb.query.id_regexp:
        query[c.DomainsQueryTarget.ID_REGEXP] = re.compile(req_pb.query.id_regexp)
    domain_pbs, total = c.list_namespace_domains(namespace_id=req_pb.namespace_id,
                                                 skip=req_pb.skip or None,
                                                 limit=req_pb.limit or None,
                                                 query=query,
                                                 sort=(req_pb.sort_target, req_pb.sort_order))
    resp_pb = api_pb2.ListDomainsResponse(total=total)
    balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states(namespace_id=req_pb.namespace_id)
    for domain_pb in domain_pbs:
        domain_pb = apicache.copy_per_balancer_statuses(entity_pb=domain_pb, balancer_state_pbs=balancer_state_pbs)
        resp_pb.domains.add().CopyFrom(domain_pb)
    return resp_pb


@domain_service_bp.method(u'GetDomain',
                          request_type=api_pb2.GetDomainRequest,
                          response_type=api_pb2.GetDomainResponse)
def get_domain(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetDomainRequest
    """
    domain.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    domain_id = req_pb.id
    domain_pb = apicache.IAwacsApiCache.instance().must_get_domain(namespace_id, domain_id)
    balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states(namespace_id=namespace_id)
    domain_pb = apicache.copy_per_balancer_statuses(entity_pb=domain_pb, balancer_state_pbs=balancer_state_pbs)
    return api_pb2.GetDomainResponse(domain=domain_pb)


@domain_service_bp.method(u'ListDomainOperations',
                          request_type=api_pb2.ListDomainOperationsRequest,
                          response_type=api_pb2.ListDomainOperationsResponse,
                          max_in_flight=5)
def list_domain_operations(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListDomainOperationsRequest
    """
    domain.validate_request(req_pb, auth_subject)
    domain_pbs, total = IAwacsApiCache.instance().list_namespace_domain_operations(namespace_id=req_pb.namespace_id,
                                                                                   skip=req_pb.skip or None,
                                                                                   limit=req_pb.limit or None)
    return api_pb2.ListDomainOperationsResponse(total=total, operations=domain_pbs)


@domain_service_bp.method(u'GetDomainOperation',
                          request_type=api_pb2.GetDomainOperationRequest,
                          response_type=api_pb2.GetDomainOperationResponse)
def get_domain_operation(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetDomainOperationRequest
    """
    domain.validate_request(req_pb, auth_subject)
    pb = apicache.IAwacsApiCache.instance().must_get_domain_operation(req_pb.namespace_id, req_pb.id)
    return api_pb2.GetDomainOperationResponse(operation=pb)


@domain_service_bp.method(u'RemoveDomain',
                          request_type=api_pb2.RemoveDomainRequest,
                          response_type=api_pb2.RemoveDomainResponse,
                          is_destructive=True)
def remove_domain(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.RemoveDomainRequest
    """
    domain.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    domain_id = req_pb.id

    cache = IAwacsCache.instance()
    domain_pb = cache.must_get_domain(namespace_id, domain_id)
    namespace_pb = cache.must_get_namespace(namespace_id)
    authorize_remove(namespace_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    updated_spec_pb = clone_pb(domain_pb.spec)
    updated_spec_pb.deleted = True
    IDao.instance().update_domain(
        namespace_id=namespace_id,
        domain_id=domain_id,
        version=req_pb.version,
        comment=u'Marked as deleted by {}'.format(auth_subject.login),
        login=auth_subject.login,
        updated_spec_pb=updated_spec_pb
    )
    return api_pb2.RemoveDomainResponse()


@domain_service_bp.method(u'CancelDomainOrder',
                          request_type=api_pb2.CancelDomainOrderRequest,
                          response_type=api_pb2.CancelDomainOrderResponse)
def cancel_domain_order(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelDomainOrderRequest
    """
    domain.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    domain_id = req_pb.id

    cache = IAwacsCache.instance()

    domain_pb = cache.must_get_domain(namespace_id, domain_id)
    namespace_pb = cache.must_get_namespace(namespace_id)
    authorize_remove(namespace_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    domain.validate_cancel_domain_order(domain_pb)

    for domain_pb in IZkStorage.instance().update_domain(namespace_id=namespace_id,
                                                         domain_id=domain_id,
                                                         domain_pb=domain_pb):
        cancel_order(domain_pb, author=auth_subject.login, comment=u'Cancelled via API')
    return api_pb2.CancelDomainOrderResponse()


@domain_service_bp.method(u'CancelDomainOperation',
                          request_type=api_pb2.CancelDomainOperationRequest,
                          response_type=api_pb2.CancelDomainOperationResponse)
def cancel_domain_operation(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelDomainOperationRequest
    """
    domain.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    domain_id = req_pb.id

    cache = IAwacsCache.instance()

    namespace_pb = cache.must_get_namespace(namespace_id)
    authorize_remove(namespace_pb, auth_subject)
    domain_op_pb = cache.must_get_domain_operation(namespace_id, domain_id)
    domain.validate_cancel_domain_operation(domain_op_pb)

    for domain_op_pb in IZkStorage.instance().update_domain_operation(namespace_id=namespace_id, domain_id=domain_id,
                                                                      domain_operation_pb=domain_op_pb):
        cancel_order(domain_op_pb, author=auth_subject.login, comment=u'Cancelled via API')
    return api_pb2.CancelDomainOperationResponse()


@domain_service_bp.method(u'GetDomainRevision',
                          request_type=api_pb2.GetDomainRevisionRequest,
                          response_type=api_pb2.GetDomainRevisionResponse)
def get_domain_revision(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetDomainRevisionRequest
    """
    domain.validate_request(req_pb, auth_subject)
    rev_id = req_pb.id

    mongo_storage = db.IMongoStorage.instance()
    rev_pb = mongo_storage.must_get_domain_rev(rev_id)
    return api_pb2.GetDomainRevisionResponse(revision=rev_pb)


@domain_service_bp.method(u'ListDomainRevisions',
                          request_type=api_pb2.ListDomainRevisionsRequest,
                          response_type=api_pb2.ListDomainRevisionsResponse,
                          max_in_flight=5)
def list_domain_revisions(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListDomainRevisionsRequest
    """
    domain.validate_request(req_pb, auth_subject)
    domain_id = req_pb.id
    namespace_id = req_pb.namespace_id

    cache = IAwacsCache.instance()
    cache.must_get_domain(namespace_id, domain_id)

    mongo_storage = db.IMongoStorage.instance()
    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = mongo_storage.list_domain_revs(namespace_id, domain_id, skip=skip, limit=limit)
    return api_pb2.ListDomainRevisionsResponse(revisions=revs.items, total=revs.total)


@domain_service_bp.method(u'ChangeDomainOperation',
                          request_type=api_pb2.ChangeDomainOperationRequest,
                          response_type=api_pb2.ChangeDomainOperationResponse)
def change_domain_operation(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ChangeDomainOperationRequest
    """
    domain.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    domain_id = req_pb.id

    zk = IZkStorage.instance()

    domain_op_pb = zk.must_get_domain_operation(namespace_id, domain_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_remove(domain_op_pb, auth_subject, acl=get_acl(namespace_pb))

    if req_pb.HasField('confirm_removal_during_transfer'):
        for domain_op_pb in zk.update_domain_operation(namespace_id, domain_id, domain_op_pb):
            if not domain_op_pb.order.approval.before_removal:
                domain_op_pb.order.approval.before_removal = True
        return api_pb2.ChangeDomainOperationResponse()
    else:
        raise rpc.exceptions.BadRequestError(u'Supported order change operations: "confirm_removal_during_transfer"')


@domain_service_bp.method(u'ResolveDomainFqdns',
                          request_type=api_pb2.ResolveDomainFqdnsRequest,
                          response_type=api_pb2.ResolveDomainFqdnsResponse,
                          max_in_flight=2)
def resolve_domain_fqdns(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ResolveDomainFqdnsRequest
    """
    domain.validate_request(req_pb, auth_subject)
    c = IAwacsCache.instance()
    resolver = dns_resolver.IDnsResolver.instance()
    resp_pb = api_pb2.ResolveDomainFqdnsResponse()

    domain_pb = c.must_get_domain(domain_id=req_pb.id, namespace_id=req_pb.namespace_id)
    if domain_pb.spec.incomplete:
        fqdns = domain_pb.order.content.fqdns
    else:
        fqdns = domain_pb.spec.yandex_balancer.config.fqdns
    for fqdn in fqdns:
        try:
            resolve_response = resolver.resolve(fqdn)
        except Exception as e:
            resp_pb.response[fqdn].error = u'Failed to resolve domain name: {}'.format(six.text_type(e))
        else:
            resp_pb.response[fqdn].cnames_chain.extend(resolve_response.cnames_chain)
            resp_pb.response[fqdn].ipv4_addrs.extend(resolve_response.ipv4_addrs)
            resp_pb.response[fqdn].ipv6_addrs.extend(resolve_response.ipv6_addrs)
    return resp_pb
