# coding: utf-8
from sepelib.core import config

from awacs.lib import rpc
from awacs.lib.rpc.exceptions import ForbiddenError
from awacs.model import apicache
from awacs.model.cache import IAwacsCache
from awacs.model.dao import IDao
from awacs.model.db import IMongoStorage
from awacs.model.util import omit_duplicate_items_from_auth
from awacs.model.zk import IZkStorage
from infra.awacs.proto import api_pb2, model_pb2
from awacs.web.auth.core import authorize_update, get_acl, authorize_create
from awacs.web.validation import endpoint_set
from .util import AwacsBlueprint

endpoint_set_service_bp = AwacsBlueprint('rpc_endpoint_set_service', __name__, '/api')


@endpoint_set_service_bp.method('CreateEndpointSet',
                                request_type=api_pb2.CreateEndpointSetRequest,
                                response_type=api_pb2.CreateEndpointSetResponse)
def create_endpoint_set(req_pb, auth_subject):
    endpoint_set.validate_request(req_pb)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    backend_version = req_pb.backend_version
    namespace_id = req_pb.meta.namespace_id
    endpoint_set_id = req_pb.meta.id

    zk = IZkStorage.instance()
    dao = IDao.instance()

    namespace_pb = zk.must_get_namespace(namespace_id=namespace_id)
    authorize_create(namespace_pb, auth_subject)

    backend_pb = zk.must_get_backend(namespace_id=namespace_id, backend_id=endpoint_set_id)

    if backend_pb.meta.version != backend_version:
        raise rpc.exceptions.ConflictError(
            'Endpoint set can only be created for '
            'the current backend version ("{}").'.format(backend_pb.meta.version))
    if backend_pb.spec.selector.type != backend_pb.spec.selector.MANUAL:
        raise rpc.exceptions.ConflictError(
            'Backend type is not MANUAL, endpoint set can not be created.')

    if zk.does_endpoint_set_exist(namespace_id, endpoint_set_id):
        raise rpc.exceptions.ConflictError(
            'Endpoint set "{}" already exists in namespace "{}".'.format(endpoint_set_id, namespace_id))

    is_author_root = auth_subject.login in config.get_value('run.root_users', default=())
    if req_pb.meta.is_system.value:
        if not is_author_root:
            raise ForbiddenError('"meta.is_system": system endpoint set only be created by roots')
        if not backend_pb.meta.is_system.value:
            raise ForbiddenError('"meta.is_system": endpoint set is marked as system, but its backend is not system')

    if req_pb.spec.is_global.value:
        if not is_author_root:
            raise ForbiddenError('"spec.is_global": global endpoint set only be created by roots')
        if not backend_pb.spec.is_global.value:
            raise ForbiddenError('"spec.is_global": endpoint set is marked as global, but its backend is not global')

    if req_pb.meta.is_system.value and req_pb.spec.is_global.value:
        raise ForbiddenError('"meta.is_system,spec.is_global": global endpoint set cannot be system at the same time')
    if backend_pb.spec.is_global.value and not req_pb.spec.is_global.value:
        raise ForbiddenError('"spec.is_global": endpoint set is not marked as global, but its backend is global')

    selector_pb = backend_pb.spec.selector
    meta_pb = model_pb2.EndpointSetMeta(
        id=endpoint_set_id,
        namespace_id=namespace_id,
        version=req_pb.meta.version,
        comment=req_pb.meta.comment,
        auth=req_pb.meta.auth
    )
    meta_pb.backend_versions.append(backend_version)
    meta_pb.resolved_from.CopyFrom(selector_pb)
    meta_pb.is_system.CopyFrom(req_pb.meta.is_system)

    endpoint_set_pb = dao.create_endpoint_set(
        meta_pb=meta_pb,
        spec_pb=req_pb.spec,
        login=auth_subject.login)
    # no need to fill the statuses because the endpoint set has just been created and
    # there is very little chance it has already been processed by ctls
    return api_pb2.CreateEndpointSetResponse(endpoint_set=endpoint_set_pb)


@endpoint_set_service_bp.method('UpdateEndpointSet',
                                request_type=api_pb2.UpdateEndpointSetRequest,
                                response_type=api_pb2.UpdateEndpointSetResponse)
def update_endpoint_set(req_pb, auth_subject):
    endpoint_set.validate_request(req_pb)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    backend_version = req_pb.backend_version
    namespace_id = req_pb.meta.namespace_id
    endpoint_set_id = req_pb.meta.id

    zk = IZkStorage.instance()
    dao = IDao.instance()

    endpoint_set_pb = zk.must_get_endpoint_set(namespace_id, endpoint_set_id)
    try:
        authorize_update(endpoint_set_pb, req_pb, auth_subject)
    except ForbiddenError:
        namespace_pb = zk.must_get_namespace(namespace_id)
        authorize_update(endpoint_set_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    backend_pb = zk.must_get_backend(namespace_id=namespace_id, backend_id=endpoint_set_id)
    if backend_pb.meta.version != backend_version:
        raise rpc.exceptions.ConflictError(
            'Endpoint set can only be updated for '
            'the current backend version ("{}").'.format(backend_pb.meta.version))
    if backend_pb.spec.selector.type != backend_pb.spec.selector.MANUAL:
        raise rpc.exceptions.ConflictError(
            'Backend type is not MANUAL, endpoint set can not be updated.')

    backend_version_added = backend_version not in endpoint_set_pb.meta.backend_versions
    spec_has_changed = req_pb.HasField('spec') and endpoint_set_pb.spec != req_pb.spec
    auth_has_changed = req_pb.meta.HasField('auth') and endpoint_set_pb.meta.auth != req_pb.meta.auth
    is_global_has_changed = (req_pb.spec.HasField('is_global') and
                             endpoint_set_pb.spec.is_global != req_pb.spec.is_global)
    is_system_has_changed = (req_pb.meta.HasField('is_system') and
                             endpoint_set_pb.meta.is_system != req_pb.meta.is_system)

    if not spec_has_changed and not auth_has_changed and not backend_version_added and not is_system_has_changed:
        return api_pb2.UpdateEndpointSetResponse(endpoint_set=endpoint_set_pb)

    is_author_root = auth_subject.login in config.get_value('run.root_users', default=())
    if is_system_has_changed:
        if not is_author_root:
            raise ForbiddenError('"meta.is_system" can only be changed by roots')
        if not req_pb.meta.is_system.value:
            raise ForbiddenError('"meta.is_system" cannot be changed from True to False')
        if not backend_pb.meta.is_system.value:
            raise ForbiddenError('"meta.is_system" cannot be set if the parent backend is not system')

    if is_global_has_changed:
        if not is_author_root:
            raise ForbiddenError('"spec.is_global" can only be changed by roots')
        if not req_pb.spec.is_global.value:
            raise ForbiddenError('"spec.is_global" cannot be changed from True to False')
        if not backend_pb.spec.is_global.value:
            raise ForbiddenError('"spec.is_global" cannot be set if the parent backend is not global')

    if (endpoint_set_pb.meta.is_system.value or req_pb.meta.is_system.value) and not is_author_root:
        raise ForbiddenError('"meta.is_system": system endpoint set can only be updated by roots')

    endpoint_set_pb = dao.update_endpoint_set(
        namespace_id=namespace_id,
        endpoint_set_id=endpoint_set_id,
        comment=req_pb.meta.comment,
        login=auth_subject.login,
        version=req_pb.meta.version,
        updated_spec_pb=req_pb.spec if spec_has_changed else None,
        updated_auth_pb=req_pb.meta.auth if auth_has_changed else None,
        updated_is_system_pb=req_pb.meta.is_system if is_system_has_changed else None,
        backend_version=backend_version,
        backend_selector_pb=backend_pb.spec.selector
    )
    if not spec_has_changed and backend_version_added:
        dao.update_resolved_from_and_add_backend_version_to_endpoint_set_rev(
            namespace_id=namespace_id,
            endpoint_set_id=endpoint_set_id,
            endpoint_set_version=endpoint_set_pb.meta.version,
            backend_version_to_add=backend_version,
            resolved_from_pb=backend_pb.spec.selector
        )
    if endpoint_set_pb.spec.is_global.value:
        balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states()
    else:
        balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states(namespace_id=namespace_id)
    endpoint_set_pb = apicache.copy_per_balancer_statuses(entity_pb=endpoint_set_pb,
                                                          balancer_state_pbs=balancer_state_pbs)
    endpoint_set_pb = apicache.copy_per_balancer_l3_statuses(endpoint_set_pb)
    return api_pb2.UpdateEndpointSetResponse(endpoint_set=endpoint_set_pb)


@endpoint_set_service_bp.method('ListEndpointSets',
                                request_type=api_pb2.ListEndpointSetsRequest,
                                response_type=api_pb2.ListEndpointSetsResponse,
                                max_in_flight=5)
def list_endpoint_sets(req_pb, _):
    endpoint_set.validate_request(req_pb)
    namespace_id = req_pb.namespace_id

    if not req_pb.HasField('field_mask'):
        req_pb.field_mask.AllFieldsFromDescriptor(model_pb2.EndpointSet.DESCRIPTOR)
    c = apicache.IAwacsApiCache.instance()

    endpoint_set_pbs, total = c.list_namespace_endpoint_sets(namespace_id=namespace_id)

    return api_pb2.ListEndpointSetsResponse(endpoint_sets=endpoint_set_pbs, total=total)


@endpoint_set_service_bp.method('GetEndpointSet',
                                need_authentication=False,
                                request_type=api_pb2.GetEndpointSetRequest,
                                response_type=api_pb2.GetEndpointSetResponse)
def get_endpoint_set(req_pb, _):
    endpoint_set.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    endpoint_set_id = req_pb.id

    if req_pb.consistency == api_pb2.STRONG:
        zk = IZkStorage.instance()
        zk.sync_balancer_states(namespace_id)
        endpoint_set_pb = zk.must_get_endpoint_set(namespace_id, endpoint_set_id, sync=True)
        if endpoint_set_pb.spec.is_global.value:
            balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states()
        else:
            balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states(namespace_id=namespace_id)
        endpoint_set_pb = apicache.copy_per_balancer_statuses(entity_pb=endpoint_set_pb,
                                                              balancer_state_pbs=balancer_state_pbs)
        endpoint_set_pb = apicache.copy_per_balancer_l3_statuses(endpoint_set_pb)
    else:
        c = apicache.IAwacsApiCache.instance()
        endpoint_set_pb = c.must_get_endpoint_set(namespace_id, endpoint_set_id)
    return api_pb2.GetEndpointSetResponse(endpoint_set=endpoint_set_pb)


@endpoint_set_service_bp.method('GetEndpointSetRevision',
                                request_type=api_pb2.GetEndpointSetRevisionRequest,
                                response_type=api_pb2.GetEndpointSetRevisionResponse)
def get_endpoint_set_revision(req_pb, _):
    endpoint_set.validate_request(req_pb)
    rev_id = req_pb.id

    mongo_storage = IMongoStorage.instance()
    rev_pb = mongo_storage.must_get_endpoint_set_rev(rev_id)
    return api_pb2.GetEndpointSetRevisionResponse(revision=rev_pb)


@endpoint_set_service_bp.method('ListEndpointSetRevisions',
                                request_type=api_pb2.ListEndpointSetRevisionsRequest,
                                response_type=api_pb2.ListEndpointSetRevisionsResponse,
                                max_in_flight=5)
def list_endpoint_set_revisions(req_pb, _):
    endpoint_set.validate_request(req_pb)
    endpoint_set_id = req_pb.id
    namespace_id = req_pb.namespace_id

    zk = IZkStorage.instance()
    zk.must_get_endpoint_set(namespace_id, endpoint_set_id)

    mongo_storage = IMongoStorage.instance()
    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = mongo_storage.list_endpoint_set_revs(namespace_id, endpoint_set_id, skip=skip, limit=limit)
    return api_pb2.ListEndpointSetRevisionsResponse(revisions=revs.items, total=revs.total)
