# coding: utf-8

import flask
import inject
import six
import ujson
from six.moves import http_client as httplib
from werkzeug.exceptions import BadGateway

from awacs.lib import l3mgrclient, rpc, racktables
from awacs.model import cache
from awacs.model.l3_balancer import errors as l3errors, l3mgr
from awacs.web.auth.core import authorize_update, get_acl
from infra.awacs.proto import api_pb2
from infra.swatlib.auth.abc import IAbcClient, AbcError
from infra.swatlib.rpc.blueprint import make_authentication_request
from .data import CFG_SCHEMA


helpers_service_bp = flask.Blueprint('helpers_service', __name__, url_prefix='/api/_helpers')


def error(message):
    return make_json_response({
        "result": 'FAIL',
        "message": message,
    }), 400


def make_json_response(doc, dumps=ujson.dumps, status=httplib.OK, headers=None, response_cls=flask.Response):
    """
    Creates response object dumping provided argument using ujson and setting Content-Type.

    :type doc: dict
    :type dumps: callable
    :type status: int
    :type headers: dict|None
    :type response_cls: class
    :rtype: flask.Response
    """
    # Use Response instead of make_response to be able to call this
    # helper without proper application context
    return response_cls(dumps(doc), status=status, content_type='application/json', headers=headers)


@helpers_service_bp.route('/abc/multicomplete/', methods=('GET',))
def abc_multicomplete():
    spec = dict(flask.request.args)
    spec['format'] = 'json'  # by default multicomplete returns padded JSON, we don't want that
    try:
        resp = IAbcClient.instance().multicomplete(spec)
    except AbcError as e:
        raise BadGateway(six.text_type(e))
    return make_json_response(resp)


@helpers_service_bp.route('/abc/services/', methods=('GET',))
def abc_services():
    spec = dict(flask.request.args)
    try:
        resp = IAbcClient.instance().list_services(spec)
    except AbcError as e:
        raise BadGateway(six.text_type(e))
    return make_json_response(resp)


@helpers_service_bp.route('/abc/members/', methods=('GET',))
def abc_members():
    spec = dict(flask.request.args)
    try:
        resp = IAbcClient.instance().list_members(spec)
    except AbcError as e:
        raise BadGateway(six.text_type(e))
    return make_json_response(resp)


@helpers_service_bp.route('/cfg-schema/', methods=('GET',))
def cfg_schema():
    return make_json_response(CFG_SCHEMA)


@helpers_service_bp.route('/l3mgr/virtual_service/<service_id>', methods=('GET',))
def get_virtual_service(service_id):
    l3mgr_client = l3mgrclient.IL3MgrClient.instance()
    resp = l3mgr_client.get_service(service_id, request_timeout=3)
    ips = set()
    if resp[u'state'] != u'INACTIVE':
        for vs_id in resp[u'config'][u'vs_id']:
            vs_config = l3mgr_client.get_vs(service_id, vs_id, request_timeout=3)
            ips.add(vs_config[u'ip'])
    return make_json_response({'ips': sorted(ips), 'fqdn': resp[u'fqdn']})


def check_balancer_auth(namespace_id, l3_balancer_id, l3_balancer_pb, namespace_pb, auth_subject):
    # a hacky way to make sure that a user has the right to touch this balancer:
    fake_update_req_pb = api_pb2.UpdateL3BalancerRequest()
    fake_update_req_pb.meta.namespace_id = namespace_id
    fake_update_req_pb.meta.id = l3_balancer_id
    fake_update_req_pb.spec.CopyFrom(l3_balancer_pb.spec)
    try:
        authorize_update(l3_balancer_pb, fake_update_req_pb, auth_subject, implicit_spec_update=True)
    except rpc.exceptions.ForbiddenError:
        authorize_update(l3_balancer_pb, fake_update_req_pb, auth_subject,
                         acl=get_acl(namespace_pb), implicit_spec_update=True)


def _enable_dynamic_weights(l3mgr_client, svc_id, comment=None):
    """
    used in l3mgrctl in arcadia
    """
    l3mgr_svc = l3mgr.Service.from_api(l3mgr_client, svc_id)
    config = l3mgr.ServiceConfig.latest_from_api_if_exists(l3mgr_client, svc_id)
    if config is None:
        return error(u'No vs found in service')
    try:
        virtual_servers = l3mgr.VirtualServers.from_l3mgr_raw_virtual_servers(svc_id, l3mgr_svc.virtual_servers)
    except l3errors.VSConfigsConflict:
        return error(u'Not all virtual services have the same config, please contact st/BALANCERSUPPORT for details')

    if virtual_servers.shared_config.get(u'DYNAMICWEIGHT'):
        return error(u'DYNAMICWEIGHT is already set')
    if virtual_servers.shared_config.get(u'DYNAMICWEIGHT_ALLOW_ZERO'):
        return error(u'DYNAMICWEIGHT_ALLOW_ZERO is already set')

    updated = virtual_servers.update_shared_config(l3mgr_client,
                                                   {u'DYNAMICWEIGHT': True,
                                                    u'DYNAMICWEIGHT_ALLOW_ZERO': True})
    if updated:
        l3mgr_config = l3mgr.ServiceConfig.create_and_process(
            l3mgr_client, svc_id, config, vs_ids=virtual_servers.vs_ids, comment=comment)
        return l3mgr_config.id


@helpers_service_bp.route('/l3mgr/enable_dynamic_weights/<namespace_id>/<l3_balancer_id>/',
                          methods=('POST',))
def l3mgr_enable_dynamic_weights(namespace_id, l3_balancer_id):
    c = cache.IAwacsCache.instance()
    l3mgr_client = l3mgrclient.IL3MgrClient.instance()

    namespace_pb = c.must_get_namespace(namespace_id)
    l3_balancer_pb = c.must_get_l3_balancer(namespace_id, l3_balancer_id)

    # authenticate
    authenticator = inject.instance(rpc.authentication.IRpcAuthenticator)
    auth_subject = authenticator.authenticate_request(
        make_authentication_request(flask.request))
    check_balancer_auth(namespace_id, l3_balancer_id, l3_balancer_pb, namespace_pb, auth_subject)

    if l3_balancer_pb.spec.use_endpoint_weights:
        return error(u'spec.use_endpoint_weights is set. Enabling dynamic weights will likely change '
                     u'current real server weights and cause unexpected behaviour.')

    svc_id = l3_balancer_pb.spec.l3mgr_service_id
    new_cfg_id = _enable_dynamic_weights(l3mgr_client, svc_id,
                                         u'Dynamic weights enabled from awacs UI by {}@'.format(auth_subject.login))
    return make_json_response({u'svc_id': svc_id, u'cfg_id': new_cfg_id})


@helpers_service_bp.route('/l3mgr/virtual_service/search/<service_fqdn>', methods=('GET',))
def get_virtual_service_id_by_fqdn(service_fqdn):
    l3mgr_client = l3mgrclient.IL3MgrClient.instance()
    for service in l3mgr_client.list_services_by_fqdn(service_fqdn, request_timeout=3):
        if service[u'fqdn'] == service_fqdn:
            return make_json_response({u'id': service[u'id']})
    raise rpc.exceptions.NotFoundError(u'Service with FQDN "{}" not found'.format(service_fqdn))


@helpers_service_bp.route('/racktables/list_services/', methods=('GET',))
def list_racktables_services():
    c = racktables.IRacktablesClient.instance()
    services = c.list_services(external_only=True)
    return make_json_response(services)
