# coding: utf-8
import re
import six

import nanny_rpc_client
from its_repo import model_pb2 as its_model_pb2

from awacs.lib import rpc
from awacs.lib.gutils import gevent_idle_iter
from awacs.lib.itsrpcclient import IItsRpcClient
from awacs.lib.rpc.exceptions import ForbiddenError
from awacs.model import db, apicache
from awacs.model.apicache import IAwacsApiCache
from awacs.model.cache import IAwacsCache
from awacs.model.util import clone_pb, omit_duplicate_items_from_auth
from awacs.model.dao import IDao
from awacs.model.zk import IZkStorage
from infra.awacs.proto import api_pb2, model_pb2
from awacs.web.auth.core import authorize_update, authorize_remove, get_acl, authorize_create
from awacs.web.validation import knob
from .util import AwacsBlueprint, forbid_action_during_namespace_order, validate_namespace_total_objects_count


knob_service_bp = AwacsBlueprint('rpc_knob_service', __name__, '/api')


@knob_service_bp.method('CreateKnob',
                        request_type=api_pb2.CreateKnobRequest,
                        response_type=api_pb2.CreateKnobResponse)
def create_knob(req_pb, auth_subject):
    knob.validate_request(req_pb)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    namespace_id = req_pb.meta.namespace_id
    knob_id = req_pb.meta.id

    cache = IAwacsCache.instance()
    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id=namespace_id)
    validate_namespace_total_objects_count('knob', len(cache.list_all_knobs(namespace_id)), namespace_pb)
    authorize_create(namespace_pb, auth_subject)

    if zk.does_knob_exist(namespace_id, knob_id):
        raise rpc.exceptions.ConflictError(
            'Knob "{}" already exists in namespace "{}".'.format(knob_id, namespace_id))

    if req_pb.spec.type == req_pb.spec.ANY:
        raise rpc.exceptions.BadRequestError('"spec.type" must not be ANY')

    knob_pb = IDao.instance().create_knob(
        meta_pb=req_pb.meta,
        spec_pb=req_pb.spec,
        login=auth_subject.login)
    # no need to fill the statuses because the knob has just been created and
    # there is very little chance it has already been processed by ctls
    return api_pb2.CreateKnobResponse(knob=knob_pb)


def get_dv_id(balancer_id, knob_pb):
    """
    :type balancer_id: str
    :type knob_pb: model_pb2.Knob
    """
    assert balancer_id or knob_pb.spec.shared
    namespace_id = knob_pb.meta.namespace_id
    knob_id = knob_pb.meta.id
    return 'awacs:{}:{}:{}'.format(namespace_id, balancer_id, knob_id)


def get_knob_balancer_ids(current_namespace_id, knob_pb):
    """
    :type current_namespace_id: str
    :type knob_pb: model_pb2.Knob
    :rtype: set[str]
    """
    rv = set()
    for status_pb in knob_pb.statuses:
        for flat_full_balancer_id, cond_pb in six.iteritems(status_pb.active):
            if cond_pb.status == 'True':
                assert ':' in flat_full_balancer_id
                namespace_id, balancer_id = flat_full_balancer_id.split(':')
                assert namespace_id == current_namespace_id
                rv.add(balancer_id)
    return rv


INTEGER_RE = re.compile(r'^[-+]?([1-9]\d*|0)$')
NUMBER_RE = re.compile('^[-+]?([0-9]*\.[0-9]+|[0-9]+)$')
KEY_RE = re.compile(r'^[\w-]+$')


def _validate_comma_separated_str_number_pairs(content):
    """
    :type content: str
    :raises: ValueError
    """
    rv = []
    for line in content.splitlines():
        parts = line.split(',')
        if len(parts) != 2:
            raise ValueError('each line must contain a string (without spaces) '
                             'and a number, separated by a single comma')
        key = parts[0].strip()
        if not KEY_RE.match(key):
            raise ValueError('each line must contain a string (without spaces) '
                             'and a number, separated by a single comma')
        value = parts[1].strip()
        if not NUMBER_RE.match(value):
            raise ValueError('each line must contain a string (without spaces) '
                             'and a number, separated by a single comma')
        rv.append((key, int(value)))
    return rv


def validate_knob_value_content(knob_pb, content):
    """
    :type knob_pb: model_pb2.Knob
    :type content: str
    :raises: ValueError
    """
    spec_pb = knob_pb.spec
    if spec_pb.type == spec_pb.ANY:
        return
    elif spec_pb.type == spec_pb.BOOLEAN:
        if content:
            raise ValueError('must be empty for boolean knob')
    elif spec_pb.type == spec_pb.INTEGER:
        if not INTEGER_RE.match(content):
            raise ValueError('must contain an integer value')
    elif spec_pb.type == spec_pb.RATE:
        if not NUMBER_RE.match(content):
            raise ValueError('must contain a floating point value between 0 and 1 inclusive')
        try:
            rate = float(content)
        except ValueError:
            raise ValueError('must contain a floating point value between 0 and 1 inclusive')
        else:
            if not 0 <= rate <= 1:
                raise ValueError('must contain a floating point value between 0 and 1 inclusive')
    elif spec_pb.type == spec_pb.YB_BACKEND_WEIGHTS:
        items = _validate_comma_separated_str_number_pairs(content)
        if not items:
            raise ValueError('must not be empty')
        data = dict(items)
        if len(data) != len(items):
            raise ValueError('must not contain duplicate keys')
    elif spec_pb.type == spec_pb.YB_EXP_HALTING_POINTS:
        pass  # TODO
    elif spec_pb.type == spec_pb.YB_WATERMARK_POLICY_PARAMS:
        items = _validate_comma_separated_str_number_pairs(content)
        if not items:
            raise ValueError('must not be empty')
        data = dict(items)
        if len(data) != len(items):
            raise ValueError('must not contain duplicate keys')
        if set(six.iterkeys(data)) != {'lo', 'hi'}:
            raise ValueError('must contain "lo" and "hi" keys')
        for value in six.itervalues(data):
            if not (0 <= value <= 1):
                raise ValueError('must contain values between 0 and 1 inclusive')
    else:
        raise AssertionError('unknown knob type {}'.format(spec_pb.type))


def get_knob_filename(knob_pb):
    """
    :type knob_pb: model_pb2.Knob
    :rtype: str
    """
    if knob_pb.spec.mode == knob_pb.spec.MANAGED:
        rv = knob_pb.meta.id
    else:
        rv = knob_pb.spec.its_watched_state.filename
        if rv.startswith('./'):
            rv = rv[2:]
    return rv


def is_knob_shared(knob_pb):
    """
    :type knob_pb: model_pb2.Knob
    :rtype: bool
    """
    return knob_pb.spec.shared


def get_balancer_nanny_service_id(balancer_pb):
    """
    :type balancer_pb: model_pb2.Balancer
    :rtype: str
    """
    return balancer_pb.spec.config_transport.nanny_static_file.service_id


@knob_service_bp.method('ListKnobs',
                        request_type=api_pb2.ListKnobsRequest,
                        response_type=api_pb2.ListKnobsResponse)
def list_knobs(req_pb, _):
    """
    :type req_pb: api_pb2.ListKnobsRequest
    """
    knob.validate_request(req_pb)
    if not req_pb.HasField('field_mask'):
        req_pb.field_mask.AllFieldsFromDescriptor(model_pb2.Knob.DESCRIPTOR)

    apicache = IAwacsApiCache.instance()

    query = {}
    sort = [-1 if req_pb.sort_order == api_pb2.DESCEND else 1]

    kwargs = dict(sort=sort,
                  query=query,
                  skip=req_pb.skip or None,
                  limit=req_pb.limit or None)

    namespace_id = req_pb.namespace_id
    balancer_id = req_pb.balancer_id

    if balancer_id:
        knob_pbs, total = apicache.list_balancer_knobs(
            namespace_id=namespace_id, balancer_id=balancer_id, **kwargs)
    else:
        knob_pbs, total = apicache.list_namespace_knobs(
            namespace_id=namespace_id, **kwargs)

    resp_pb = api_pb2.ListKnobsResponse(total=total)
    for knob_pb in gevent_idle_iter(knob_pbs, idle_period=30):
        req_pb.field_mask.MergeMessage(knob_pb, resp_pb.knobs.add())
    return resp_pb


def get_dynamic_values_from_its(namespace_id):
    its_client = IItsRpcClient.instance()
    its_resp_pb = its_client.list_dynamic_values(labels={'awacs-namespace': namespace_id})
    rv = {}
    for dv_pb in its_resp_pb.dynamic_values:
        rv[dv_pb.meta.id] = dv_pb
    return rv


@knob_service_bp.method('ListAnnotatedKnobs',
                        request_type=api_pb2.ListAnnotatedKnobsRequest,
                        response_type=api_pb2.ListAnnotatedKnobsResponse,
                        max_in_flight=5)
def list_annotated_knobs(req_pb, _):
    """
    :type req_pb: api_pb2.ListAnnotatedKnobsRequest
    """
    knob.validate_request(req_pb)

    apicache = IAwacsApiCache.instance()

    resp_pb = api_pb2.ListAnnotatedKnobsResponse()

    namespace_id = req_pb.namespace_id
    dvs = get_dynamic_values_from_its(namespace_id)
    knob_pbs, total = apicache.list_namespace_knobs(namespace_id=namespace_id, query={
        apicache.KnobsQueryTarget.MODE: model_pb2.KnobSpec.MANAGED,
    })

    balancer_annotated_knob_pbs = {}  # type: dict[str, api_pb2.ListAnnotatedKnobsResponse.BalancerAnnotatedKnobs]
    for knob_pb in knob_pbs:
        if is_knob_shared(knob_pb):
            annotated_knob_pb = resp_pb.shared_knobs.add(knob=knob_pb)
            dv_id = get_dv_id('', knob_pb)
            if dv_id in dvs:
                db_pb = dvs[dv_id]
                annotated_knob_pb.value.version = db_pb.meta.version
                annotated_knob_pb.value.content = db_pb.spec.content
        else:
            for balancer_id in get_knob_balancer_ids(namespace_id, knob_pb):
                if balancer_id not in balancer_annotated_knob_pbs:
                    balancer_annotated_knob_pbs[balancer_id] = resp_pb.balancer_knobs.add(balancer_id=balancer_id)
                annotated_knob_pb = balancer_annotated_knob_pbs[balancer_id].knobs.add(knob=knob_pb)
                dv_id = get_dv_id(balancer_id, knob_pb)
                if dv_id in dvs:
                    db_pb = dvs[dv_id]
                    annotated_knob_pb.value.version = db_pb.meta.version
                    annotated_knob_pb.value.content = db_pb.spec.content

    return resp_pb


@knob_service_bp.method('UpdateKnobValue',
                        request_type=api_pb2.UpdateKnobValueRequest,
                        response_type=api_pb2.UpdateKnobValueResponse)
def update_knob_value(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateKnobValueRequest
    """
    knob.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    knob_id = req_pb.id
    login = auth_subject.login

    c = apicache.IAwacsApiCache.instance()
    knob_pb = c.must_get_knob(namespace_id, knob_id)
    try:
        authorize_update(knob_pb, req_pb, auth_subject)
    except ForbiddenError:
        namespace_pb = c.must_get_namespace(namespace_id)
        authorize_update(knob_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    if is_knob_shared(knob_pb):
        if req_pb.balancer_id:
            raise rpc.exceptions.BadRequestError('"balancer_id" must not be specified for shared knob.')
    else:
        if not req_pb.balancer_id:
            raise rpc.exceptions.BadRequestError('"balancer_id" must be specified.')

    try:
        validate_knob_value_content(knob_pb, req_pb.content)
    except ValueError as e:
        raise rpc.exceptions.BadRequestError('"content": {}'.format(e))

    dv_id = get_dv_id(req_pb.balancer_id, knob_pb)

    its_client = IItsRpcClient.instance()
    if req_pb.version:
        its_client.update_dynamic_value_content(dv_id, req_pb.version, req_pb.content, author=login)
    else:
        spec_pb = its_model_pb2.DynamicValueSpec()
        spec_pb.content = req_pb.content
        spec_pb.filename = get_knob_filename(knob_pb)
        if is_knob_shared(knob_pb):
            balancer_ids = get_knob_balancer_ids(namespace_id, knob_pb)
            nanny_service_ids = set()
            for balancer_id in balancer_ids:
                balancer_pb = c.must_get_balancer(namespace_id, balancer_id)
                nanny_service_ids.add(get_balancer_nanny_service_id(balancer_pb))
            spec_pb.selector.type = its_model_pb2.DynamicValueSelector.NANNY_SERVICES
            spec_pb.selector.nanny_service_ids.extend(sorted(nanny_service_ids))
        else:
            balancer_pb = c.must_get_balancer(namespace_id, req_pb.balancer_id)
            spec_pb.selector.type = its_model_pb2.DynamicValueSelector.NANNY_SERVICE
            spec_pb.selector.nanny_service_id = get_balancer_nanny_service_id(balancer_pb)

        try:
            its_client.create_dynamic_value(dv_id,
                                            spec_pb=spec_pb,
                                            author=login,
                                            labels={'awacs-namespace': namespace_id},
                                            comment='')
        except nanny_rpc_client.exceptions.HttpError as e:
            raise rpc.exceptions.BadRequestError(six.text_type(e))
    return api_pb2.UpdateKnobValueResponse()


@knob_service_bp.method('RemoveKnobValue',
                        request_type=api_pb2.RemoveKnobValueRequest,
                        response_type=api_pb2.RemoveKnobValueResponse,
                        is_destructive=True)
def remove_knob_value(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.RemoveKnobValueRequest
    """
    knob.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    knob_id = req_pb.id
    login = auth_subject.login

    c = apicache.IAwacsApiCache.instance()
    knob_pb = c.must_get_knob(namespace_id, knob_id)
    namespace_pb = c.must_get_namespace(namespace_id)
    try:
        authorize_update(knob_pb, req_pb, auth_subject)
    except ForbiddenError:
        authorize_update(knob_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    if is_knob_shared(knob_pb):
        if req_pb.balancer_id:
            raise rpc.exceptions.BadRequestError('"balancer_id" must not be specified for shared knob.')
    else:
        if not req_pb.balancer_id:
            raise rpc.exceptions.BadRequestError('"balancer_id" must be specified.')

    dv_id = get_dv_id(req_pb.balancer_id, knob_pb)

    its_client = IItsRpcClient.instance()
    try:
        its_client.remove_dynamic_value(dv_id, version=req_pb.version, author=login)
    except nanny_rpc_client.exceptions.HttpError as e:
        raise rpc.exceptions.BadRequestError(six.text_type(e))
    return api_pb2.RemoveKnobValueResponse()


@knob_service_bp.method('GetKnob',
                        request_type=api_pb2.GetKnobRequest,
                        response_type=api_pb2.GetKnobResponse)
def get_knob(req_pb, _):
    """
    :type req_pb: api_pb2.GetKnobRequest
    """
    knob.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    knob_id = req_pb.id

    if req_pb.consistency == api_pb2.STRONG:
        zk = IZkStorage.instance()
        zk.sync_balancer_states(namespace_id)
        knob_pb = zk.must_get_knob(namespace_id, knob_id, sync=True)
        balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states()
        knob_pb = apicache.copy_per_balancer_statuses(entity_pb=knob_pb, balancer_state_pbs=balancer_state_pbs)
    else:
        c = apicache.IAwacsApiCache.instance()
        knob_pb = c.must_get_knob(namespace_id, knob_id)
    return api_pb2.GetKnobResponse(knob=knob_pb)


@knob_service_bp.method('UpdateKnob',
                        request_type=api_pb2.UpdateKnobRequest,
                        response_type=api_pb2.UpdateKnobResponse)
def update_knob(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateKnobRequest
    """
    knob.validate_request(req_pb)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    namespace_id = req_pb.meta.namespace_id
    knob_id = req_pb.meta.id

    zk = IZkStorage.instance()

    knob_pb = zk.must_get_knob(namespace_id, knob_id)
    try:
        authorize_update(knob_pb, req_pb, auth_subject)
    except ForbiddenError:
        namespace_pb = zk.must_get_namespace(namespace_id)
        authorize_update(knob_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    if req_pb.spec.mode == req_pb.spec.WATCHED:
        its_location_paths = set(knob_pb.spec.its_watched_state.its_location_paths)
        for path in its_location_paths:
            if path.startswith('balancer/all-service'):
                raise rpc.exceptions.BadRequestError('Knob is watched from "balancer/all-service" ITS location and '
                                                     'can not be converted to MANAGED yet')
        ruchka_id = knob_pb.spec.its_watched_state.ruchka_id
        if ruchka_id == 'cplb_balancer_load_switch':
            raise rpc.exceptions.BadRequestError('Knob is watched from "cplb_balancer_load_switch" ITS ruchka and '
                                                 'can not be converted to MANAGED yet')

    elif req_pb.spec.mode == req_pb.spec.MANAGED:
        req_pb.spec.ClearField('its_watched_state')

    spec_has_changed = req_pb.HasField('spec') and knob_pb.spec != req_pb.spec
    auth_has_changed = req_pb.meta.HasField('auth') and knob_pb.meta.auth != req_pb.meta.auth
    if knob_pb.spec.mode == knob_pb.spec.MANAGED:
        if spec_has_changed:
            raise rpc.exceptions.BadRequestError('"spec" of MANAGED knobs can not be changed')

    if not spec_has_changed and not auth_has_changed:
        return api_pb2.UpdateKnobResponse(knob=knob_pb)

    if spec_has_changed:
        if req_pb.spec.type == req_pb.spec.ANY:
            raise rpc.exceptions.BadRequestError('"spec.type" must not be ANY')

    knob_pb = IDao.instance().update_knob(
        namespace_id=namespace_id,
        knob_id=knob_id,
        comment=req_pb.meta.comment,
        login=auth_subject.login,
        version=req_pb.meta.version,
        updated_spec_pb=req_pb.spec if spec_has_changed else None,
        updated_auth_pb=req_pb.meta.auth if auth_has_changed else None
    )
    balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states()
    knob_pb = apicache.copy_per_balancer_statuses(entity_pb=knob_pb, balancer_state_pbs=balancer_state_pbs)
    return api_pb2.UpdateKnobResponse(knob=knob_pb)


@knob_service_bp.method('RemoveKnob',
                        request_type=api_pb2.RemoveKnobRequest,
                        response_type=api_pb2.RemoveKnobResponse,
                        is_destructive=True)
def remove_knob(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.RemoveKnobRequest
    """
    knob.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    knob_id = req_pb.id

    zk = IZkStorage.instance()

    knob_pb = zk.must_get_knob(namespace_id, knob_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_remove(knob_pb, auth_subject)
    except ForbiddenError:
        authorize_remove(knob_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    updated_spec_pb = clone_pb(knob_pb.spec)
    updated_spec_pb.deleted = True

    IDao.instance().update_knob(
        namespace_id=namespace_id,
        knob_id=knob_id,
        version=req_pb.version,
        comment='Marked as deleted by {}'.format(auth_subject.login),
        login=auth_subject.login,
        updated_spec_pb=updated_spec_pb
    )

    return api_pb2.RemoveKnobResponse()


@knob_service_bp.method('GetKnobRevision',
                        request_type=api_pb2.GetKnobRevisionRequest,
                        response_type=api_pb2.GetKnobRevisionResponse)
def get_knob_revision(req_pb, _):
    """
    :type req_pb: api_pb2.GetKnobRevisionRequest
    """
    knob.validate_request(req_pb)
    rev_id = req_pb.id

    mongo_storage = db.IMongoStorage.instance()
    rev_pb = mongo_storage.must_get_knob_rev(rev_id)
    return api_pb2.GetKnobRevisionResponse(revision=rev_pb)


@knob_service_bp.method('ListKnobRevisions',
                        request_type=api_pb2.ListKnobRevisionsRequest,
                        response_type=api_pb2.ListKnobRevisionsResponse,
                        max_in_flight=5)
def list_knob_revisions(req_pb, _):
    """
    :type req_pb: api_pb2.ListKnobRevisionsRequest
    """
    knob.validate_request(req_pb)
    knob_id = req_pb.id
    namespace_id = req_pb.namespace_id

    zk = IZkStorage.instance()
    zk.must_get_knob(namespace_id, knob_id)

    mongo_storage = db.IMongoStorage.instance()
    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = mongo_storage.list_knob_revs(namespace_id, knob_id, skip=skip, limit=limit)
    return api_pb2.ListKnobRevisionsResponse(revisions=revs.items, total=revs.total)
