# coding: utf-8
import logging

import six

from awacs.lib import rpc, l3mgrclient
from awacs.lib.gutils import gevent_idle_iter
from awacs.lib.order_processor.model import cancel_order
from awacs.model import db, cache, apicache, dao
from awacs.model.l3_balancer import l3mgr, l3_balancer as l3b
from awacs.model.namespace.util import is_dzen
from awacs.model.util import omit_duplicate_items_from_auth, clone_pb
from awacs.model.zk import IZkStorage
from awacs.web.auth.core import authorize_update, authorize_remove, get_acl, authorize_create
from awacs.web.validation import l3_balancer
from infra.awacs.proto import api_pb2, model_pb2
from infra.swatlib.auth import abc
from infra.swatlib.rpc.exceptions import BadRequestError
from .util import AwacsBlueprint, forbid_action_during_namespace_order, validate_namespace_total_objects_count
from .validation import util


logger = logging.getLogger(__name__)
l3_balancer_service_bp = AwacsBlueprint('rpc_l3_balancer_service', __name__, '/api')


@l3_balancer_service_bp.method('CreateL3Balancer',
                               request_type=api_pb2.CreateL3BalancerRequest,
                               response_type=api_pb2.CreateL3BalancerResponse)
def create_l3_balancer(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateL3BalancerRequest
    """

    # TODO: AWACS-1009: remove backwards compat for old-style L3 balancer orders
    if req_pb.HasField('spec') and req_pb.HasField('order'):
        # an old way of setting real servers using spec, now "order" has its own "real_servers"
        logger.info(u'Deprecated: L3 balancer "%s:%s" created via "spec" + "order" by "%s"',
                    req_pb.meta.namespace_id, req_pb.meta.id, auth_subject.login)
        if req_pb.spec.HasField('real_servers') and not req_pb.order.HasField('real_servers'):
            req_pb.order.real_servers.CopyFrom(req_pb.spec.real_servers)
        if not req_pb.order.use_endpoint_weights:
            req_pb.order.use_endpoint_weights = req_pb.spec.use_endpoint_weights
        if not req_pb.order.preserve_foreign_real_servers:
            req_pb.order.preserve_foreign_real_servers = req_pb.spec.preserve_foreign_real_servers
        if not req_pb.order.ctl_version:
            req_pb.order.ctl_version = req_pb.spec.ctl_version
    elif req_pb.HasField('spec') and not req_pb.HasField('order'):
        # an old way to import existing L3 balancer from L3 manager, superseded by ImportL3Balancer
        logger.info(u'Deprecated: L3 balancer "%s:%s" imported via "CreateL3Balancer" by "%s"',
                    req_pb.meta.namespace_id, req_pb.meta.id, auth_subject.login)

    l3_balancer.validate_request(req_pb, auth_subject)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    l3_balancer_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id

    c = cache.IAwacsCache.instance()
    namespace_l3_balancers_count = len(c.list_all_l3_balancers(namespace_id))
    zk = IZkStorage.instance()
    namespace_pb = zk.must_get_namespace(namespace_id=namespace_id)
    validate_namespace_total_objects_count('l3_balancer', namespace_l3_balancers_count, namespace_pb)
    authorize_create(namespace_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    if is_dzen(namespace_pb):
        raise rpc.exceptions.ForbiddenError(
            u'Ordering L3 balancers is not allowed in dzen namespaces, please use import')

    if (namespace_pb.spec.easy_mode_settings.prohibit_explicit_l3_selector.value and
            req_pb.order.real_servers.type != req_pb.order.real_servers.BALANCERS):
        raise rpc.exceptions.ForbiddenError(
            u'Only L3 balancers with BALANCERS selector can be created in this namespace')

    if zk.does_l3_balancer_exist(namespace_id, l3_balancer_id):
        raise rpc.exceptions.ConflictError(
            u'L3 balancer "{}" already exists in namespace "{}"'.format(l3_balancer_id, namespace_id))

    l3mgr_client = l3mgrclient.IL3MgrClient.instance()
    abc_client = abc.IAbcClient.instance()
    if req_pb.HasField('spec') and not req_pb.HasField('order'):
        # TODO: deprecate and remove: AWACS-1009
        l3_balancer.validate_l3mgr_service_id_not_used_in_awacs(namespace_id, l3_balancer_id,
                                                                req_pb.spec.l3mgr_service_id)
        l3_balancer.validate_nanny_robot_has_l3_access(l3mgr_client, req_pb.spec.l3mgr_service_id)
    elif req_pb.HasField('order'):
        if req_pb.order.abc_service_id:
            field_name = u'order.content.abc_service_id'
            abc_service_id = req_pb.order.abc_service_id
            util.validate_user_belongs_to_abc_service(auth_subject.login, abc_service_id, field_name)
            l3_balancer.validate_nanny_robot_is_responsible_for_l3(abc_service_id, field_name)
            l3_balancer.validate_abc_service_has_load_balancers_quota(l3mgr_client,
                                                                      abc_client,
                                                                      abc_service_id,
                                                                      field_name)
        else:
            # TODO: deprecate and remove: AWACS-1009
            # this is already prohibited for ctl_version >= 2, but we still support it for v1
            logger.info(u'Deprecated: L3 balancer "%s:%s" created without "order.abc_service_id" by "%s"',
                        req_pb.meta.namespace_id, req_pb.meta.id, auth_subject.login)

        for service in l3mgr_client.list_services_by_fqdn(fqdn=req_pb.order.fqdn, full=True, request_timeout=3):
            if service[u'fqdn'] == req_pb.order.fqdn:
                raise rpc.exceptions.ConflictError(
                    u'L3 balancer with FQDN "{}" already exists in L3mgr. You can use "Add existing L3 Balancer" '
                    u'instead of creating it.'.format(req_pb.order.fqdn))

    l3_balancer_pb, l3_balancer_state_pb = dao.IDao.instance().create_l3_balancer(
        meta_pb=req_pb.meta,
        spec_pb=req_pb.spec,
        order_content_pb=req_pb.order if req_pb.HasField('order') else None,
        login=auth_subject.login)

    apicache.copy_l3_statuses(l3_balancer_pb, l3_balancer_state_pb)
    return api_pb2.CreateL3BalancerResponse(l3_balancer=l3_balancer_pb)


@l3_balancer_service_bp.method('ImportL3Balancer',
                               request_type=api_pb2.ImportL3BalancerRequest,
                               response_type=api_pb2.ImportL3BalancerResponse)
def import_l3_balancer(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ImportL3BalancerRequest
    """
    l3_balancer.validate_request(req_pb, auth_subject)
    omit_duplicate_items_from_auth(req_pb.meta.auth)

    l3_balancer_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id
    zk = IZkStorage.instance()
    namespace_pb = zk.must_get_namespace(namespace_id=namespace_id)
    validate_namespace_total_objects_count('l3_balancer',
                                           len(cache.IAwacsCache.instance().list_all_l3_balancers(namespace_id)),
                                           namespace_pb)
    authorize_create(namespace_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    if (namespace_pb.spec.easy_mode_settings.prohibit_explicit_l3_selector.value and
            req_pb.spec.real_servers.type != req_pb.spec.real_servers.BALANCERS):
        raise rpc.exceptions.ForbiddenError(
            u'Only L3 balancers with BALANCERS selector can be created in this namespace')

    if zk.does_l3_balancer_exist(namespace_id, l3_balancer_id):
        raise rpc.exceptions.ConflictError(
            u'L3 balancer "{}" already exists in namespace "{}"'.format(l3_balancer_id, namespace_id))

    l3mgr_client = l3mgrclient.IL3MgrClient.instance()
    l3mgr_service_id = req_pb.spec.l3mgr_service_id
    l3_balancer.validate_l3mgr_service_id_not_used_in_awacs(namespace_id, l3_balancer_id, l3mgr_service_id)
    l3_balancer.validate_nanny_robot_has_l3_access(l3mgr_client, l3mgr_service_id)

    spec_pb = clone_pb(req_pb.spec)

    # check RS consistency before importing, to make sure that transport can handle it
    l3mgr_service = l3mgr.Service.from_api(l3mgr_client, l3mgr_service_id)
    l3mgr.RSGroup.from_l3mgr_virtual_servers(l3mgr_service.virtual_servers)

    spec_pb.ClearField('skip_tests')
    if is_dzen(namespace_pb):
        expected_abc_service_id = 'vk_dzen_l3_management'
        if l3mgr_service.abc_slug != expected_abc_service_id:
            raise rpc.exceptions.ForbiddenError(
                u'Only L3 balancers with "{}" ABC service '
                u'can be created in dzen awacs namespace'.format(expected_abc_service_id))
        spec_pb.skip_tests.value = True

    l3_mgr_metadata = l3b.make_l3mgr_balancer_meta(namespace_id, l3_balancer_id)
    l3mgr_client.update_meta(svc_id=req_pb.spec.l3mgr_service_id, data=l3_mgr_metadata)
    l3_balancer_pb, l3_balancer_state_pb = dao.IDao.instance().create_l3_balancer(
        meta_pb=req_pb.meta,
        spec_pb=spec_pb,
        login=auth_subject.login)

    apicache.copy_l3_statuses(l3_balancer_pb, l3_balancer_state_pb)
    return api_pb2.ImportL3BalancerResponse(l3_balancer=l3_balancer_pb)


@l3_balancer_service_bp.method('ListL3Balancers',
                               request_type=api_pb2.ListL3BalancersRequest,
                               response_type=api_pb2.ListL3BalancersResponse,
                               max_in_flight=5)
def list_l3_balancers(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListL3BalancersRequest
    """
    l3_balancer.validate_request(req_pb, auth_subject)
    if not req_pb.HasField('field_mask'):
        req_pb.field_mask.AllFieldsFromDescriptor(model_pb2.L3Balancer.DESCRIPTOR)
    c = apicache.IAwacsApiCache.instance()
    l3_balancer_pbs, total = c.list_l3_balancers(namespace_id=req_pb.namespace_id,
                                                 skip=req_pb.skip or None,
                                                 limit=req_pb.limit or None)
    resp_pb = api_pb2.ListL3BalancersResponse(total=total)
    for l3_balancer_pb in gevent_idle_iter(l3_balancer_pbs, idle_period=30):
        req_pb.field_mask.MergeMessage(l3_balancer_pb, resp_pb.l3_balancers.add())
    return resp_pb


@l3_balancer_service_bp.method('GetL3Balancer',
                               request_type=api_pb2.GetL3BalancerRequest,
                               response_type=api_pb2.GetL3BalancerResponse)
def get_l3_balancer(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetL3BalancerRequest
    """
    l3_balancer.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    l3_balancer_id = req_pb.id

    if req_pb.consistency == api_pb2.STRONG:
        zk = IZkStorage.instance()
        l3_balancer_pb = zk.must_get_l3_balancer(namespace_id, l3_balancer_id, sync=True)
        l3_balancer_state_pb = zk.must_get_l3_balancer_state(namespace_id, l3_balancer_id, sync=True)
        l3_balancer_pb = apicache.copy_l3_statuses(l3_balancer_pb, l3_balancer_state_pb)
    else:
        c = apicache.IAwacsApiCache.instance()
        l3_balancer_pb = c.must_get_l3_balancer(namespace_id=namespace_id, l3_balancer_id=l3_balancer_id)
    return api_pb2.GetL3BalancerResponse(l3_balancer=l3_balancer_pb)


@l3_balancer_service_bp.method('GetL3BalancerState',
                               request_type=api_pb2.GetL3BalancerStateRequest,
                               response_type=api_pb2.GetL3BalancerStateResponse)
def get_l3_balancer_state(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetL3BalancerStateRequest
    """
    l3_balancer.validate_request(req_pb, auth_subject)
    l3_balancer_id = req_pb.id
    namespace_id = req_pb.namespace_id

    zk = IZkStorage.instance()
    l3_balancer_state_pb = zk.must_get_l3_balancer_state(namespace_id, l3_balancer_id)
    return api_pb2.GetL3BalancerStateResponse(state=l3_balancer_state_pb)


@l3_balancer_service_bp.method('UpdateL3Balancer',
                               request_type=api_pb2.UpdateL3BalancerRequest,
                               response_type=api_pb2.UpdateL3BalancerResponse)
def update_l3_balancer(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateL3BalancerRequest
    """
    l3_balancer.validate_request(req_pb, auth_subject)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    l3_balancer_id = req_pb.meta.id
    namespace_id = req_pb.meta.namespace_id

    zk = IZkStorage.instance()

    l3_balancer_pb = zk.must_get_l3_balancer(namespace_id, l3_balancer_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_update(l3_balancer_pb, req_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_update(l3_balancer_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    spec_has_changed = req_pb.HasField('spec') and l3_balancer_pb.spec != req_pb.spec
    auth_has_changed = req_pb.meta.HasField('auth') and l3_balancer_pb.meta.auth != req_pb.meta.auth
    transport_paused_has_changed = (req_pb.meta.HasField('transport_paused') and
                                    l3_balancer_pb.meta.transport_paused.value != req_pb.meta.transport_paused.value)

    if not any((spec_has_changed, auth_has_changed, transport_paused_has_changed)):
        return api_pb2.UpdateL3BalancerResponse(l3_balancer=l3_balancer_pb)

    if spec_has_changed:
        if (namespace_pb.spec.easy_mode_settings.prohibit_explicit_l3_selector.value and
                req_pb.spec.real_servers.type != l3_balancer_pb.spec.real_servers.type and
                req_pb.spec.real_servers.type != req_pb.spec.real_servers.BALANCERS):
            raise rpc.exceptions.ForbiddenError(
                'Only L3 balancers with BALANCERS selector are allowed in this namespace')
        l3_balancer.validate_preserve_foreign_rs(l3_balancer_pb.spec.preserve_foreign_real_servers,
                                                 req_pb.spec.preserve_foreign_real_servers,
                                                 auth_subject.login, 'spec')
        if l3_balancer_pb.spec.l3mgr_service_id != req_pb.spec.l3mgr_service_id:
            raise BadRequestError('"spec.l3mgr_service_id": cannot change for existing balancer')
        l3_balancer.validate_update_l3_with_dns_records(namespace_id, l3_balancer_id, req_pb.spec, l3_balancer_pb.spec)
        if is_dzen(namespace_pb):
            if l3_balancer_pb.spec.skip_tests != req_pb.spec.skip_tests:
                raise BadRequestError('"spec.skip_tests": cannot be changed for the time being')

    l3_balancer_pb = dao.IDao.instance().update_l3_balancer(
        namespace_id=namespace_id,
        l3_balancer_id=l3_balancer_id,
        version=req_pb.meta.version,
        comment=req_pb.meta.comment,
        login=auth_subject.login,
        updated_spec_pb=req_pb.spec if spec_has_changed else None,
        updated_auth_pb=req_pb.meta.auth if auth_has_changed else None,
        updated_transport_paused_pb=req_pb.meta.transport_paused if transport_paused_has_changed else None
    )

    l3_balancer_state_pb = cache.IAwacsCache.instance().get_l3_balancer_state_or_empty(
        namespace_id=namespace_id, l3_balancer_id=l3_balancer_id)
    apicache.copy_l3_statuses(l3_balancer_pb, l3_balancer_state_pb)

    return api_pb2.UpdateL3BalancerResponse(l3_balancer=l3_balancer_pb)


@l3_balancer_service_bp.method('CheckL3BalancerMigration',
                               request_type=api_pb2.CheckL3BalancerMigrationRequest,
                               response_type=api_pb2.CheckL3BalancerMigrationResponse,
                               max_in_flight=5)
def check_l3_balancer_migration(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CheckL3BalancerMigrationRequest
    """
    l3_balancer.validate_request(req_pb, auth_subject)
    l3mgr_client = l3mgrclient.IL3MgrClient.instance()
    conflict_pbs = check_config_conflicts(l3mgr_client, req_pb.l3mgr_service_id)
    return api_pb2.CheckL3BalancerMigrationResponse(vs_config_conflicts=conflict_pbs)


def check_config_conflicts(l3mgr_client, svc_id):
    latest_config = l3mgr.ServiceConfig.latest_from_api_if_exists(l3mgr_client, svc_id)
    if latest_config is None or not latest_config.vs_ids:
        raise BadRequestError(u'service in L3 Manager does not have any VS configs')
    conflict_pbs = []
    for vs_id in latest_config.vs_ids:
        raw_data = l3mgr_client.get_vs(svc_id, vs_id)
        vs = l3mgr.VirtualServer.from_raw_data(svc_id, raw_data)
        conflict_pb = api_pb2.CheckL3BalancerMigrationResponse.ConfigValuesConflict(
            vs_id=six.text_type(vs.id),
            ip=six.text_type(vs.ip),
            port=six.text_type(vs.port))
        for key, value in six.iteritems(raw_data[u'config']):
            if key == l3mgr.VSConfigKey.check_url:
                continue  # we support any values here

            is_known_option = False
            is_expected_value = False
            for known_options in (vs.default_config, vs.ignored_config_values):  # py2 has no collections.ChainMap :(
                if key in known_options:
                    is_known_option = True
                    expected_value = known_options[key]
                    if value in (expected_value, six.text_type(expected_value)):
                        is_expected_value = True
                        break
            if not is_known_option and not value:
                continue  # we don't know about this option, but it's falsy anyway, assume it's a L3mgr default
            if is_expected_value:
                continue  # we know about this option and it matches our defaults

            # these options depend on some conditions, check them
            if key == u'CONNECT_IP':
                if value is not None and value != vs.ip:
                    diff = conflict_pb.diff[key]
                    diff.expected = vs.ip
                    diff.actual = six.text_type(value)
                    diff.hint = u'Health checks should be directed to the IP of this virtual server.'
            elif key == l3mgr.VSConfigKey.connect_port:
                if value is not None and six.text_type(value) != six.text_type(vs.port):
                    diff = conflict_pb.diff[key]
                    diff.expected = six.text_type(vs.port)
                    diff.actual = six.text_type(value)
                    diff.hint = u'Health checks should be directed to the port of this virtual server.'
            elif key == l3mgr.VSConfigKey.check_type:
                hint = None
                if value not in (u'HTTP_GET', u'SSL_GET'):
                    hint = (u'Awacs supports only two types of checks: '
                            u'SSL_GET for port 443, and HTTP_GET for other ports.')
                elif value == u'HTTP_GET' and vs.port == 443:
                    hint = u'Health check for port 443 must check SSL connection, not plain HTTP.'
                elif value == u'SSL_GET' and vs.port != 443:
                    hint = u'Health check for ports except 443 must check plain HTTP connection, not SSL.'

                if hint:
                    diff = conflict_pb.diff[key]
                    diff.expected = u'SSL_GET' if vs.port == 443 else u'HTTP_GET'
                    diff.actual = six.text_type(value)
                    diff.hint = hint
            else:
                # remaining options are non-empty
                diff = conflict_pb.diff[key]
                diff.actual = six.text_type(value)
                diff.expected = six.text_type(vs.get_default_config_value(key) or vs.get_ignored_config_value(key))
                # provide some custom hints
                if key in (l3mgr.VSConfigKey.dynamicweight,
                           l3mgr.VSConfigKey.dynamicweight_allow_zero,
                           l3mgr.VSConfigKey.dynamicweight_ratio):
                    diff.hint = u'Reduces connection errors when L7 balancer is restarted.'
                elif key == l3mgr.VSConfigKey.announce:
                    diff.hint = (u"Health checks for this virtual server are currently ignored, "
                                 u"this is not recommended. Read more: https://clubs.at.yandex-team.ru/traffic/954")
                elif key == l3mgr.VSConfigKey.mh_fallback:
                    diff.hint = u"Used for future compatibility with Maglev balancing, safe to change"
        if conflict_pb.diff:
            conflict_pbs.append(conflict_pb)
    return conflict_pbs


@l3_balancer_service_bp.method('UpdateL3BalancerState',
                               request_type=api_pb2.UpdateL3BalancerStateRequest,
                               response_type=api_pb2.UpdateL3BalancerStateResponse)
def update_l3_balancer_state(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateL3BalancerStateRequest
    """
    l3_balancer.validate_request(req_pb, auth_subject)
    l3_balancer_id = req_pb.id
    namespace_id = req_pb.namespace_id

    zk = IZkStorage.instance()

    l3_balancer_pb = zk.must_get_l3_balancer(namespace_id, l3_balancer_id)
    try:
        authorize_update(l3_balancer_pb, req_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        namespace_pb = zk.must_get_namespace(namespace_id)
        authorize_update(l3_balancer_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    if req_pb.type == req_pb.SKIP_IN_PROGRESS_L3MGR_CONFIG:
        state_pb = zk.must_get_l3_balancer_state(namespace_id, l3_balancer_id)
        h = l3b.L3BalancerStateHandler(state_pb)
        h.reset_in_progress_vector(vector=h.generate_vectors().in_progress,
                                   author=auth_subject.login,
                                   comment=u'Manually forcing creation of fresh config in L3Mgr')
        return api_pb2.UpdateL3BalancerStateResponse(state=state_pb)
    else:
        raise AssertionError('unknown UpdateL3BalancerStateRequest.type: {}'.format(req_pb.type))


@l3_balancer_service_bp.method('RemoveL3Balancer',
                               request_type=api_pb2.RemoveL3BalancerRequest,
                               response_type=api_pb2.RemoveL3BalancerResponse,
                               is_destructive=True)
def remove_l3_balancer(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.RemoveL3BalancerRequest
    """
    l3_balancer.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    l3_balancer_id = req_pb.id

    zk = IZkStorage.instance()

    l3_balancer_pb = zk.must_get_l3_balancer(namespace_id, l3_balancer_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_remove(l3_balancer_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_remove(l3_balancer_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    l3_balancer.validate_remove_l3_balancer(l3_balancer_pb)

    dao.IDao.instance().delete_l3_balancer(namespace_id=namespace_id, l3_balancer_id=l3_balancer_id)
    return api_pb2.RemoveL3BalancerResponse()


@l3_balancer_service_bp.method('GetL3BalancerRevision',
                               request_type=api_pb2.GetL3BalancerRevisionRequest,
                               response_type=api_pb2.GetL3BalancerRevisionResponse)
def get_l3_balancer_revision(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetL3BalancerRevisionRequest
    """
    l3_balancer.validate_request(req_pb, auth_subject)
    rev_id = req_pb.id

    rev_pb = db.IMongoStorage.instance().must_get_l3_balancer_rev(rev_id)
    return api_pb2.GetL3BalancerRevisionResponse(revision=rev_pb)


@l3_balancer_service_bp.method('ListL3BalancerRevisions',
                               request_type=api_pb2.ListL3BalancerRevisionsRequest,
                               response_type=api_pb2.ListL3BalancerRevisionsResponse,
                               max_in_flight=5)
def list_l3_balancer_revisions(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ListL3BalancerRevisionsRequest
    """
    l3_balancer.validate_request(req_pb, auth_subject)
    l3_balancer_id = req_pb.id
    namespace_id = req_pb.namespace_id

    zk = IZkStorage.instance()
    zk.must_get_l3_balancer(namespace_id, l3_balancer_id)

    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = db.IMongoStorage.instance().list_l3_balancer_revs(
        namespace_id=namespace_id, l3_balancer_id=l3_balancer_id, skip=skip, limit=limit)
    return api_pb2.ListL3BalancerRevisionsResponse(revisions=revs.items, total=revs.total)


@l3_balancer_service_bp.method('CancelL3BalancerOrder',
                               request_type=api_pb2.CancelL3BalancerOrderRequest,
                               response_type=api_pb2.CancelL3BalancerOrderResponse,
                               is_destructive=True)
def cancel_l3_balancer_order(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelL3BalancerOrderRequest
    """
    l3_balancer.validate_request(req_pb, auth_subject)
    namespace_id = req_pb.namespace_id
    l3_balancer_id = req_pb.id

    zk = IZkStorage.instance()

    l3_balancer_pb = zk.must_get_l3_balancer(namespace_id, l3_balancer_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_remove(l3_balancer_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_remove(l3_balancer_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    l3_balancer.validate_cancel_l3_balancer_order(req_pb, l3_balancer_pb, auth_subject)

    for l3_balancer_pb in zk.update_l3_balancer(namespace_id=namespace_id,
                                                l3_balancer_id=l3_balancer_id,
                                                l3_balancer_pb=l3_balancer_pb):
        comment = 'Cancelled order'
        if req_pb.force:
            comment = 'Force cancelled order'
        cancel_order(l3_balancer_pb, author=auth_subject.login, comment=comment, forced=req_pb.force)

    return api_pb2.CancelL3BalancerOrderResponse()
