# coding: utf-8
from awacs.lib import rpc
from awacs.lib.order_processor.model import cancel_order, is_order_in_progress
from awacs.model import objects, apicache, util
from awacs.model.zk import IZkStorage
from infra.awacs.proto import api_pb2
from awacs.web.auth.core import authorize_update, get_acl, authorize_create, authorize_remove
from awacs.web.validation import l7heavy_config
from .util import AwacsBlueprint, forbid_action_during_namespace_order

l7heavy_config_service_bp = AwacsBlueprint('rpc_l7heavy_config_service', __name__, '/api')


@l7heavy_config_service_bp.method('CreateL7HeavyConfig',
                                   request_type=api_pb2.CreateL7HeavyConfigRequest,
                                   response_type=api_pb2.CreateL7HeavyConfigResponse)
def create_l7heavy_config(req_pb, auth_subject):
    l7heavy_config.validate_request(req_pb)
    ns_id = req_pb.meta.namespace_id
    l7hc_id = req_pb.meta.id

    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id=ns_id)
    authorize_create(namespace_pb, auth_subject)
    if objects.L7HeavyConfig.zk.get(ns_id, l7hc_id) is not None:
        raise rpc.exceptions.ConflictError(
            'L7HeavyConfig "{}" already exists in namespace "{}".'.format(l7hc_id, ns_id))

    if (req_pb.order.group_id == util.COMMON_L7HEAVY_GROUP_ID and
            len(objects.WeightSection.cache.list(namespace_id=ns_id)) == 0 and not req_pb.order.weight_sections_to_create):
        raise rpc.exceptions.BadRequestError('L7Heavy config can not be created in group "common": no weight sections '
                                             'exist in namespace. Please, create weight sections first or choose '
                                             'another group - you will be allowed to change it later.')

    l7hc_pb, l7hc_state_pb = objects.L7HeavyConfig.create(req_pb.meta, req_pb.order, auth_subject.login)
    return api_pb2.CreateL7HeavyConfigResponse(l7heavy_config=l7hc_pb)


@l7heavy_config_service_bp.method('UpdateL7HeavyConfig',
                                   request_type=api_pb2.UpdateL7HeavyConfigRequest,
                                   response_type=api_pb2.UpdateL7HeavyConfigResponse)
def update_l7heavy_config(req_pb, auth_subject):
    l7heavy_config.validate_request(req_pb)
    ns_id = req_pb.meta.namespace_id
    l7hc_id = req_pb.meta.id

    zk = IZkStorage.instance()

    l7hc_pb = objects.L7HeavyConfig.zk.must_get(ns_id, l7hc_id)
    namespace_pb = zk.must_get_namespace(ns_id)
    authorize_update(l7hc_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))
    if is_order_in_progress(l7hc_pb):
        raise rpc.exceptions.BadRequestError('Can not update L7Heavy config while order is in progress.')

    spec_has_changed = req_pb.HasField('spec') and l7hc_pb.spec != req_pb.spec
    transport_paused_has_changed = (req_pb.meta.HasField('transport_paused') and
                                    l7hc_pb.meta.transport_paused.value != req_pb.meta.transport_paused.value)

    if spec_has_changed and l7hc_pb.spec.l7_heavy_config_id != req_pb.spec.l7_heavy_config_id:
        raise rpc.exceptions.BadRequestError('"spec.l7_heavy_config_id": can not be cnahged')

    if not any((spec_has_changed, transport_paused_has_changed)):
        return api_pb2.UpdateL7HeavyConfigResponse(l7heavy_config=l7hc_pb)

    l7hc_pb = objects.L7HeavyConfig.update(
        ns_id, l7hc_id, req_pb.meta.version, req_pb.meta.comment, auth_subject.login,
        updated_spec_pb=req_pb.spec if spec_has_changed else None,
        updated_transport_paused_pb=req_pb.meta.transport_paused if transport_paused_has_changed else None
    )

    return api_pb2.UpdateL7HeavyConfigResponse(l7heavy_config=l7hc_pb)


@l7heavy_config_service_bp.method('RemoveL7HeavyConfig',
                               request_type=api_pb2.RemoveL7HeavyConfigRequest,
                               response_type=api_pb2.RemoveL7HeavyConfigResponse,
                               is_destructive=True)
def remove_l7heavy_config(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.RemoveL7HeavyConfigRequest
    """
    l7heavy_config.validate_request(req_pb)
    ns_id = req_pb.namespace_id
    l7hc_id = req_pb.id

    zk = IZkStorage.instance()

    l7hc_pb = objects.L7HeavyConfig.zk.must_get(ns_id, l7hc_id)
    namespace_pb = zk.must_get_namespace(ns_id)
    authorize_remove(l7hc_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    l7heavy_config.validate_remove_l7heavy_config(l7hc_pb)

    l7hc_pb.spec.state = req_pb.state
    objects.L7HeavyConfig.update(
        ns_id, l7hc_id, l7hc_pb.meta.version, 'Marked as deleted by {}'.format(auth_subject.login), auth_subject.login,
        updated_spec_pb=l7hc_pb.spec
    )
    return api_pb2.RemoveL7HeavyConfigResponse()


@l7heavy_config_service_bp.method('GetL7HeavyConfig',
                                   request_type=api_pb2.GetL7HeavyConfigRequest,
                                   response_type=api_pb2.GetL7HeavyConfigResponse)
def get_l7heavy_config(req_pb, _):
    l7heavy_config.validate_request(req_pb)
    ns_id = req_pb.namespace_id
    l7hc_id = req_pb.id

    if req_pb.consistency == api_pb2.STRONG:
        l7hc_pb = objects.L7HeavyConfig.zk.must_get(ns_id, l7hc_id, sync=True)
        l7hc_state_pb = objects.L7HeavyConfig.state.zk.must_get(ns_id, l7hc_id, sync=True)
    else:
        l7hc_pb = objects.L7HeavyConfig.cache.must_get(ns_id, l7hc_id)
        l7hc_state_pb = objects.L7HeavyConfig.state.cache.must_get(ns_id, l7hc_id)
    l7hc_pb = apicache.copy_l7heavy_config_statuses(l7hc_pb, l7hc_state_pb)
    return api_pb2.GetL7HeavyConfigResponse(l7heavy_config=l7hc_pb)


@l7heavy_config_service_bp.method('CancelL7HeavyConfigOrder',
                               request_type=api_pb2.CancelL7HeavyConfigOrderRequest,
                               response_type=api_pb2.CancelL7HeavyConfigOrderResponse,
                               is_destructive=True)
def cancel_l7heavy_config_order(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelL7HeavyConfigOrderRequest
    """
    l7heavy_config.validate_request(req_pb)
    ns_id = req_pb.namespace_id
    l7hc_id = req_pb.id

    zk = IZkStorage.instance()

    l7hc_pb = objects.L7HeavyConfig.zk.must_get(ns_id, l7hc_id)
    namespace_pb = zk.must_get_namespace(ns_id)
    authorize_remove(l7hc_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)
    l7heavy_config.validate_cancel_l7heavy_config_order(req_pb, l7hc_pb, auth_subject)

    for l7hc_pb in objects.L7HeavyConfig.zk.update(ns_id, l7hc_id, l7hc_pb):
        comment = 'Cancelled order'
        if req_pb.force_cancel:
            comment = 'Force cancelled order'
        cancel_order(l7hc_pb, author=auth_subject.login, comment=comment, forced=req_pb.force_cancel)

    return api_pb2.CancelL7HeavyConfigOrderResponse()


@l7heavy_config_service_bp.method('GetL7HeavyConfigRevision',
                                   request_type=api_pb2.GetL7HeavyConfigRevisionRequest,
                                   response_type=api_pb2.GetL7HeavyConfigRevisionResponse)
def get_l7heavy_config_revision(req_pb, _):
    l7heavy_config.validate_request(req_pb)
    rev_pb = objects.L7HeavyConfig.mongo.must_get_rev(req_pb.id)
    return api_pb2.GetL7HeavyConfigRevisionResponse(revision=rev_pb)


@l7heavy_config_service_bp.method('ListL7HeavyConfigRevisions',
                                   request_type=api_pb2.ListL7HeavyConfigRevisionsRequest,
                                   response_type=api_pb2.ListL7HeavyConfigRevisionsResponse,
                                   max_in_flight=5)
def list_l7heavy_config_revisions(req_pb, _):
    l7heavy_config.validate_request(req_pb)
    l7hc_id = req_pb.id
    ns_id = req_pb.namespace_id

    objects.L7HeavyConfig.zk.must_get(ns_id, l7hc_id)

    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = objects.L7HeavyConfig.mongo.list_revs(ns_id, l7hc_id, skip=skip, limit=limit)
    return api_pb2.ListL7HeavyConfigRevisionsResponse(revisions=revs.items, total=revs.total)


@l7heavy_config_service_bp.method('GetL7HeavyConfigState',
                            request_type=api_pb2.GetL7HeavyConfigStateRequest,
                            response_type=api_pb2.GetL7HeavyConfigStateResponse)
def get_l7heavy_config_state(req_pb, _):
    l7heavy_config.validate_request(req_pb)
    l7hc_id = req_pb.id
    ns_id = req_pb.namespace_id

    l7hc_state_pb = objects.L7HeavyConfig.state.zk.must_get(ns_id, l7hc_id)
    return api_pb2.GetL7HeavyConfigStateResponse(state=l7hc_state_pb)
