# coding: utf-8
import uuid

import flask
import six
from itertools import chain
from sepelib.core import config

from awacs.lib import rpc
from awacs.lib.gutils import gevent_idle_iter
from awacs.lib.order_processor.model import cancel_order
from awacs.lib.rpc import exceptions
from awacs.model import alerting, cache, db, apicache, objects
from awacs.model.balancer.state_handler import L7BalancerStateHandler
from awacs.model.balancer.vector import KnobVersion
from awacs.model.dao import IDao
from awacs.model.db import find_knob_revision_spec
from awacs.model.uiaspectsutil import fill_ui_namespace_aspects
from awacs.model.util import clone_pb, omit_duplicate_items_from_auth, WEIGHTS_KNOB_ID
from awacs.model.zk import IZkStorage
from awacs.web.auth import authorize_update, authorize_remove, authorize_root_operation
from awacs.web.util import AwacsBlueprint, forbid_action_during_namespace_order
from awacs.wrappers import rps_limiter_settings
from awacs.wrappers.errors import ValidationError
from infra.awacs.proto import api_pb2, model_pb2
from .auth.core import is_root
from .validation import namespace, proto_readonly, util


namespace_service_bp = AwacsBlueprint('rpc_namespace_service', __name__, '/api')


@namespace_service_bp.method('ListNamespaces',
                             request_type=api_pb2.ListNamespacesRequest,
                             response_type=api_pb2.ListNamespacesResponse,
                             max_in_flight=5)
def list_namespaces(req_pb, _):
    """
    :type req_pb: api_pb2.ListNamespacesRequest
    :rtype: api_pb2.ListNamespacesResponse
    """
    namespace.validate_request(req_pb)
    if not req_pb.HasField('field_mask'):
        req_pb.field_mask.AllFieldsFromDescriptor(model_pb2.Namespace.DESCRIPTOR)
    c = cache.IAwacsCache.instance()

    query = {}
    if req_pb.query.id_in:
        query[c.NamespacesQueryTarget.ID_IN] = req_pb.query.id_in
    if req_pb.query.category_in:
        query[c.NamespacesQueryTarget.CATEGORY_IN] = req_pb.query.category_in
    if req_pb.query.abc_service_id_in:
        query[c.NamespacesQueryTarget.ABC_SERVICE_ID_IN] = req_pb.query.abc_service_id_in
    if req_pb.query.layout_type_in:
        query[c.NamespacesQueryTarget.LAYOUT_TYPE_IN] = req_pb.query.layout_type_in

    namespace_pbs, total = c.list_namespaces(skip=req_pb.skip or None,
                                             limit=req_pb.limit or None,
                                             query=query)

    resp_pb = api_pb2.ListNamespacesResponse(total=total)
    for namespace_pb in gevent_idle_iter(namespace_pbs, idle_period=50):
        req_pb.field_mask.MergeMessage(namespace_pb, resp_pb.namespaces.add())
    return resp_pb


@namespace_service_bp.method('ListNamespaceSummaries',
                             request_type=api_pb2.ListNamespaceSummariesRequest,
                             response_type=api_pb2.ListNamespaceSummariesResponse,
                             max_in_flight=5)
def list_namespace_summaries(req_pb, _):
    """
    :type req_pb: api_pb2.ListNamespaceSummariesRequest
    :rtype: api_pb2.ListNamespaceSummariesResponse
    """
    namespace.validate_request(req_pb)
    c = cache.IAwacsCache.instance()

    query = {}
    if req_pb.query.layout_type_in:
        query[c.NamespacesQueryTarget.LAYOUT_TYPE_IN] = req_pb.query.layout_type_in

    resp_pb = api_pb2.ListNamespaceSummariesResponse()
    for namespace_pb in gevent_idle_iter(c.list_all_namespaces(query=query), idle_period=1000):
        resp_pb.summaries.add(
            id=namespace_pb.meta.id,
            category=namespace_pb.meta.category)
    return resp_pb


@namespace_service_bp.method('GetNamespace',
                             request_type=api_pb2.GetNamespaceRequest,
                             response_type=api_pb2.GetNamespaceResponse)
def get_namespace(req_pb, _):
    namespace.validate_request(req_pb)
    namespace_id = req_pb.id
    if req_pb.consistency == api_pb2.STRONG:
        zk = IZkStorage.instance()
        namespace_pb = zk.must_get_namespace(namespace_id, sync=True)
    else:
        c = cache.IAwacsCache.instance()
        namespace_pb = c.must_get_namespace(namespace_id=namespace_id)
    return api_pb2.GetNamespaceResponse(namespace=namespace_pb)


@namespace_service_bp.method('CreateNamespace',
                             request_type=api_pb2.CreateNamespaceRequest,
                             response_type=api_pb2.CreateNamespaceResponse)
def create_namespace(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateNamespaceRequest
    :rtype: api_pb2.CreateNamespaceResponse
    """
    dry_run = flask.request.headers.get('X-Deploy-UI-Dry-Run') == '1'
    namespace.validate_request(req_pb)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    login = auth_subject.login

    namespace_id = req_pb.meta.id

    zk = IZkStorage.instance()
    if zk.does_namespace_exist(namespace_id):
        raise rpc.exceptions.ConflictError(u'Namespace "{}" already exists'.format(namespace_id))

    _cache = cache.IAwacsCache.instance()
    if _cache.does_namespace_normalised_name_exist(namespace_id):
        raise rpc.exceptions.ConflictError(u'Namespace "{}" already exists'.format(namespace_id))

    util.validate_user_belongs_to_abc_service(auth_subject.login, req_pb.meta.abc_service_id, u'meta.abc_service_id')

    project_map = config.get_value(u'namespace_annotations.robot_projects', {})
    category_map = config.get_value(u'namespace_annotations.category_projects', {})
    project = project_map.get(login, category_map.get(req_pb.meta.category, 'unknown'))

    prj = req_pb.order.instance_tags.prj if req_pb.order.HasField('instance_tags') else namespace_id
    namespace.validate_namespace_normalized_prj(namespace_id, prj)

    if req_pb.HasField('order'):
        namespace.validate_order_content(order_content_pb=req_pb.order, namespace_id=namespace_id,
                                         auth_subject=auth_subject, abc_service_id=req_pb.meta.abc_service_id,
                                         validate_yp_endpoint_sets=req_pb.validate_yp_endpoint_sets)

        namespace_pb = IDao.instance().create_namespace(
            meta_pb=req_pb.meta,
            order_content_pb=req_pb.order,
            login=auth_subject.login,
            dry_run=dry_run,
            project=project,
        )
    else:
        order_content_pb = model_pb2.NamespaceOrder.Content()
        order_content_pb.flow_type = order_content_pb.EMPTY
        if not req_pb.meta.auth.staff.owners.group_ids:
            raise rpc.exceptions.BadRequestError(
                "Creating empty namespaces without any staff group in owners is forbidden"
            )
        try:
            order_content_pb.alerting_simple_settings.notify_staff_group_id = int(
                req_pb.meta.auth.staff.owners.group_ids[0])
        except ValueError:
            raise rpc.exceptions.BadRequestError(
                "Creating empty namespaces requires valid staff group id in owners, got group that cannot be parsed: %r" % (
                    req_pb.meta.auth.staff.owners.group_ids[0],
                )
            )

        namespace_pb = IDao.instance().create_namespace(
            meta_pb=req_pb.meta,
            order_content_pb=order_content_pb,
            login=auth_subject.login,
            dry_run=dry_run,
            project=project,
        )

    return api_pb2.CreateNamespaceResponse(namespace=namespace_pb)


@namespace_service_bp.method('CreateNamespaceOnAllocatedResources',
                             request_type=api_pb2.CreateNamespaceOnAllocatedResourcesRequest,
                             response_type=api_pb2.CreateNamespaceOnAllocatedResourcesResponse)
def create_namespace_on_allocated_resources(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateNamespaceOnAllocatedResourcesRequest
    :rtype: api_pb2.CreateNamespaceOnAllocatedResourcesResponse
    """
    raise rpc.exceptions.BadRequestError('Creating GENCFG balancers is not supported')


@namespace_service_bp.method('UpdateNamespace',
                             request_type=api_pb2.UpdateNamespaceRequest,
                             response_type=api_pb2.UpdateNamespaceResponse)
def update_namespace(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateNamespaceRequest
    :rtype: api_pb2.UpdateNamespaceResponse
    """
    if req_pb.spec.layout_type == req_pb.spec.NS_LAYOUT_L3_ONLY:
        # Apply constraints before validation
        namespace.apply_l3_only_layout_constraints(req_pb.spec)
    elif req_pb.spec.layout_type == req_pb.spec.NS_LAYOUT_GLOBAL:
        namespace.apply_global_layout_constraints(req_pb.spec)
    if req_pb.spec.cloud_type == model_pb2.CT_AZURE:
        namespace.apply_external_layout_constraints(req_pb.spec)

    namespace.validate_request(req_pb)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    namespace_id = req_pb.meta.id

    zk = IZkStorage.instance()
    _cache = cache.IAwacsCache.instance()

    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_update(namespace_pb, req_pb, auth_subject)

    auth_has_changed = req_pb.meta.HasField('auth') and namespace_pb.meta.auth != req_pb.meta.auth
    if auth_has_changed and not req_pb.meta.auth.staff.owners.logins and not req_pb.meta.auth.staff.owners.group_ids:
        raise exceptions.BadRequestError('"auth.staff.owners": at least one of "logins" and "groups" must be not empty')
    webauth_has_changed = req_pb.meta.webauth and namespace_pb.meta.webauth != req_pb.meta.webauth
    observers_has_changed = req_pb.meta.observers and namespace_pb.meta.observers != req_pb.meta.observers
    if (webauth_has_changed and
            req_pb.meta.webauth.responsible.logins != namespace_pb.meta.webauth.responsible.logins and
            not req_pb.meta.webauth.responsible.logins
        ):
        result = _cache.list_balancers(namespace_id=namespace_pb.meta.id)
        for balancer_pb in result.items:
            if (balancer_pb.spec.yandex_balancer.config.l7_macro.webauth.action ==
                    balancer_pb.spec.yandex_balancer.config.l7_macro.webauth.AUTHENTICATE_USING_IDM):
                raise rpc.exceptions.BadRequestError(
                    '"meta.webauth.responsible.logins": can not be empty if webauth.action is set to '
                    'AUTHENTICATE_USING_IDM in balancers'
                )

    is_author_root = util.is_root(auth_subject.login)
    if req_pb.HasField('spec'):
        for attr in ('easy_mode_settings', 'object_upper_limits', 'modules_whitelist',
                     'rps_limiter_allowed_installations'):
            if not req_pb.spec.HasField(attr):
                getattr(req_pb.spec, attr).CopyFrom(getattr(namespace_pb.spec, attr))
            elif not is_author_root and getattr(req_pb.spec, attr) != getattr(namespace_pb.spec, attr):
                raise rpc.exceptions.ForbiddenError('"spec.{}": can be changed only by roots'.format(attr))
        if not is_author_root and req_pb.spec.customizable_components != namespace_pb.spec.customizable_components:
            raise rpc.exceptions.ForbiddenError('"spec.customizable_components": can be changed only by roots')

        if req_pb.spec.modules_whitelist != namespace_pb.spec.modules_whitelist:
            removed = set(module_name for module_name in namespace_pb.spec.modules_whitelist.modules
                          if module_name not in req_pb.spec.modules_whitelist.modules)
            util.validate_unwhitelisted_modules_nonexistence(namespace_id, removed)

        if req_pb.spec.rps_limiter_allowed_installations != namespace_pb.spec.rps_limiter_allowed_installations:
            for installation_name in req_pb.spec.rps_limiter_allowed_installations.installations:
                try:
                    rps_limiter_settings.validate_installation(
                        installation_name, req_pb.spec.rps_limiter_allowed_installations.installations)
                except ValidationError as e:
                    raise rpc.exceptions.BadRequestError(
                        '"spec.rps_limiter.rps_limiter_allowed_installations": ' + e.message)

    spec_has_changed = req_pb.HasField('spec') and namespace_pb.spec != req_pb.spec
    if spec_has_changed and namespace_pb.spec.incomplete:
        raise exceptions.BadRequestError('Namespace Order not processed')

    if spec_has_changed and not is_author_root:
        proto_readonly.validate_readonly_fields(req_pb.spec, namespace_pb.spec)
        if req_pb.spec.HasField('its') and not namespace_pb.spec.HasField('its'):
            req_pb.spec.its.ctl_version = namespace.MAX_ITS_CTL_VERSION
            namespace.validate_namespaces_with_enabled_its_count()
        if (req_pb.spec.HasField('its') and namespace_pb.spec.HasField('its') and
                req_pb.spec.its.ctl_version != namespace_pb.spec.its.ctl_version):
            raise exceptions.BadRequestError('"spec.its.ctl_version": can only be changed by roots')
        if (any(knob_pb.its_ruchka_id == WEIGHTS_KNOB_ID for knob_pb in namespace_pb.spec.its.knobs.common_knobs) and
                not any(knob_pb.its_ruchka_id == WEIGHTS_KNOB_ID for knob_pb in req_pb.spec.its.knobs.common_knobs)):
            l7heavy_config_pb = objects.L7HeavyConfig.cache.get(namespace_id, namespace_id)
            if l7heavy_config_pb is not None:
                raise exceptions.BadRequestError('"spec.its.knobs.common_knobs": "{}" can not be removed while '
                                                 'L7Heavy config exists'.format(WEIGHTS_KNOB_ID))

    category_has_changed = req_pb.meta.category and namespace_pb.meta.category != req_pb.meta.category
    abc_service_id_has_changed = (req_pb.meta.abc_service_id and
                                  namespace_pb.meta.abc_service_id != req_pb.meta.abc_service_id)
    its_knobs_sync_enabled_has_changed = namespace_pb.meta.its_knobs_sync_enabled != req_pb.meta.its_knobs_sync_enabled

    if abc_service_id_has_changed:
        util.validate_user_belongs_to_abc_service(auth_subject.login, req_pb.meta.abc_service_id, 'meta.abc_service_id')

    has_annotations = len(req_pb.meta.annotations) > 0
    annotations_has_changed = (
        has_annotations
        and (len(req_pb.meta.annotations) != len(namespace_pb.meta.annotations)
             or any(namespace_pb.meta.annotations.get(k) != req_pb.meta.annotations[k]
                    for k in req_pb.meta.annotations)
             )
    )
    project_has_changed = (
        has_annotations
        and req_pb.meta.annotations.get('project') != namespace_pb.meta.annotations.get('project')
    )
    creator_has_changed = (
        has_annotations
        and req_pb.meta.annotations.get('creator') != namespace_pb.meta.annotations.get('creator')
    )

    if (
            not auth_has_changed
            and not category_has_changed
            and not webauth_has_changed
            and not observers_has_changed
            and not abc_service_id_has_changed
            and not its_knobs_sync_enabled_has_changed
            and not spec_has_changed
            and not annotations_has_changed
    ):
        return api_pb2.UpdateNamespaceResponse(namespace=namespace_pb)

    if project_has_changed or creator_has_changed:
        authorize_root_operation(auth_subject, 'update namespace immutable annotations')

    if spec_has_changed:
        namespace.validate_alerting(namespace_id, namespace_pb.spec, req_pb.spec)

    if spec_has_changed and namespace_pb.spec.balancer_constraints != req_pb.spec.balancer_constraints:
        if not is_author_root:
            raise rpc.exceptions.ForbiddenError('"spec.balancer_constraints": can be changed only by roots')
        namespace.validate_namespace_normalized_prj(namespace_id, req_pb.spec.balancer_constraints.instance_tags.prj,
                                                    field_name="spec.balancer_constraints.instance_tags.prj")

    if namespace_pb.spec.layout_type != req_pb.spec.layout_type:
        if not is_author_root:
            raise rpc.exceptions.ForbiddenError('"spec.layout_type": can be changed only by roots')
        if req_pb.spec.layout_type == req_pb.spec.NS_LAYOUT_L3_ONLY:
            backend_pbs = _cache.list_all_backends(namespace_id=namespace_id)
            for backend_pb in backend_pbs:
                if backend_pb.spec.selector.type in (backend_pb.spec.selector.YP_ENDPOINT_SETS_SD,
                                                     backend_pb.spec.selector.BALANCERS):
                    raise rpc.exceptions.ForbiddenError(
                        '"spec.layout_type": can not be updated to NS_LAYOUT_L3_ONLY: backend "{}" has '
                        'forbidden selector type'.format(backend_pb.meta.id))
        elif req_pb.spec.layout_type == req_pb.spec.NS_LAYOUT_GLOBAL:
            backend_pbs = _cache.list_all_backends(namespace_id=namespace_id)
            for backend_pb in backend_pbs:
                if backend_pb.spec.selector.type != backend_pb.spec.selector.YP_ENDPOINT_SETS_SD:
                    raise rpc.exceptions.ForbiddenError(
                        '"spec.layout_type": can not be updated to NS_LAYOUT_GLOBAL: backend "{}" has '
                        'forbidden selector type'.format(backend_pb.meta.id))

    if namespace_pb.spec.env_type != req_pb.spec.env_type:
        if (
                namespace_pb.spec.env_type != model_pb2.NamespaceSpec.NS_ENV_UNKNOWN
                or req_pb.spec.env_type == model_pb2.NamespaceSpec.NS_ENV_UNKNOWN
        ):
            authorize_root_operation(auth_subject, 'perform restricted namespace load mode transition')
        elif req_pb.spec.env_type == model_pb2.NamespaceSpec.NS_ENV_TESTING:
            balancer_pbs = _cache.list_all_balancers(namespace_id=namespace_id)
            if any(
                    balancer_pb.spec.env_type != model_pb2.BalancerSpec.L7_ENV_TESTING
                    for balancer_pb in balancer_pbs
            ):
                raise rpc.exceptions.BadRequestError(
                    "Namespace cannot be marked TESTING while non-TESTING balancers exist"
                )
        elif req_pb.spec.env_type == model_pb2.NamespaceSpec.NS_ENV_PRODUCTION:
            balancer_pbs = _cache.list_all_balancers(namespace_id=namespace_id)
            if any(
                    balancer_pb.spec.env_type == model_pb2.BalancerSpec.L7_ENV_TESTING
                    for balancer_pb in balancer_pbs
            ):
                raise rpc.exceptions.BadRequestError(
                    "Namespace cannot be marked PRODUCTION while TESTING balancers exist"
                )

    if not spec_has_changed and not req_pb.meta.version:
        # maintain backward compatibility
        req_pb.meta.version = namespace_pb.meta.version

    namespace_pb = IDao.instance().update_namespace(
        namespace_id=namespace_id,
        version=req_pb.meta.version,
        comment=req_pb.meta.comment,
        login=auth_subject.login,
        updated_auth_pb=req_pb.meta.auth if auth_has_changed else None,
        updated_category=req_pb.meta.category if category_has_changed else None,
        updated_webauth_pb=req_pb.meta.webauth if webauth_has_changed else None,
        updated_observers_pb=req_pb.meta.observers if observers_has_changed else None,
        updated_abc_service_id=req_pb.meta.abc_service_id if abc_service_id_has_changed else None,
        updated_its_knobs_sync_enabled=req_pb.meta.its_knobs_sync_enabled if its_knobs_sync_enabled_has_changed else None,
        updated_spec_pb=req_pb.spec if spec_has_changed else None,
        updated_annotations=req_pb.meta.annotations if annotations_has_changed else None,
    )

    return api_pb2.UpdateNamespaceResponse(namespace=namespace_pb)


@namespace_service_bp.method('UpdateNamespaceOrderContext',
                             request_type=api_pb2.UpdateNamespaceOrderContextRequest,
                             response_type=api_pb2.UpdateNamespaceOrderContextResponse)
def update_namespace_order_context(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateNamespaceOrderContextRequest
    :rtype: api_pb2.UpdateNamespaceOrderContextResponse
    """
    namespace.validate_request(req_pb)
    namespace_id = req_pb.id

    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_update(namespace_pb, req_pb, auth_subject)

    if not namespace_pb.HasField('order'):
        raise exceptions.BadRequestError(u'Namespace has no order.')

    for namespace_pb in zk.update_namespace(namespace_id, namespace_pb=namespace_pb):
        order_pb = namespace_pb.order
        for field_pb in req_pb.fields:
            if field_pb.value:
                order_pb.progress.context[field_pb.key] = field_pb.value
            elif field_pb.key in order_pb.progress.context:
                del order_pb.progress.context[field_pb.key]
            order_pb.progress.state.attempts = 0

    return api_pb2.UpdateNamespaceOrderContextResponse()


@namespace_service_bp.method('CancelNamespaceOrder',
                             request_type=api_pb2.CancelNamespaceOrderRequest,
                             response_type=api_pb2.CancelNamespaceOrderResponse)
def cancel_namespace_order(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelNamespaceOrderRequest
    :rtype: api_pb2.CancelNamespaceOrderResponse
    """
    namespace.validate_request(req_pb)
    namespace_id = req_pb.id

    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_remove(namespace_pb, auth_subject)

    if not namespace_pb.HasField('order'):
        raise exceptions.BadRequestError(u'Namespace has no order.')

    if req_pb.force:
        if auth_subject.login not in config.get_value('run.root_users', ()):
            raise rpc.exceptions.ForbiddenError('Only root users can force cancel namespace order')
    else:
        namespace.validate_cancel_namespace_order(namespace_pb)
    for namespace_pb in zk.update_namespace(namespace_id, namespace_pb=namespace_pb):
        cancel_order(namespace_pb, author=auth_subject.login, comment='Cancelled in UI')
        if req_pb.force:
            order_pb = namespace_pb.order
            order_pb.progress.state.id = 'FINISH'
            order_pb.status.status = 'CANCELLED'
            namespace_pb.meta.ClearField('auth')
            namespace_pb.meta.auth.type = namespace_pb.meta.auth.STAFF
            namespace_pb.meta.auth.staff.owners.logins.append(auth_subject.login)
            namespace_pb.meta.comment = 'Force cancelled by {}'.format(auth_subject.login)
    return api_pb2.CancelNamespaceOrderResponse()


@namespace_service_bp.method('ListNamespaceRevisions',
                             request_type=api_pb2.ListNamespaceRevisionsRequest,
                             response_type=api_pb2.ListNamespaceRevisionsResponse)
def list_namespace_revisions(req_pb, _):
    """
    :type req_pb: api_pb2.ListNamespaceRevisionsRequest
    :rtype: api_pb2.ListNamespaceRevisionsResponse
    """
    namespace.validate_request(req_pb)
    namespace_id = req_pb.id

    zk = IZkStorage.instance()
    zk.must_get_namespace(namespace_id)

    mongo_storage = db.IMongoStorage.instance()
    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = mongo_storage.list_namespace_revs(namespace_id, skip=skip, limit=limit)
    return api_pb2.ListNamespaceRevisionsResponse(revisions=revs.items, total=revs.total)


@namespace_service_bp.method('GetNamespaceRevision',
                             request_type=api_pb2.GetNamespaceRevisionRequest,
                             response_type=api_pb2.GetNamespaceRevisionResponse)
def get_namespace_revision(req_pb, _):
    """
    :type req_pb: api_pb2.GetNamespaceRevisionRequest
    :rtype: api_pb2.GetNamespaceRevisionResponse
    """
    namespace.validate_request(req_pb)
    rev_id = req_pb.id

    mongo_storage = db.IMongoStorage.instance()
    rev_pb = mongo_storage.must_get_namespace_rev(rev_id)
    return api_pb2.GetNamespaceRevisionResponse(revision=rev_pb)


@namespace_service_bp.method('RemoveNamespace',
                             request_type=api_pb2.RemoveNamespaceRequest,
                             response_type=api_pb2.RemoveNamespaceResponse,
                             is_destructive=True)
def remove_namespace(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.RemoveNamespaceRequest
    :rtype: api_pb2.RemoveNamespaceResponse
    """
    namespace.validate_request(req_pb)
    namespace_id = req_pb.id

    zk = IZkStorage.instance()
    namespace_pb = zk.must_get_namespace(namespace_id)
    authorize_remove(namespace_pb, auth_subject)

    req_version = req_pb.version if req_pb.version else namespace_pb.meta.version
    if namespace_pb.meta.version != req_version:
        raise rpc.exceptions.BadRequestError(u'Namespace removing conflict: assumed version="{}", current="{}"'.format(
            req_version, namespace_pb.meta.version))

    _dao = IDao.instance()
    can_delete_namespace, reason = _dao.can_delete_namespace(namespace_pb)
    if not can_delete_namespace:
        raise rpc.exceptions.BadRequestError(reason)

    updated_spec_pb = clone_pb(namespace_pb.spec)
    updated_spec_pb.deleted = True

    _dao.update_namespace(
        namespace_id=namespace_id,
        version=req_version,
        comment='Marked as deleted by {}'.format(auth_subject.login),
        login=auth_subject.login,
        updated_spec_pb=updated_spec_pb
    )

    return api_pb2.RemoveNamespaceResponse()


@namespace_service_bp.method('GetNamespaceAspectsSet',
                             request_type=api_pb2.GetNamespaceAspectsSetRequest,
                             response_type=api_pb2.GetNamespaceAspectsSetResponse)
def get_namespace_aspects_set(req_pb, _):
    namespace.validate_request(req_pb)
    namespace_id = req_pb.id

    zk = IZkStorage.instance()
    zk.must_get_namespace(namespace_id)

    c = cache.IAwacsCache.instance()
    namespace_aspects_set_pb = c.must_get_namespace_aspects_set(namespace_id=namespace_id)

    filled_namespace_aspects_set_pb = clone_pb(namespace_aspects_set_pb)
    fill_ui_namespace_aspects(namespace_id, filled_namespace_aspects_set_pb.content)

    return api_pb2.GetNamespaceAspectsSetResponse(aspects_set=filled_namespace_aspects_set_pb)


@namespace_service_bp.method('GetNamespaceKnobs',
                             request_type=api_pb2.GetNamespaceKnobsRequest,
                             response_type=api_pb2.GetNamespaceKnobsResponse)
def get_knobs_configuration(req_pb, _):
    """
    :type req_pb: api_pb2.GetNamespaceKnobsRequest
    :rtype: api_pb2.GetNamespaceKnobsResponse
    """
    namespace.validate_request(req_pb)
    namespace_id = req_pb.namespace_id

    resp_pb = api_pb2.GetNamespaceKnobsResponse()

    c = cache.IAwacsCache.instance()
    balancer_state_pbs = c.list_all_balancer_states(namespace_id=namespace_id)
    for balancer_state_pb in balancer_state_pbs:

        balancer_pb = c.must_get_balancer(namespace_id=balancer_state_pb.namespace_id,
                                          balancer_id=balancer_state_pb.balancer_id)
        transport_pb = balancer_pb.spec.config_transport
        if transport_pb.type != model_pb2.NANNY_STATIC_FILE:
            raise rpc.exceptions.BadRequestError(u'Only Nanny-powered balancers are supported at the moment.')
        active_knob_revisions = {}
        h = L7BalancerStateHandler(balancer_state_pb)
        for full_id, revs in h.iter_knob_items():
            active_rev = revs.select_active_rev()
            if active_rev:
                active_ver = KnobVersion.from_rev_status_pb(full_id, active_rev.pb)
                if not active_ver.deleted:
                    active_knob_revisions[full_id] = active_ver

        balancer_resp_pb = resp_pb.balancers.add(balancer_id=balancer_state_pb.balancer_id)
        for full_id, active_version in six.iteritems(active_knob_revisions):
            active_spec_pb = find_knob_revision_spec(active_version)
            nanny_service_id = transport_pb.nanny_static_file.service_id
            filename = active_spec_pb.its_watched_state.filename.lstrip('./')
            dynamic_value_id = u'n@{}:{}'.format(nanny_service_id, filename)
            balancer_resp_pb.knobs.add(
                knob_id=full_id[1],
                nanny_service_id=nanny_service_id,
                dynamic_value_id=dynamic_value_id,
                filename=filename)

    return resp_pb


@namespace_service_bp.method('GetNamespaceAlertingConfig',
                             request_type=api_pb2.GetNamespaceAlertingConfigRequest,
                             response_type=api_pb2.GetNamespaceAlertingConfigResponse)
def get_namespace_alerting_config(req_pb, _):
    namespace.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    if req_pb.consistency == api_pb2.STRONG:
        zk = IZkStorage.instance()
        namespace_pb = zk.must_get_namespace(namespace_id, sync=True)
    else:
        c = cache.IAwacsCache.instance()
        namespace_pb = c.must_get_namespace(namespace_id=namespace_id)

    current_config = None
    current_version = None
    if namespace_pb.spec.HasField('alerting'):
        alerting_config = alerting.get_config(namespace_pb.spec.alerting.version)
        current_version = alerting_config.version
        if not alerting_config:
            raise RuntimeError('AlertingConfig with version: "{}" does not exists'.format(
                namespace_pb.spec.alerting.version))
        alerting_prefix = config.get_value('alerting.name_prefix', 'undefined')
        current_config = api_pb2.GetNamespaceAlertingConfigResponse.AlertingConfig()
        current_config.alerting_prefix = alerting_prefix
        current_config.juggler_namespace = alerting_config.build_juggler_namespace(alerting_prefix,
                                                                                   namespace_pb.meta.id)
        current_config.yasm_alerts_prefix = alerting_config.build_yasm_alert_prefix(alerting_prefix,
                                                                                    namespace_pb.meta.id)

        current_config.wiki_alerts_url = alerting.WIKI_MONITORING_DOC_URL
        current_config.wiki_alerts_actions_url = alerting.WIKI_MONITORING_ACTION_ITEMS_URL

        _cache = cache.IAwacsCache.instance()
        balancer_pbs = _cache.list_all_balancers(namespace_id=namespace_id)
        locations = [b.meta.location.yp_cluster or b.meta.location.gencfg_dc
                     for b in balancer_pbs if b.meta.HasField('location')]
        if locations:
            current_config.yasm_alerts_panel_url = alerting.make_yasm_alerts_panel_url(
                alerting_prefix=alerting_prefix,
                namespace_id=namespace_id,
                locations=locations
            )

        for yasm_alert in chain(*alerting_config.yasm_alerts_by_groups.values()):  # type: alerting.YasmAlert
            yasm_alert_pb = current_config.yasm_alerts.add()
            yasm_alert_pb.group = yasm_alert.notify_group
            yasm_alert_pb.name = yasm_alert.name
            yasm_alert_pb.description = yasm_alert.description
            yasm_alert_pb.signal = yasm_alert.signal
            if yasm_alert.warn_threshold:
                yasm_alert_pb.warn_threshold[:] = [str(t) for t in yasm_alert.warn_threshold]
            if yasm_alert.crit_threshold:
                yasm_alert_pb.crit_threshold[:] = [str(t) for t in yasm_alert.crit_threshold]
            yasm_alert_pb.flaps_stable_time = yasm_alert.flaps_stable_time
    return api_pb2.GetNamespaceAlertingConfigResponse(
        available_versions=[
            str(v)
            for v in alerting.get_versions()
            if v >= alerting.CURRENT_VERSION
            or v == current_version
        ],
        current_config=current_config,
        alerting_sync_status=namespace_pb.alerting_sync_status
    )


@namespace_service_bp.method('ApplyNamespaceAlertingPreset',
                             request_type=api_pb2.ApplyNamespaceAlertingPresetRequest,
                             response_type=api_pb2.ApplyNamespaceAlertingPresetResponse,
                             )
def apply_namespace_alerting_preset(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.ApplyNamespaceAlertingPresetRequest
    :rtype: api_pb2.ApplyNamespaceAlertingPresetResponse
    """
    authorize_root_operation(auth_subject, 'update namespace preset')
    namespace.validate_request(req_pb)
    namespace_id = req_pb.namespace_id

    zk = IZkStorage.instance()
    namespace_pb = zk.must_get_namespace(namespace_id)

    namespace.validate_preset(
        namespace_pb.meta.annotations.get('project', 'default'),
        req_pb.preset,
    )

    if namespace_pb.spec.preset == req_pb.preset:
        raise exceptions.BadRequestError('Namespace preset not changed')

    version = req_pb.version or namespace_pb.meta.version

    alerting.apply_alerting_preset(namespace_pb.spec, req_pb.preset)

    namespace_pb = IDao.instance().update_namespace(
        namespace_id=namespace_id,
        version=version,
        comment='Apply alerting preset {}'.format(
            model_pb2.NamespaceSpec.NamespaceSettingsPreset.Name(req_pb.preset)
        ),
        login=auth_subject.login,
        updated_spec_pb=namespace_pb.spec,
    )

    return api_pb2.ApplyNamespaceAlertingPresetResponse(namespace=namespace_pb)


@namespace_service_bp.method(u'CreateNamespaceOperation',
                             request_type=api_pb2.CreateNamespaceOperationRequest,
                             response_type=api_pb2.CreateNamespaceOperationResponse)
def create_namespace_operation(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateNamespaceOperationRequest
    """
    req_pb.meta.id = six.text_type(uuid.uuid4())
    namespace.validate_namespace_op_create(req_pb, auth_subject)

    namespace_pb = IZkStorage.instance().must_get_namespace(req_pb.meta.namespace_id)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    if objects.NamespaceOperation.get_operation_name(req_pb.order) == objects.NamespaceOperation.IMPORT_VS_FROM_L3MGR:
        if config.get_value(u'run.restrict_l3_migration_in_api', False) and not is_root(auth_subject):
            raise exceptions.ForbiddenError(u'Migration is temporarily restricted, '
                                            u'please create a ticket in st/BALANCERSUPPORT if you need '
                                            u'to enable full config management')

    pb = objects.NamespaceOperation.create(
        meta_pb=req_pb.meta,
        order_content_pb=req_pb.order,
        login=auth_subject.login)
    return api_pb2.CreateNamespaceOperationResponse(operation=pb)


@namespace_service_bp.method(u'CancelNamespaceOperation',
                             request_type=api_pb2.CancelNamespaceOperationRequest,
                             response_type=api_pb2.CancelNamespaceOperationResponse)
def cancel_namespace_operation(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CancelNamespaceOperationRequest
    """
    namespace.validate_namespace_op_cancel(req_pb, auth_subject)

    comment = u'Cancelled via API'
    if req_pb.force:
        comment += u' (forced)'
    objects.NamespaceOperation.zk.cancel_order(
        namespace_id=req_pb.namespace_id,
        op_id=req_pb.id,
        author=auth_subject.login,
        comment=comment)
    return api_pb2.CancelNamespaceOperationResponse()


@namespace_service_bp.method(u'ListNamespaceOperations',
                             request_type=api_pb2.ListNamespaceOperationsRequest,
                             response_type=api_pb2.ListNamespaceOperationsResponse,
                             max_in_flight=5)
def list_namespace_operations(req_pb, _):
    """
    :type req_pb: api_pb2.ListNamespaceOperationsRequest
    """
    namespace.validate_request(req_pb)
    namespace_op_pbs, total = apicache.IAwacsApiCache.instance().list_namespace_operations(
        namespace_id=req_pb.namespace_id,
        skip=req_pb.skip or None,
        limit=req_pb.limit or None)
    return api_pb2.ListNamespaceOperationsResponse(total=total, operations=namespace_op_pbs)


@namespace_service_bp.method(u'GetNamespaceOperation',
                             request_type=api_pb2.GetNamespaceOperationRequest,
                             response_type=api_pb2.GetNamespaceOperationResponse)
def get_namespace_operation(req_pb, _):
    """
    :type req_pb: api_pb2.GetNamespaceOperationRequest
    """
    namespace.validate_request(req_pb)
    pb = objects.NamespaceOperation.cache.must_get(req_pb.namespace_id, req_pb.id)
    return api_pb2.GetNamespaceOperationResponse(operation=pb)
