# coding: utf-8
import collections
import logging

import itertools
import ujson
import six
from datetime import datetime, timedelta

from awacs.lib import rpc
from awacs.model import util
from awacs.model.db import IMongoStorage
from infra.awacs.proto import api_pb2, model_pb2
from .util import AwacsBlueprint


statistics_service_bp = AwacsBlueprint('rpc_statistics_service', __name__, '/api')
log = logging.getLogger()

SRE_LOGINS = frozenset(('keepclean', 'moridin'))
IGNORED_LOGINS = frozenset(('romanovich',))
MOST_ACTIVE_USERS_LIMIT = 100


@statistics_service_bp.method('GetStatistics',
                              request_type=api_pb2.GetStatisticsRequest,
                              response_type=api_pb2.GetStatisticsResponse,
                              max_in_flight=1)
def get_statistics(req_pb, auth_subject):
    if not req_pb.id:
        raise rpc.exceptions.BadRequestError('"id" must be set')
    if req_pb.id != 'sre_vs_users':
        raise rpc.exceptions.BadRequestError('"id" must be "sre_vs_users"')

    updates = collections.defaultdict(collections.Counter)
    most_active_users = collections.Counter()
    db = IMongoStorage.instance()
    fields = ('ctime', 'author')
    spec = {
        'author': {'$ne': util.NANNY_ROBOT_LOGIN},
    }
    year_ago = (datetime.utcnow() - timedelta(days=365)).date()
    for item in itertools.chain(
            db.raw_list_l3_balancer_revs(spec=spec, fields=fields),
            db.raw_list_balancer_revs(spec=spec, fields=fields),
            db.raw_list_upstream_revs(spec=spec, fields=fields),
            db.raw_list_backend_revs(spec=spec, fields=fields),
    ):
        author = item['author']
        if author in IGNORED_LOGINS:
            continue
        date = datetime.fromtimestamp(item['ctime'] / 1000).date()
        is_sre = author in SRE_LOGINS
        author_group = 'sre' if is_sre else 'users'
        updates[date][author_group] += 1
        if not is_sre and date >= year_ago:
            most_active_users[author] += 1

    data = {
        'sre_vs_users': sorted(updates.items()),
        'most_active_users': most_active_users.most_common(MOST_ACTIVE_USERS_LIMIT)
    }
    return api_pb2.GetStatisticsResponse(data=ujson.dumps(data))


@statistics_service_bp.method('GetNotificationsStatistics',
                              request_type=api_pb2.GetNotificationsStatisticsRequest,
                              response_type=api_pb2.GetNotificationsStatisticsResponse,
                              max_in_flight=1)
def get_notifications_statistics(req_pb, auth_subject):
    today = datetime.utcnow().date()
    year_ago = today - timedelta(days=365)

    date_from = req_pb.start.ToDatetime().date() if req_pb.HasField('start') else year_ago
    date_to = req_pb.end.ToDatetime().date() if req_pb.HasField('end') else today

    updates = collections.defaultdict(collections.Counter)
    db = IMongoStorage.instance()

    for item in db.list_notifications_statistics_entries(
        start=date_from,
        end=date_to,
        limit=(date_to - date_from).days,
    ):
        ts = item.start.ToDatetime().strftime('%Y-%m-%d')
        for method, count in six.iteritems(item.date_statistics.alerts_sent):
            updates[ts][method] += count

    data = {
        'notifications_by_day': list(sorted(six.iteritems(updates))),
    }
    return api_pb2.GetNotificationsStatisticsResponse(data=ujson.dumps(data))


@statistics_service_bp.method('GetUsageStatisticsEntry',
                              request_type=api_pb2.GetUsageStatisticsEntryRequest,
                              response_type=api_pb2.GetUsageStatisticsEntryResponse,
                              max_in_flight=1)
def get_usage_statistics_entry(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetUsageStatisticsEntryRequest
    """
    db = IMongoStorage.instance()
    entry_pb = db.must_get_usage_statistics_entry(span=model_pb2.UsageStatisticsEntry.Span.Name(req_pb.span),
                                                  start=req_pb.start.ToDatetime())
    resp_pb = api_pb2.GetUsageStatisticsEntryResponse()
    resp_pb.entry.CopyFrom(entry_pb)
    return resp_pb


@statistics_service_bp.method('GetLoadStatisticsEntry',
                              request_type=api_pb2.GetLoadStatisticsEntryRequest,
                              response_type=api_pb2.GetLoadStatisticsEntryResponse,
                              max_in_flight=1)
def get_load_statistics_entry(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetLoadStatisticsEntryRequest
    """
    db = IMongoStorage.instance()
    entry_pb = db.must_get_load_statistics_entry(span=model_pb2.LoadStatisticsEntry.Span.Name(req_pb.span),
                                                 start=req_pb.start.ToDatetime())
    resp_pb = api_pb2.GetLoadStatisticsEntryResponse()
    resp_pb.entry.CopyFrom(entry_pb)
    return resp_pb


@statistics_service_bp.method('GetRps',
                              request_type=api_pb2.GetRpsRequest,
                              response_type=api_pb2.GetRpsResponse)
def get_rps(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetRpsRequest
    """
    db = IMongoStorage.instance()
    if req_pb.yesterday:
        start = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0) - timedelta(hours=24)
    else:
        start = req_pb.date.ToDatetime().replace(hour=0, minute=0, second=0, microsecond=0)
    entry_pb = db.must_get_load_statistics_entry(span='DATE', start=start)
    resp_pb = api_pb2.GetRpsResponse()
    if not req_pb.namespace_id:
        raise rpc.exceptions.BadRequestError('"namespace_id" must be set')
    if req_pb.upstream_id:
        full_upstream_id = req_pb.namespace_id + '/' + req_pb.upstream_id
        by_upstream_rps = entry_pb.date_statistics.by_upstream.get(full_upstream_id)
        if by_upstream_rps is None:
            raise rpc.exceptions.NotFoundError('Rps statistics for upstream "{}:{}" does not exist'
                                               .format(req_pb.namespace_id, req_pb.upstream_id))
        resp_pb.rps.CopyFrom(by_upstream_rps)
    elif req_pb.balancer_id:
        full_balancer_id = req_pb.namespace_id + '/' + req_pb.balancer_id
        by_balancer_rps = entry_pb.date_statistics.by_balancer.get(full_balancer_id)
        if by_balancer_rps is None:
            raise rpc.exceptions.NotFoundError('Rps statistics for balancer "{}:{}" does not exist'
                                               .format(req_pb.namespace_id, req_pb.balancer_id))
        resp_pb.rps.CopyFrom(by_balancer_rps)
    else:
        by_namespace_rps = entry_pb.date_statistics.by_namespace.get(req_pb.namespace_id)
        if by_namespace_rps is None:
            raise rpc.exceptions.NotFoundError('Rps statistics for namespace "{}" does not exist'
                                               .format(req_pb.namespace_id))
        resp_pb.rps.CopyFrom(by_namespace_rps)
    return resp_pb
