# coding: utf-8
import logging
from collections import namedtuple

import inject
import os
import re
import six

from awacs.lib import rpc, zookeeper_client
from awacs.model import objects
from awacs.model.cache import IAwacsCache
from infra.awacs.proto import api_pb2
from infra.swatlib.gevent.exclusiveservice2 import ExclusiveService
from .util import AwacsBlueprint


status_service_bp = AwacsBlueprint('rpc_status_service', __name__, '/api/status')

ctl_fields_info = namedtuple('ctl_info', ('pb_field_name', 'cache_getter'))

supported_ctls = {
    'namespace-ctl': ctl_fields_info('namespace_ctl_id', 'get_namespace'),
    'namespace-order-ctl': ctl_fields_info('namespace_order_ctl_id', 'get_namespace'),
    'namespace-operation-ctl': ctl_fields_info('namespace_operation_ctl_id', objects.NamespaceOperation),

    'l3-balancer-ctl': ctl_fields_info('l3_balancer_ctl_id', 'get_l3_balancer'),
    'l3-balancer-ctl-v2': ctl_fields_info('l3_balancer_ctl_id', 'get_l3_balancer'),
    'l3-balancer-order-ctl': ctl_fields_info('l3_balancer_order_ctl_id', 'get_l3_balancer'),

    'balancer-ctl': ctl_fields_info('balancer_ctl_id', 'get_balancer'),
    'balancer-order-ctl': ctl_fields_info('balancer_order_ctl_id', 'get_balancer'),
    'balancer-removal-ctl': ctl_fields_info('balancer_removal_ctl_id', 'get_balancer'),
    'balancer-op-ctl': ctl_fields_info('balancer_operation_ctl_id', 'get_balancer_operation'),

    'backend-ctl': ctl_fields_info('backend_ctl_id', 'get_backend'),

    'cert-ctl': ctl_fields_info('cert_ctl_id', 'get_cert'),
    'cert-order-ctl': ctl_fields_info('cert_order_ctl_id', 'get_cert'),
    'cert-renewal-ctl': ctl_fields_info('cert_renewal_ctl_id', 'get_cert_renewal'),
    'cert-renewal-order-ctl': ctl_fields_info('cert_renewal_order_ctl_id', 'get_cert_renewal'),

    'domain-ctl': ctl_fields_info('domain_ctl_id', 'get_domain'),
    'domain-order-ctl': ctl_fields_info('domain_order_ctl_id', 'get_domain'),
    'domain-op-ctl': ctl_fields_info('domain_operation_ctl_id', 'get_domain_operation'),

    'dns-record-ctl': ctl_fields_info('dns_record_ctl_id', 'get_dns_record'),
    'dns-record-order-ctl': ctl_fields_info('dns_record_order_ctl_id', 'get_dns_record'),
    'dns-record-op-ctl': ctl_fields_info('dns_record_operation_ctl_id', 'get_dns_record_operation'),
    'dns-record-removal-ctl': ctl_fields_info('dns_record_removal_ctl_id', 'get_dns_record'),
}

CTL_NAME_RE = re.compile(r'(?P<ctl_type>[\w-]+)\("(?P<ctl_id>.*)"\)')


def get_full_ctl_id(name):
    if ':' in name:
        return name.split(':')
    return None


def process_ctl(cache, ctl_id, ctl_info, service_pb):
    """
    :type cache: AwacsCache
    :type ctl_id: (six.text_type, six._text_type) | six.text_type
    :type ctl_info: ctl_fields_info
    :param service_pb:
    :rtype: bool
    """
    full_ctl_id = get_full_ctl_id(ctl_id)
    if full_ctl_id is None:
        namespace_id = ctl_id
        if getattr(cache, ctl_info.cache_getter)(namespace_id):
            getattr(service_pb, ctl_info.pb_field_name).id = namespace_id
            return True
        else:
            return False
    else:
        namespace_id, obj_id = full_ctl_id
        if isinstance(ctl_info.cache_getter, six.string_types):
            cached_pb = getattr(cache, ctl_info.cache_getter)(namespace_id, obj_id)
        else:
            model = ctl_info.cache_getter
            cached_pb = model.cache.get(namespace_id, obj_id)
        if cached_pb:
            field_pb = getattr(service_pb, ctl_info.pb_field_name)
            field_pb.namespace_id = namespace_id
            field_pb.id = obj_id
            return True
        else:
            return False


@status_service_bp.method('ListSingletonContenders',
                          request_type=api_pb2.ListSingletonContendersRequest,
                          response_type=api_pb2.ListSingletonContendersResponse)
def list_singleton_contenders(req_pb, auth_subject):
    if not req_pb.id:
        raise rpc.exceptions.BadRequestError('"id" must be set')
    zk = inject.instance(zookeeper_client.IZookeeperClient)
    zk_path = os.path.join(ExclusiveService.ZK_PATH, req_pb.id)
    lock = zk.lock(zk_path)
    return api_pb2.ListSingletonContendersResponse(contenders=lock.contenders())


@status_service_bp.method('GetCtlStatuses',
                          request_type=api_pb2.GetCtlStatusesRequest,
                          response_type=api_pb2.GetCtlStatusesResponse,
                          max_in_flight=1)
def get_ctl_statuses(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetCtlStatusesRequest
    """
    cache = IAwacsCache.instance()
    services = cache.list_exclusive_services()
    log = logging.getLogger('get_ctl_statuses')

    ctl_pbs = []
    for name, contenders in sorted(services.items()):
        # name looks like namespace-ctl("namespace-id") or backend-ctl("namespace-id:backend-id"),
        # contenders is a list of instance names in kazoo's Lock order

        m = CTL_NAME_RE.match(name)
        if m is None:
            log.info('Unrecognized ctl name: %s', name)
            continue

        service_pb = api_pb2.GetCtlStatusesResponse.Ctl()
        ctl_type = m.group('ctl_type')
        ctl_id = m.group('ctl_id')
        ctl_info = supported_ctls.get(ctl_type)

        if ctl_info:
            try:
                process_ctl(cache, ctl_id, ctl_info, service_pb)
            except:
                log.exception('Failed to process ctl_id "%s" with ctl_info "%s"', ctl_id, ctl_info)
        else:
            log.warn('Unsupported ctl: type %s, id %s', ctl_type, ctl_id)

        if contenders:
            service_pb.leader = contenders[0]
            service_pb.contenders.extend(contenders[1:])

        if service_pb != api_pb2.GetCtlStatusesResponse.Ctl():
            ctl_pbs.append(service_pb)

    return api_pb2.GetCtlStatusesResponse(ctls=ctl_pbs)
